% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dna.R
\name{dnapath}
\alias{dnapath}
\title{Differential Network Analysis Using Gene Pathways}
\usage{
dnapath(
  x,
  pathway_list,
  group_labels = NULL,
  network_inference = run_pcor,
  n_perm = 100,
  lp = 2,
  seed = NULL,
  verbose = FALSE,
  mc.cores = 1,
  ...
)
}
\arguments{
\item{x}{The gene expression data to be analyzed. This can be either (1) a
list of two matrices or data frames that contain the gene expression profile
from each of two populations (groups) -- with rows corresponding to samples
and columns to genes -- or (2) a single matrix or data frame
that contains the expression profiles for both groups. For case (2), the
\code{group_labels} argument must be specified to identify which rows belong to which
group.}

\item{pathway_list}{A single vector or list of vectors containing gene names
to indicate pathway membership. The vectors are used to subset the columns
of the matrices in \code{x}. A pathway list can be obtained using
\code{\link{get_reactome_pathways}}. If NULL, then the entire expression
dataset is analyzed as a single network (this approach is not recommended
unless there are only a small number of genes).}

\item{group_labels}{If \code{x} is a single matrix or data frame, \code{group_labels} must
be specified to label each row. \code{group_labels} is a matrix each row
corresponding to a in \code{x}. This matrix may either (1) have a single column
containing the group label for each observation, or (2) individual columns
representing each group with values in \verb{[0, 1]} representing the probability
that the patient in that row is in each group. In the latter case, if the
rows do not sum to 1, then each entry will be divided by its row sum.}

\item{network_inference}{A function used to infer the pathway network. It
should take in an n by p matrix and return a p by p matrix of association
scores. (Built-in options include: \code{\link{run_aracne}},
\code{\link{run_bc3net}}, \code{\link{run_c3net}},
\code{\link{run_clr}}, \code{\link{run_corr}}, \code{\link{run_genie3}},
\code{\link{run_glasso}}, \code{\link{run_mrnet}},
\code{\link{run_pcor}}, and \code{\link{run_silencer}}.)
Defaults to \code{\link{run_pcor}} for partial correlations.}

\item{n_perm}{The number of random permutations to perform during
permutation testing. If \code{n_perm == 1}, the permutation tests are not performed.
If \code{n_perm} is larger than the number of possible permutations, \code{n_perm} will
be set to this value with a warning message.}

\item{lp}{The lp value used to compute differential connectivity
scores. (Note: If a vector is provided, then the results are returned as
a list of \code{dnapath_list} objects, one result for each value of \code{lp}. This
option is available so that network inference methods only need to be run
once for each pathway when multple values of \code{lp} are being considered. This
may be useful when conducting simulation studies).}

\item{seed}{(Optional) Used to set.seed prior to permutation test for
each pathway. This allows results for individual pathways to be easily
reproduced.}

\item{verbose}{Set to TRUE to turn on messages.}

\item{mc.cores}{Used in \code{\link[parallel]{mclapply}} to run the differential
network analysis in parallel across pathways. Must be set to 1 if on a Windows
machine.}

\item{...}{Additional arguments are passed into the network inference function.}
}
\value{
A 'dnapath_list' or 'dnapath' object containing results for each
pathway in \code{pathway_list}.
}
\description{
Integrates pathways into the differential network analysis of gene expression
data \insertCite{grimes19}{dnapath}.
}
\examples{
data(meso)
data(p53_pathways)
set.seed(0)
results <- dnapath(x = meso$gene_expression, pathway_list = p53_pathways,
                   group_labels = meso$groups, n_perm = 10)
results
summary(results) # Summary over all pathways in the pathway list.
# Remove results for pathways with p-values above 0.2.
top_results <- filter_pathways(results, 0.2)
# Sort the top results by the pathway DC score.
top_results <- sort(top_results, by = "dc_score")
top_results
summary(top_results[[1]])  # Summary of pathway 1.
plot(results[[1]]) # Plot of the differential network for pathway 1.

# Use ... to adjust arguments in the network inference function.
# For example, using run_corr() with method = "spearman":
results <- dnapath(x = meso$gene_expression, pathway_list = p53_pathways,
                   group_labels = meso$groups, n_perm = 10,
                   network_inference = run_corr,
                   method = "spearman")
results
}
\references{
\insertRef{grimes19}{dnapath}
}
\seealso{
\code{\link{filter_pathways}}, \code{\link{summary.dnapath_list}}
\code{\link{subset.dnapath_list}}, \code{\link{sort.dnapath_list}},
\code{\link{plot.dnapath}}, \code{\link{rename_genes}}
}
