\name{doRNG}
\alias{\%dorng\%}
\alias{doRNG}
\alias{infoDoRNG}
\alias{registerDoRNG}
\title{Reproducible Parallel Foreach Backend}
\usage{
  infoDoRNG(data, item)

  obj \%dorng\% ex

  registerDoRNG(seed, once = TRUE)
}
\arguments{
  \item{data}{a list of data used by the backend}

  \item{item}{the data item requested, as a character
  string (e.g. 'name', 'workers', 'version')}

  \item{obj}{a foreach object as returned by a call to
  \code{\link{foreach}}.}

  \item{ex}{the \code{R} expression to evaluate.}

  \item{seed}{a numerical seed to use (as a single or
  6-length numerical value)}

  \item{once}{a logical to indicate if the RNG sequence
  should be seeded at the beginning of each loop or only at
  the first loop.}
}
\value{
  \code{infoDoRNG} returns the requested info (usually as a
  character string or a numeric value).

  \code{\%dorng\%} returns the result of the foreach loop.
  See \code{\link{\%dopar\%}}.
}
\description{
  \code{infoDoRNG} returns information about the doRNG
  backend.

  \code{\%dorng\%} provides an alternative operator
  \code{\%dopar\%}, that ensures reproducible foreach
  loops.

  Note that (re-)registering a foreach backend other than
  doRNG, after a call to \code{registerDoRNG} disables
  doRNG -- which then needs to be registered.
}
\examples{
library(doParallel)
registerDoParallel(2)

# standard \%dopar\% loops are _not_ reproducible
set.seed(1234)
s1 <- foreach(i=1:4) \%dopar\% { runif(1) }
set.seed(1234)
s2 <- foreach(i=1:4) \%dopar\% { runif(1) }
identical(s1, s2)
\dontshow{ stopifnot( !identical(s1,s2) ) }

# single \%dorng\% loops are reproducible
s1 <- foreach(i=1:4, .options.RNG=1234) \%dorng\% { runif(1) }
s2 <- foreach(i=1:4, .options.RNG=1234) \%dorng\% { runif(1) }
identical(s1, s2)
\dontshow{ stopifnot( identical(s1,s2) ) }

# multiple \%dorng\% loops are reproducible
seed <- doRNGseed()
s1 <- foreach(i=1:4) \%dorng\% { runif(1) }
s2 <- foreach(i=1:4) \%dorng\% { runif(1) }

doRNGseed(seed)
s1.2 <- foreach(i=1:4) \%dorng\% { runif(1) }
s2.2 <- foreach(i=1:4) \%dorng\% { runif(1) }
identical(s1, s1.2) && identical(s2, s2.2)
\dontshow{ stopifnot( identical(s1,s1.2) && identical(s2,s2.2) ) }

# Works with SNOW-like and MPI clusters

\dontrun{
# SNOW-like cluster
cl <- makeCluster(2)
registerDoParallel(cl)

s1 <- foreach(i=1:4, .options.RNG=1234) \%dorng\% { runif(1) }
s2 <- foreach(i=1:4, .options.RNG=1234) \%dorng\% { runif(1) }
identical(s1, s2)
stopifnot( identical(s1,s2) )

stopCluster(cl)
registerDoSEQ()

# MPI cluster
library(doMPI)
cl <- startMPIcluster(2)
registerDoMPI(cl)

s1 <- foreach(i=1:4, .options.RNG=1234) \%dorng\% { runif(1) }
s2 <- foreach(i=1:4, .options.RNG=1234) \%dorng\% { runif(1) }
identical(s1, s2)
stopifnot( identical(s1,s2) )

closeCluster(cl)
registerDoSEQ()
}

\dontshow{
if( Sys.info()['user'] == 'renaud' ){
# SNOW-like cluster
cl <- makeCluster(2)
registerDoParallel(cl)

# standard \%dopar\% loops are _not_ reproducible
set.seed(1234)
s1 <- foreach(i=1:4) \%dopar\% { runif(1) }
set.seed(1234)
s2 <- foreach(i=1:4) \%dopar\% { runif(1) }
identical(s1, s2)
stopifnot( !identical(s1,s2) )

# \%dorng\% loops ensure reproducibility
s1 <- foreach(i=1:4, .options.RNG=1234) \%dorng\% { runif(1) }
s2 <- foreach(i=1:4, .options.RNG=1234) \%dorng\% { runif(1) }
identical(s1, s2)
stopifnot( identical(s1,s2) )

stopCluster(cl)
registerDoSEQ()

# Works with doMPI
library(doMPI)
cl <- startMPIcluster(2)
registerDoMPI(cl)

# standard \%dopar\% loops are _not_ reproducible
set.seed(1234)
s1 <- foreach(i=1:4) \%dopar\% { runif(1) }
set.seed(1234)
s2 <- foreach(i=1:4) \%dopar\% { runif(1) }
identical(s1, s2)
stopifnot( !identical(s1,s2) )

# \%dorng\% loops ensure reproducibility
s1 <- foreach(i=1:4, .options.RNG=1234) \%dorng\% { runif(1) }
s2 <- foreach(i=1:4, .options.RNG=1234) \%dorng\% { runif(1) }
identical(s1, s2)
stopifnot( identical(s1,s2) )

closeCluster(cl)
registerDoSEQ()
}
}
library(doParallel)
cl <- makeCluster(2)
registerDoParallel(cl)

# One can make existing \%dopar\% loops reproducible using \%dorng\% loops or registerDoRNG
r1 <- foreach(i=1:4, .options.RNG=1234) \%dorng\% { runif(1) }
registerDoRNG(1234)
r2 <- foreach(i=1:4) \%dopar\% { runif(1) }
identical(r1, r2)
\dontshow{ stopifnot( identical(r1, r2) ) }
stopCluster(cl)

# Registering another foreach backend disables doRNG
cl <- makeCluster(3)
registerDoParallel(cl)
s1 <- foreach(i=1:4) \%dopar\% { runif(1) }
s2 <- foreach(i=1:4) \%dopar\% { runif(1) }
identical(s1, s2)
\dontshow{ stopifnot( !identical(s1, s2) ) }

# doRNG is re-nabled by re-registering it
registerDoRNG(1234)
r3 <- foreach(i=1:4) \%dopar\% { runif(1) }
identical(r2, r3)
# NB: the results are identical independently of the task scheduling
# (r2 used 2 nodes, while r3 used 3 nodes)
\dontshow{ stopifnot( identical(r2, r3) ) }

# argument `once=FALSE` reseed doRNG's seed at the begining each loop
registerDoRNG(1234, once=FALSE)
r1 <- foreach(i=1:4) \%dopar\% { runif(1) }
r2 <- foreach(i=1:4) \%dopar\% { runif(1) }
identical(r1, r2)
\dontshow{ stopifnot( identical(r1, r2) && identical(r1, r3)) }

# Once doRNG is registered the seed can also be passed as an option to \%dopar\%
r1.2 <- foreach(i=1:4, .options.RNG=456) \%dopar\% { runif(1) }
r2.2 <- foreach(i=1:4, .options.RNG=456) \%dopar\% { runif(1) }
identical(r1.2, r2.2) && !identical(r1.2, r1)
\dontshow{ stopifnot( identical(r1.2, r2.2) && !identical(r1.2, r1) ) }

stopCluster(cl)
}
\author{
  Renaud Gaujoux
}
\seealso{
  \code{\link{foreach}},
  \code{\link[doParallel]{doParallel}} ,
  \code{\link[doParallel]{registerDoParallel}},
  \code{\link[doMPI]{doMPI}}
}

