% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/DBA.R
\name{DBA}
\alias{DBA}
\title{DTW Barycenter Averaging}
\usage{
DBA(X, center = NULL, max.iter = 20L, norm = "L1", window.size = NULL,
  delta = 0.001, error.check = TRUE, trace = FALSE, ...)
}
\arguments{
\item{X}{A data matrix where each row is a time series, or a list where each element is a time series.}

\item{center}{Optionally, a time series to use as reference. It must be a numeric vector. Defaults to a
random series of \code{X} if \code{NULL}.}

\item{max.iter}{Maximum number of iterations allowed.}

\item{norm}{Norm for the local cost matrix of DTW. Either "L1" for Manhattan distance or "L2" for Euclidean
distance.}

\item{window.size}{Window constraint for the DTW calculations. \code{NULL} means no constraint.}

\item{delta}{At iteration \code{i}, if \code{all(abs(center_{i}} \code{ - center_{i-1})} \code{ < delta)},
convergence is assumed.}

\item{error.check}{Should inconsistencies in the data be checked?}

\item{trace}{If \code{TRUE}, the current iteration is printed to screen.}

\item{...}{Further arguments for \code{\link[dtw]{dtw}}, e.g. \code{step.pattern}. Do not provide
\code{window.type} here, just set \code{window.size} to the desired value.}
}
\value{
The average time series.
}
\description{
A global averaging method for time series under DTW (Petitjean, Ketterlin and Gancarski, 2011).
}
\details{
This function tries to find the optimum average series between a group of time series in DTW space. Refer to
the cited article for specific details on the algorithm.

If a given series reference is provided in \code{center}, the algorithm should always converge to the same
result provided the elements of \code{X} keep the same values, although their order may change.
}
\section{Parallel Computing}{


Please note that running tasks in parallel does \strong{not} guarantee faster computations.
The overhead introduced is sometimes too large, and it's better to run tasks sequentially.

The user can register a parallel backend, e.g. with the \code{doParallel} package, in order to attempt to
speed up the calculations (see the examples).
}
\examples{

# Sample data
data(uciCT)

# Obtain an average for the first 5 time series
dtw.avg <- DBA(CharTraj[1:5], CharTraj[[1]], trace = TRUE)
plot(dtw.avg, type="l")

# Change the provided order
dtw.avg2 <- DBA(CharTraj[5:1], CharTraj[[1]], trace = TRUE)

# Same result?
all(dtw.avg == dtw.avg2)

\dontrun{
#### Running DBA with parallel support
# For such a small dataset, this is probably slower in parallel
require(doParallel)

# Create parallel workers
cl <- makeCluster(detectCores())
invisible(clusterEvalQ(cl, library(dtwclust)))
registerDoParallel(cl)

# DTW Average
cen <- DBA(CharTraj[1:5], CharTraj[[1]], trace = TRUE)

# Stop parallel workers
stopCluster(cl)

# Return to sequential computations
registerDoSEQ()
}

}
\references{
Petitjean F, Ketterlin A and Gancarski P (2011). ``A global averaging method for dynamic time
warping, with applications to clustering.'' \emph{Pattern Recognition}, \strong{44}(3), pp. 678 -
693. ISSN 0031-3203, \url{http://dx.doi.org/10.1016/j.patcog.2010.09.013},
\url{http://www.sciencedirect.com/science/article/pii/S003132031000453X}.
}

