#load.source("util.r")
# source("initial.r")
# source("root_zone.r")
# source("unsat_zone.r")
# source("distribute.r")
# source("update_subsurface.r")
# source("balance.r")
# source("results.r")
# source("debug.r")
# source("routing.r")
# source("route_ovf.r")
# source("defs.r")
require(xts)
require(deSolve)

# ********************************************************************
# main routine for the Dynamic TOPMODEL for the areal groupings identified by a catchment analysis
# see Beven and Freer (2001), Page and Beven (2007) for a description of the model
# *********************************************
# Notes on units:
# lengths are in m and times in hrs, coverted if necessary
# rainfall and pe in m/hr (will be converted from mm/hr as usual convention)
# storage in rain equivalent units e.g. m
# base flows expressed as specific fluxes per plan area: m^3/hr per m^2
# input flows expressed as total flux (m^3/hr)
# ****************************************************************************************************************
# groups: data frame or matrix constructed by running the catchment classification algorithmn
# should minimally comprise the group id and plan area "area". Group ID is identified
# with the river channel network according to examination of upslope contributing areas
#
# weights: flux transistion matrix - ngroup x ngroup weighting matrix, mth column of nth row is the proportion of
# flow out of nth to the mth groups

# ****************************************************************************************************************
# Routing
# ---------
# specifies how overland flow and in channel should be routed to the outlet
# ****************************************************************************************************************
# Parameters
# ----------------------------------------------------------------------------------------------------------------
# Parameter                                         Units         Typical values
# 																	min			max
# dt: outer time steps								              hr
# nt: number of inner time steps                    -
# ****************************************************************************************************************
# Catchment-wide parameters
# ----------------------------------------------------------------------------------------------------------------
# Parameter                                         Units         Typical values (see for example Beven and Freer, 2001, Page et al 2006)
#                                                                 Lower             Upper
# ----------------------------------------------------------------------------------------------------------------
# vchan   :   channel wave routing velocity           m/hr          1000              5000
# vof     : overland flow wave velocity                m/hr          100              500
# ****************************************************************************************************************
# dqds    ; gradient function of Q - S realtionship in subsurface. Defaults to exponential, but other forms allow different transmissivity profiles to be examined
# groups: HSU information comprising id, plan area and physical parameters.
# Takes values from def.hsu.par for any that are unspecified
# ----------------------------------------------------------------------------------------------------------------
# Parameter                                         Units         Typical values (see Beven and Freer, 2001 (1), Beven 1997, Page et al 2006 (2))                                                                  Lower             Upper
# ----------------------------------------------------------------------------------------------------------------
# m     :   form of exponential decline in          m             0.005             0.025
#           conductivity
# SRmax:    max root zone storage                   m             0.005 (2)         0.2 (2)
# SRInit:   initial root zone storage               m             0                 0.3
# LnTo  :   lateral saturated transmissivity        m^2/hr-1      -7 (2)            8
# sd_max  :   max effective deficit of saturated zone m             0.1               0.8
# td    :   unsaturated zone time delay             hr/m          0.1 (1, 2)        40 (2)
# ****************************************************************************************************************
#' Run Dynamic TOPMODEL using the catchment areal groupings (response units) for a discretisation.
#' @details The grouping (HRU) table may be generated by the discretise method and includes each indexed channel as separate group. See Metcalfe et al. (2015) for descriptions of the parameters maintained in this table.
#' @details Evapotranspiration input can be generated using the approx.pe.ts method
#' @details If disp.par$graphics.show = T then the output will be displayed graphically whilst the simulation is in progress. Otherwise simulated specific discharges
#' @export run.dtm
#' @import deSolve
#' @import xts
#' @author Peter Metcalfe
#' @param groups Data frame of ngroup areal group definitions along with their hydrological parameters.
#' @param weights The flux distribution (weighting) ngroup*ngroup matrix. Usually generated by the discretise method.
#' @param rain A time series of rainfall data in m/hr. One column per gauge if multiple gauges used.  Use aggregate_obs to apply a different time interval to this and the other observation data.
#' @param routing Channel routing table comprises a two-column data.frame or matrix. Its first column should be average flow distance to the outlet in m, the second the proportions of the catchment channel network within each distance category. Can be generated by make.routing.table
#' @param qobs Optional time series of observation data
#' @param qt0 Initial specific discharge (m/hr)
#' @param pe Time series of potential evapotranspiration, at the same time step as rainfall data
#' @param dt Time step (hours). Defaults to the interval used by the rainfall data
#' @param vchan Default channel routing velocity (m/hr)
#' @param vof Default overland flow routing velocity (m/hr).
#' @param ichan Integer index of the "channel" group. Defaults to 1
#' @param i.out For multi-channel systems, the index of the outlet reach
#' @param sim.start Optional start time for simulation in any format that can be coerced into a POSIXct instance. Defaults to start of rainfall data
#' @param sim.end Optional end time of simulation in any format that can be coerced into a POSIXct instance. Defaults to end of rainfall datA
#' @param disp.par List of graphical routing parameters. A set of defaults are retrieved by calling disp.par()
#' @param ntt Number of inner time steps used in subsurface routing algorithm.
#' @param dqds Function to supplies a custom flux-storage relationship as the kinematic wave celerity. If not supplied then exponential relationship used
#' @return A list containing run output and input data. These include
#' @return qsim: time series of specific discharges (m/hr) at the specified time interval. can be converted to absolutre discharges by multiying by catch.area
#' @return catch.area: the catchment area in m^2, calculated from the areas in the groups table
#' @return data.in: the parameters supplied as input to the call to run.dtm
#' @return sim.start: start of simulation
#' @return sim.end: end time of simulation
#' @return fluxes: a list comprising, for each response unit the specific base flows qbf, specific upslope inputs qin, drainage fluxes quz, and any overland flow qof, all in m/hr
#' @return storages: a list comprising, for each response unit, root zone and unsaturated storage and total storage deficit (all m)
#' @references Beven, K., & Freer, J. (2001). A dynamic topmodel. Hydrological processes, 15(10), 1993-2011.
#' @references Metcalfe, P., Beven, K., & Freer, J. (2015). Dynamic TOPMODEL: A new implementation in R and its sensitivity to time and space steps. Environmental Modelling & Software, 72, 155-172.

#' @examples
#'\dontrun{
#' require(dynatopmodel)
#' data(brompton)
#' # September 2012 storm event
#' # The response is sensitive to the size of the channels, but many are small.
#' # Set an overall width of 2m.
#' chans <- build_chans(dem=brompton$dem, drn=brompton$drn, chan.width=2)
#'
#' # discretisation by reverse distance from nearest channel. The raster brompton$flowdists
#' # gives the D8 flow pathway distance for every area in the catchment
#' layers <- addLayer(brompton$dem, 2000-brompton$flowdists)
#' disc  <- discretise(layers, cuts=c(flowdists=5), chans=chans, area.thresh=3/100)
#'
#' Network routing table
#' routing <- build_routing_table(brompton$dem, chans)
#'
#' # Here we apply the same parameter values to all groups. Suggest applying smaller m and td values to
#' # the closest areas to simulate a fast response due to the artificial drainage.
#' # It would also be possible to supply a custom transmissivity profile that has
#' # a discontinuity at the depth of the drains
#' groups <- disc$groups
#' groups$m <- 0.011
#' groups$td <-  42
#' # a very high transmissivity prevents saturation flow as there appears be little
#' groups$ln_t0 <- 18
#' groups$srz_max <- 0.1
#' # initial root zone storage
#' groups$srz0 <- 0.87
#' # quite slow channel flow, which might be expected with the shallow and reedy
#' # reaches in this catchment
#' groups$vchan <- 750
#'
#' # Observations at a 15 minute time step
#' dt <- 0.25
#' obs <- list(rain=brompton$rain,
#' pe=brompton$pe,
#' qobs=brompton$qobs)
#' obs <- aggregate_obs(obs, dt=dt)
#'
#' # parameters for graphics output
#' par <- disp.par(int.time=24)
#'
#' # Note max.q in mm/hr
#' par$max.q <- 1000*max(obs$qobs, na.rm=TRUE)
#' sim.start <- "2012-09-23"
#' sim.end <- "2012-10-01"
#'
#' # Ensure output goes to a new window
#' options("device"="X11")
#' # take initial discharge from the observations
#' qt0 <- as.numeric(obs$qobs[sim.start][1])
#'
#' # Run the model across the September 2012 storm event using 2 inner time steps
#' and a 15 minute interval
#'  storm.run <- run.dtm(groups=groups,
#'    weights=disc$weights,
#'    rain=obs$rain,
#'    pe=obs$pe,
#'    qobs=obs$qobs,
#'    qt0=qt0,
#'    sim.start=sim.start,
#'    sim.end=sim.end,
#'    routing=routing,
#'    disp.par=par,
#'    ntt=2)
#'  # show run statistics
#'  cat("NSE=", NSE(storm.run$qsim, storm.run$qobs))
#'  cat("Time at peak =", format(time_at_peak(storm.run$qsim)))
#'}

run.dtm <- function(groups,
                    weights,
                    rain,
                    routing,
                    qobs=NULL,
                    qt0=1e-4,
                    pe=NULL,
                    dt=NULL,
                    ntt=1,
					          ichan=1,
                    i.out=ichan[1],
					          vchan = 1000,
					          vof = 100,
                    dqds=NULL,
                    sim.start=NA,
                    sim.end=NA,
                    disp.par = disp.par())
{
  run.par = def.run.par()
  start.time <- Sys.time()

  # setup input variable for run using supplied data, and copy the updated values back to
  # to the current environment
  data.in <- init.input(groups, dt, ntt,
  			weights, rain, pe, routing,
  			ichan=ichan, i.out=i.out, qobs=qobs,
  			qt0=qt0,
  			dqds=dqds,
        disp.par,
  			run.par,
        sim.start,
  			sim.end,
        calling.env=environment())

  catch.area <- sum(groups$area)
  storage.in <- current.storage(groups, stores, ichan)
  text.out <- stdout()

  # complementary weighting matrix, scaled by groups' areas
  A <- diag(1/groups$area) %*% t(weights) %*% diag(groups$area)

  while(time <= sim.end)
  {
    # Allocate rainfall to groups depending on gauge specified, then
    # add in any overland excess distributed downslope in a previous timestep (flux)
    flows$rain <- as.vector(allocate.rain(rain, groups, it) +
                                stores$ex/dt)
    stores[,"ex"] <- 0

    pe.dist <- allocate.PE(pe, groups, it)
   # stores <- SaturatedEvap(groups,flows,stores,peDist)
    # subsurface flux routing and deficit update
   updated <- update.subsurface(groups,
   							flows=flows, stores=stores,
   							w=weights,
                            pe = pe.dist,
                            tm=time,
                            ntt=ntt,
                            dt=dt,
                            ichan=ichan,
                            dqds=dqds)

   flows <- updated$flows
   stores <- updated$stores

   # redistribute overland flow
   flows$qof  <- dist.eigen(groups, A=A, ex=stores$ex, vof=groups$vof, dt=dt, ichan=ichan, tm=time)/dt

   # route to outlet or update current
   Qr <- route.channel.flows(groups, flows, delays=routing,
                             weights, Qr, it, dt, ichan)

   fluxes[it,,]<- as.matrix(flows[, c("qbf", "qin", "uz", "rain", "ae", "ex", "qof")])

   qr <- Qr[it,]/catch.area

   # save output
   storages[it,,]<- as.matrix(stores[, c("srz", "suz", "sd", "ex")])

  # overall actual evap
  evap[it,"ae"] <- weighted.average(flows$ae, groups$area)

	disp.results(it,
    tm=time,
    qr=Qr[,ichan]*1000/catch.area, # calculated specific discharge at outlet, mm/hr
    rain=rain*1000,            # rain and ae converted back to mm/hr
    evap=evap*1000,  #
    groups=groups,
    flows=flows,
    stores=stores,
    qobs=qobs*1000,  # this should have been converted to specific discharge. It also is in mm/hr
    ichan=ichan,   # channel indicators
    text.out=text.out,
    log.msg="",
    start = sim.start,
    end = sim.end,
    disp.par=disp.par,
    run.par=run.par)

    # overland flow contribution is the total input to the channels (doesn't matter where it ends up after that)
    Qof[it,]<- sum(flows[ichan,]$qof*groups[ichan,]$area)

    # update index and time step
    time <- time + dt*3600
    it <- it+1
	# remove excess storage etc
    flows[, c("pex", "ex", "exus")] <- 0 #ResetFluxes(flows, ichan)
  }
  # sum up
  it <- it-1
  qsim <- subset_zoo(Qr, sim.start, sim.end)/catch.area  # specific flux
  qobs <- subset_zoo(qobs, sim.start, sim.end)
  wb <- water.balance(groups, stores, dt=dt, storage.in, ichan, qsim,
                      rain=fluxes[,,4], ae=fluxes[,,5])
	dur <- difftime(Sys.time(), start.time, units="secs")
  # total ovf is amount transferred to outlet (includes rain directly to channel?)
  # in this formulation all excess flow is routed immediately and then removed
  ovf <- dt*sum(Qof)/catch.area

  # print water balance effciences, run time etc
  RunSummary(groups, stores, storage.in, ichan, qsim, qobs, start.time,
                         rain, ae=evap[,"ae"], disp.par$text.out)

  tms <- index(qsim)
  # convert fluxes to a named list of time series
  fluxes <- apply(fluxes, MARGIN=3, function(x){list(x)})
  fluxes <- lapply(fluxes, function(x)xts(x[[1]], order.by=tms))

  names(fluxes) <- c("qbf", "qin", "uz", "rain", "ae", "ex", "qof")

  # convert storages to a list
  storages <- apply(storages, MARGIN=3, function(x){list(x)})
  storages <- lapply(storages, function(x)xts(x[[1]], order.by=tms))
  names(storages) <- c("srz", "suz", "sd", "ex")

  return(list("qsim"=qsim,
              "sim.start"=sim.start,
              "sim.end"=sim.end,
              "fluxes"=fluxes,
              "storages"=storages,
              "qobs"=qobs,
              "ae"=evap[,"ae"],
              "ovf"=ovf,
              "rain"=rain,
              catch.area=catch.area,
              data.in=data.in))
}




