% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/epw.R
\docType{class}
\name{Epw}
\alias{Epw}
\title{Read, and modify an EnergyPlus Weather File (EPW)}
\description{
Reading an EPW file starts with function \code{\link[=read_epw]{read_epw()}}, which parses an EPW
file and returns an \code{Epw} object. The parsing process is basically the same
as [EnergyPlus/WeatherManager.cc] in EnergyPlus, with some simplifications.
}
\details{
An EPW file can be divided into two parts, headers and weather data. The
first eight lines of a standard EPW file are normally headers which contains
data of location, design conditions, typical/extreme periods, ground
temperatures, holidays/daylight savings, data periods and other comments.
\code{Epw} class provides methods to directly extract those data. For details on
the data structure of EPW file, please see "Chapter 2 - Weather Converter
Program" in EnergyPlus "Auxiliary Programs" documentation. An online version
can be found \href{https://bigladdersoftware.com/epx/docs/}{here}.

There are about 35 variables in the core weather data. However, not all of
them are used by EnergyPlus. Actually, despite of date and time columns, only
13 columns are used:
\enumerate{
\item dry bulb temperature
\item dew point temperature
\item relative humidity
\item atmospheric pressure
\item horizontal infrared radiation intensity from sky
\item direct normal radiation
\item diffuse horizontal radiation
\item wind direction
\item wind speed
\item present weather observation
\item present weather codes
\item snow depth
\item liquid precipitation depth
}

\strong{Note} the \code{hour} column in the core weather data corresponds to the period
from \strong{(Hour-1)th} to \strong{(Hour)th}. For instance, if the number of interval
per hour is 1, hour of 1 on a certain day corresponds to the period between
00:00:01 to 01:00:00, Hour of 2 corresponds to the period between
01:00:01 to 02:00:00, and etc. Currently, in EnergyPlus the minute column is
\strong{not used} to determine currently sub-hour time. For instance, if the
number of interval per hour is 2, there is no difference between two rows
with following time columns (a) Hour 1, Minute 0; Hour 1, Minute 30 and (b)
Hour 1, Minute 30; Hour 1, Minute 60. Only the number of rows count.
When EnergyPlus reads the EPW file, both (a) and (b) represent the same time
period: 00:00:00 - 00:30:00 and 00:30:00 - 01:00:00.
Missing data on the weather file used can be summarized in the eplusout.err
file, if \code{DisplayWeatherMissingDataWarnings} is turned on in
\code{Output:Diagnostics} object. In EnergyPlus, missing data is shown only for
fields that EnergyPlus will use. EnergyPlus will fill some missing data
automatically during simulation. Likewise out of range values are counted for
each occurrence and summarized. However, note that the out of range values
will \strong{not be changed} by EnergyPlus and could affect your simulation.

\code{Epw} class provides methods to easily extract and inspect those abnormal
(missing and out of range) weather data and also to know what kind of actions
that EnergyPlus will perform on those data.

EnergyPlus energy model calibration often uses actual measured weather data.
In order to streamline the error-prone process of creating custom EPW file,
\code{Epw} provides methods to direction add, replace the core weather data.
}
\examples{

## ------------------------------------------------
## Method `Epw$new`
## ------------------------------------------------

\dontrun{
# read an EPW file from EnergyPlus website
path_base <- "https://energyplus.net/weather-download"
path_region <- "north_and_central_america_wmo_region_4/USA/CA"
path_file <- "USA_CA_San.Francisco.Intl.AP.724940_TMY3/USA_CA_San.Francisco.Intl.AP.724940_TMY3.epw"
path_epw <- file.path(path_base, path_region, path_file)
epw <- read_epw(path_epw)

# read an EPW file distributed with EnergyPlus
if (is_avail_eplus(8.8)) {
    path_epw <- file.path(
        eplus_config(8.8)$dir,
        "WeatherData",
        "USA_CA_San.Francisco.Intl.AP.724940_TMY3.epw"
    )
    epw <- read_epw(path_epw)
}
}


## ------------------------------------------------
## Method `Epw$path`
## ------------------------------------------------

\dontrun{
# get path
epw$path()
}


## ------------------------------------------------
## Method `Epw$definition`
## ------------------------------------------------

\dontrun{
# get path
epw$definition("LOCATION")
}


## ------------------------------------------------
## Method `Epw$location`
## ------------------------------------------------

\dontrun{
epw$location()

# modify location data
epw$location(city = "MyCity")
}


## ------------------------------------------------
## Method `Epw$design_condition`
## ------------------------------------------------

\dontrun{
epw$design_condition()
}


## ------------------------------------------------
## Method `Epw$typical_extreme_period`
## ------------------------------------------------

\dontrun{
epw$typical_extreme_period()
}


## ------------------------------------------------
## Method `Epw$ground_temperature`
## ------------------------------------------------

\dontrun{
epw$ground_temperature()
}


## ------------------------------------------------
## Method `Epw$holiday`
## ------------------------------------------------

\dontrun{
epw$holiday()

# add daylight saving time
epw$holiday(dst = c(3.10, 11.3))
}


## ------------------------------------------------
## Method `Epw$comment1`
## ------------------------------------------------

\dontrun{
epw$comment1()

epw$comment1("Comment1")
}


## ------------------------------------------------
## Method `Epw$comment2`
## ------------------------------------------------

\dontrun{
epw$comment2()

epw$comment2("Comment2")
}


## ------------------------------------------------
## Method `Epw$num_period`
## ------------------------------------------------

\dontrun{
epw$num_period()
}


## ------------------------------------------------
## Method `Epw$interval`
## ------------------------------------------------

\dontrun{
epw$interval()
}


## ------------------------------------------------
## Method `Epw$period`
## ------------------------------------------------

\dontrun{
# modify data period name
epw$period(1, name = "test")

# change start day of week
epw$period(1, start_day_of_week = 3)
}


## ------------------------------------------------
## Method `Epw$missing_code`
## ------------------------------------------------

\dontrun{
epw$missing_code()
}


## ------------------------------------------------
## Method `Epw$initial_missing_value`
## ------------------------------------------------

\dontrun{
epw$initial_missing_value()
}


## ------------------------------------------------
## Method `Epw$range_exist`
## ------------------------------------------------

\dontrun{
epw$range_exist()
}


## ------------------------------------------------
## Method `Epw$range_valid`
## ------------------------------------------------

\dontrun{
epw$range_valid()
}


## ------------------------------------------------
## Method `Epw$fill_action`
## ------------------------------------------------

\dontrun{
epw$fill_action("missing")

epw$fill_action("out_of_range")
}


## ------------------------------------------------
## Method `Epw$data`
## ------------------------------------------------

\dontrun{
# get weather data
str(epw$data())

# get weather data but change the year to 2018
# the year column is not changed by default, only the returned datetime column
head(epw$data(start_year = 2018)$datetime)
str(epw$data(start_year = 2018)$year)
# you can update the year column too
head(epw$data(start_year = 2018, update = TRUE)$year)

# change the time zone of datetime column in the returned weather data
attributes(epw$data()$datetime)
attributes(epw$data(tz = "Etc/GMT+8")$datetime)
}


## ------------------------------------------------
## Method `Epw$abnormal_data`
## ------------------------------------------------

\dontrun{
epw$abnormal_data()

# only check if there are any abnormal values in air temperature and
# liquid precipitation rate
epw$abnormal_data(cols = c("dry_bulb_temperature", "liquid_precip_rate"))

# save as above, but only return date time columns plus those 2 columns
epw$abnormal_data(cols = c("dry_bulb_temperature", "liquid_precip_rate"),
    keep_all = FALSE
)

# same as above, but only check for missing values
epw$abnormal_data(cols = c("dry_bulb_temperature", "liquid_precip_rate"),
    type = "missing"
)

# same as above, but only check for out-of-range values
epw$abnormal_data(cols = c("dry_bulb_temperature", "liquid_precip_rate"),
    type = "out_of_range"
)
}


## ------------------------------------------------
## Method `Epw$redundant_data`
## ------------------------------------------------

\dontrun{
epw$redundant_data()
}


## ------------------------------------------------
## Method `Epw$make_na`
## ------------------------------------------------

\dontrun{
# turn all missing values into NAs
summary(epw$data()$liquid_precip_rate)
epw$make_na(missing = TRUE)
summary(epw$data()$liquid_precip_rate)
}


## ------------------------------------------------
## Method `Epw$fill_abnormal`
## ------------------------------------------------

\dontrun{
# turn all missing values into NAs
summary(epw$data()$liquid_precip_rate)
epw$fill_abnormal(missing = TRUE)
summary(epw$data()$liquid_precip_rate)
}


## ------------------------------------------------
## Method `Epw$add_unit`
## ------------------------------------------------

\dontrun{
# get weather data with units
epw$add_unit()
head(epw$data())

# with units specified, you can easily perform unit conversion using units
# package
t_dry_bulb <- epw$data()$dry_bulb_temperature
units(t_dry_bulb) <- with(units::ud_units, "kelvin")

head(t_dry_bulb)
}


## ------------------------------------------------
## Method `Epw$drop_unit`
## ------------------------------------------------

\dontrun{
epw$drop_unit()
epw$data()
}


## ------------------------------------------------
## Method `Epw$purge`
## ------------------------------------------------

\dontrun{
epw$purge()
}


## ------------------------------------------------
## Method `Epw$add`
## ------------------------------------------------

\dontrun{
# will fail since date time in input data has already been covered by
# existing data period
try(epw$add(epw$data()), silent = TRUE)
}


## ------------------------------------------------
## Method `Epw$set`
## ------------------------------------------------

\dontrun{
# change the weather data
epw$set(epw$data())
}


## ------------------------------------------------
## Method `Epw$is_unsaved`
## ------------------------------------------------

\dontrun{
epw$is_unsaved()
}


## ------------------------------------------------
## Method `Epw$save`
## ------------------------------------------------

\dontrun{
# save the weather file
epw$save(file.path(tempdir(), "weather.epw"), overwrite = TRUE)
}


## ------------------------------------------------
## Method `Epw$print`
## ------------------------------------------------

\dontrun{
epw$print()
}

}
\author{
Hongyuan Jia
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-Epw-new}{\code{Epw$new()}}
\item \href{#method-Epw-path}{\code{Epw$path()}}
\item \href{#method-Epw-definition}{\code{Epw$definition()}}
\item \href{#method-Epw-location}{\code{Epw$location()}}
\item \href{#method-Epw-design_condition}{\code{Epw$design_condition()}}
\item \href{#method-Epw-typical_extreme_period}{\code{Epw$typical_extreme_period()}}
\item \href{#method-Epw-ground_temperature}{\code{Epw$ground_temperature()}}
\item \href{#method-Epw-holiday}{\code{Epw$holiday()}}
\item \href{#method-Epw-comment1}{\code{Epw$comment1()}}
\item \href{#method-Epw-comment2}{\code{Epw$comment2()}}
\item \href{#method-Epw-num_period}{\code{Epw$num_period()}}
\item \href{#method-Epw-interval}{\code{Epw$interval()}}
\item \href{#method-Epw-period}{\code{Epw$period()}}
\item \href{#method-Epw-missing_code}{\code{Epw$missing_code()}}
\item \href{#method-Epw-initial_missing_value}{\code{Epw$initial_missing_value()}}
\item \href{#method-Epw-range_exist}{\code{Epw$range_exist()}}
\item \href{#method-Epw-range_valid}{\code{Epw$range_valid()}}
\item \href{#method-Epw-fill_action}{\code{Epw$fill_action()}}
\item \href{#method-Epw-data}{\code{Epw$data()}}
\item \href{#method-Epw-abnormal_data}{\code{Epw$abnormal_data()}}
\item \href{#method-Epw-redundant_data}{\code{Epw$redundant_data()}}
\item \href{#method-Epw-make_na}{\code{Epw$make_na()}}
\item \href{#method-Epw-fill_abnormal}{\code{Epw$fill_abnormal()}}
\item \href{#method-Epw-add_unit}{\code{Epw$add_unit()}}
\item \href{#method-Epw-drop_unit}{\code{Epw$drop_unit()}}
\item \href{#method-Epw-purge}{\code{Epw$purge()}}
\item \href{#method-Epw-add}{\code{Epw$add()}}
\item \href{#method-Epw-set}{\code{Epw$set()}}
\item \href{#method-Epw-del}{\code{Epw$del()}}
\item \href{#method-Epw-is_unsaved}{\code{Epw$is_unsaved()}}
\item \href{#method-Epw-save}{\code{Epw$save()}}
\item \href{#method-Epw-print}{\code{Epw$print()}}
\item \href{#method-Epw-clone}{\code{Epw$clone()}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Epw-new"></a>}}
\if{latex}{\out{\hypertarget{method-Epw-new}{}}}
\subsection{Method \code{new()}}{
Create an \code{Epw} object
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Epw$new(path, encoding = "unknown")}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{path}}{Either a path, a connection, or literal data (either a
single string or a raw vector) to an EnergyPlus Weather File
(EPW).  If a file path, that file usually has a extension
\code{.epw}.}

\item{\code{encoding}}{The file encoding of input IDD. Should be one of
\code{"unknown"}, \verb{"Latin-1" and }"UTF-8"\verb{. The default is }"unknown"` which means that the file is encoded in the native
encoding.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
It takes an EnergyPlus Weather File (EPW) as input and returns an
\code{Epw} object.
}

\subsection{Returns}{
An \code{Epw} object.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# read an EPW file from EnergyPlus website
path_base <- "https://energyplus.net/weather-download"
path_region <- "north_and_central_america_wmo_region_4/USA/CA"
path_file <- "USA_CA_San.Francisco.Intl.AP.724940_TMY3/USA_CA_San.Francisco.Intl.AP.724940_TMY3.epw"
path_epw <- file.path(path_base, path_region, path_file)
epw <- read_epw(path_epw)

# read an EPW file distributed with EnergyPlus
if (is_avail_eplus(8.8)) {
    path_epw <- file.path(
        eplus_config(8.8)$dir,
        "WeatherData",
        "USA_CA_San.Francisco.Intl.AP.724940_TMY3.epw"
    )
    epw <- read_epw(path_epw)
}
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Epw-path"></a>}}
\if{latex}{\out{\hypertarget{method-Epw-path}{}}}
\subsection{Method \code{path()}}{
Get the file path of current \code{Epw}
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Epw$path()}\if{html}{\out{</div>}}
}

\subsection{Details}{
\verb{$path()} returns the full path of current \code{Epw} or \code{NULL} if the
\code{Epw} object is created using a character vector and not saved
locally.
}

\subsection{Returns}{
\code{NULL} or a single string.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# get path
epw$path()
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Epw-definition"></a>}}
\if{latex}{\out{\hypertarget{method-Epw-definition}{}}}
\subsection{Method \code{definition()}}{
Get the \link{IddObject} object for specified EPW class.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Epw$definition(class)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{class}}{A single string.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$definition()} returns an \link{IddObject} of given EPW class. \link{IddObject}
contains all data used for parsing that EPW class.

Currently, all supported EPW classes are:
\itemize{
\item \code{LOCATION}
\item \verb{DESIGN CONDITIONS}
\item \verb{TYPICAL/EXTREME PERIODS}
\item \verb{GROUND TEMPERATURES}
\item \verb{HOLIDAYS/DAYLIGHT SAVINGS}
\item \verb{COMMENTS 1}
\item \verb{COMMENTS 2}
\item \verb{DATA PERIODS}
\item \verb{WEATHER DATA}
}
}

\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# get path
epw$definition("LOCATION")
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Epw-location"></a>}}
\if{latex}{\out{\hypertarget{method-Epw-location}{}}}
\subsection{Method \code{location()}}{
Get and modify LOCATION header
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Epw$location(
  city,
  state_province,
  country,
  data_source,
  wmo_number,
  latitude,
  longitude,
  time_zone,
  elevation
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{city}}{A string of city name recorded in the \code{LOCATION} header.}

\item{\code{state_province}}{A string of state or province name recorded in
the \code{LOCATION} header.}

\item{\code{country}}{A string of country name recorded in the \code{LOCATION}
header.}

\item{\code{data_source}}{A string of data source recorded in the \code{LOCATION}
header.}

\item{\code{wmo_number}}{A string of WMO (World Meteorological Organization)
number recorded in the \code{LOCATION} header.}

\item{\code{latitude}}{A number of latitude recorded in the \code{LOCATION}
header. North latitude is positive and south latitude is
negative. Should in range \verb{[-90, +90]}.}

\item{\code{longitude}}{A number of longitude recorded in the \code{LOCATION}
header. East longitude is positive and west longitude is
negative. Should in range \verb{[-180, +180]}.}

\item{\code{time_zone}}{A number of time zone recorded in the \code{LOCATION}
header. Usually presented as the offset hours from UTC time.
Should in range \verb{[-12, +14]}.}

\item{\code{elevation}}{A number of elevation recorded in the \code{LOCATION}
header. Should in range \verb{[-1000, 9999.9)}.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$location()} takes new values for \code{LOCATION} header fields and
returns the parsed values of \code{LOCATION} header in a list format. If
no input is given, current \code{LOCATION} header value is returned.
}

\subsection{Returns}{
A named list of 9 elements.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
epw$location()

# modify location data
epw$location(city = "MyCity")
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Epw-design_condition"></a>}}
\if{latex}{\out{\hypertarget{method-Epw-design_condition}{}}}
\subsection{Method \code{design_condition()}}{
Get DESIGN CONDITION header
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Epw$design_condition()}\if{html}{\out{</div>}}
}

\subsection{Details}{
\verb{$design_condition()} returns the parsed values of \verb{DESIGN CONDITION}
header in a list format with 4 elements:
\itemize{
\item \code{source}: A string of source field
\item \code{heating}: A list, usually of length 16, of the heading design conditions
\item \code{cooling}: A list, usually of length 32, of the cooling design conditions
\item \code{extremes}: A list, usually of length 16, of the extreme design conditions
}

For the meaning of each element, please see ASHRAE Handbook of Fundamentals.
}

\subsection{Returns}{
A named list of 4 elements.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
epw$design_condition()
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Epw-typical_extreme_period"></a>}}
\if{latex}{\out{\hypertarget{method-Epw-typical_extreme_period}{}}}
\subsection{Method \code{typical_extreme_period()}}{
Get TYPICAL/EXTREME header
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Epw$typical_extreme_period()}\if{html}{\out{</div>}}
}

\subsection{Details}{
\verb{$typical_extreme_period()} returns the parsed values of \verb{TYPICAL/EXTREME PERIOD} header in a \link[data.table:data.table]{data.table} format with 6
columns:
\itemize{
\item \code{index}: Integer type. The index of typical or extreme period record
\item \code{name}: Character type. The name of typical or extreme period record
\item \code{type}: Character type. The type of period. Possible value: \code{typical} and
\code{extreme}
\item \code{start_day}: Date type with customized formatting. The start day of the
period
\item \code{start_day}: Date type with customized formatting. The end day of the
period
}
}

\subsection{Returns}{
A \code{\link[data.table:data.table]{data.table::data.table()}} with 6 columns.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
epw$typical_extreme_period()
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Epw-ground_temperature"></a>}}
\if{latex}{\out{\hypertarget{method-Epw-ground_temperature}{}}}
\subsection{Method \code{ground_temperature()}}{
Get GROUND TEMPERATURE header
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Epw$ground_temperature()}\if{html}{\out{</div>}}
}

\subsection{Details}{
\verb{$ground_temperature()} returns the parsed values of \verb{GROUND TEMPERATURE}
header in a \link[data.table:data.table]{data.table} format with 17 columns:
\itemize{
\item \code{index}: Integer type. The index of ground temperature record
\item \code{depth}: Numeric type. The depth of the ground temperature is measured
\item \code{soil_conductivity}: Numeric type. The soil conductivity at measured depth
\item \code{soil_density}: Numeric type. The soil density at measured depth
\item \verb{soil_specific heat}: Numeric type. The soil specific heat at measured depth
\item \code{January} to \code{December}: Numeric type. The measured group
temperature for each month.
}
}

\subsection{Returns}{
A \code{\link[data.table:data.table]{data.table::data.table()}} with 17 columns.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
epw$ground_temperature()
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Epw-holiday"></a>}}
\if{latex}{\out{\hypertarget{method-Epw-holiday}{}}}
\subsection{Method \code{holiday()}}{
Get and modify HOLIDAYS/DAYLIGHT SAVINGS header
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Epw$holiday(leapyear, dst, holiday)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{leapyear}}{Either \code{TRUE} or \code{FALSE}.}

\item{\code{dst}}{A length 2 EPW date specifications identifying the start
and end of daylight saving time. For example, \code{c(3.10, 10.3)}.}

\item{\code{holiday}}{a list or a data.frame containing two elements
(columns) \code{name} and \code{day} where \code{name} are the holiday names
and \code{day} are valid EPW date specifications. For example:

\if{html}{\out{<div class="sourceCode">}}\preformatted{list(name = c("New Year's Day", "Christmas Day"), day = c("1.1", "25 Dec"))
}\if{html}{\out{</div>}}}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$holiday()} takes new value for leap year indicator, daylight saving time
and holiday specifications, set these new values and returns the parsed values
of \verb{HOLIDAYS/DAYLIGHT SAVINGS} header. If no input is given, current values
of \verb{HOLIDAYS/DAYLIGHT SAVINGS} header is returned. It returns a list of 3
elements:
\itemize{
\item \code{leapyear}: A single logical vector. \code{TRUE} means that the weather data
contains leap year data
\item \code{dst}: A Date vector contains the start and end day of daylight saving time
\item \code{holiday}: A \link[data.table:data.table]{data.table} contains 2 columns. If
no holiday specified, an empty \link[data.table:data.table]{data.table}
\itemize{
\item \code{name}: Name of the holiday
\item \code{day}: Date of the holiday
}
}

Validation process below is performed when changing the \code{leapyear}
indicator:
\itemize{
\item If current record of \code{leapyear} is \code{TRUE}, but new input is \code{FALSE}, the
modification is only conducted when all data periods do not cover Feb 29.
\item If current record of \code{leapyear} is \code{FALSE}, but new input is \code{TRUE}, the
modification is only conducted when TMY data periods do not across Feb,
e.g. [01/02, 02/28], [03/01, 12/31]; for AMY data, it is always OK.
}

The date specifications in \code{dst} and \code{holiday} should follow the rules of
\strong{"Table 2.14: Weather File Date File Interpretation"} in
"AuxiliaryPrograms" documentation. eplusr is able to handle all those kinds of
formats automatically. Basically, 5 formats are allowed:
\enumerate{
\item A single integer is interpreted as the Julian day of year. For example,
\code{1}, \code{2}, \code{3} and \code{4} will be parsed and presented as \verb{1st day}, \verb{2nd day}, \verb{3rd day} and \verb{4th day}.
\item A single number is interpreted as \code{Month.Day}. For example, \code{1.2} and \code{5.6}
will be parsed and presented as \verb{Jan 02} and \verb{May 06}.
\item A string giving \code{MonthName / DayNumber}, \code{DayNumber / MonthName}, and
\code{MonthNumber / DayNumber}. A year number can be also included. For
example, \code{"Jan/1"}, \code{"05/Dec"}, \code{"7/8"}, \code{"02/10/2019"}, and
\code{"2019/04/05"} will be parsed and presented as \verb{Jan 02}, \verb{Dec 06}, \verb{Jul 8}, \code{2019-02-10} and \code{2019-04-15}.
\item A string giving \verb{number Weekday in Month}. For example, \code{"2 Sunday in Jan"} will be parsed and presented as \verb{2th Sunday in January}.
\item A string giving \verb{Last Weekday in Month}. For example, \code{"last Sunday in Dec"} will be parsed and presented as \verb{Last Sunday in December}.
}

For convenience, besides all the formats described above, \code{dst} and days in
\code{holiday} also accept standard Dates input. They will be treated as the same
way as No.3 format described above.
}

\subsection{Returns}{
A named list of 3 elements.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
epw$holiday()

# add daylight saving time
epw$holiday(dst = c(3.10, 11.3))
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Epw-comment1"></a>}}
\if{latex}{\out{\hypertarget{method-Epw-comment1}{}}}
\subsection{Method \code{comment1()}}{
Get and modify COMMENT1 header
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Epw$comment1(comment)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{comment}}{A string of new comments.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$comment1()} takes a single string of new comments and replaces the
old comment with input one. If \code{NULL} is given, the comment is
removed. Empty string or a string that contains only spaces will be
treated as \code{NULL}. If no input is given, current comment is returned.
If no comments exist, \code{NULL} is returned.
}

\subsection{Returns}{
A single string.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
epw$comment1()

epw$comment1("Comment1")
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Epw-comment2"></a>}}
\if{latex}{\out{\hypertarget{method-Epw-comment2}{}}}
\subsection{Method \code{comment2()}}{
Get and modify COMMENT2 header
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Epw$comment2(comment)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{comment}}{A string of new comments.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$comment2()} takes a single string of new comments and replaces the
old comment with input one. If \code{NULL} is given, the comment is
removed. Empty string or a string that contains only spaces will be
treated as \code{NULL}. If no input is given, current comment is returned.
If no comments exist, \code{NULL} is returned.
}

\subsection{Returns}{
A single string.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
epw$comment2()

epw$comment2("Comment2")
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Epw-num_period"></a>}}
\if{latex}{\out{\hypertarget{method-Epw-num_period}{}}}
\subsection{Method \code{num_period()}}{
Get number of data periods in DATA PERIODS header
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Epw$num_period()}\if{html}{\out{</div>}}
}

\subsection{Details}{
\verb{$num_period()} returns a single positive integer of how many data
periods current \code{Epw} contains.
}

\subsection{Returns}{
A single integer.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
epw$num_period()
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Epw-interval"></a>}}
\if{latex}{\out{\hypertarget{method-Epw-interval}{}}}
\subsection{Method \code{interval()}}{
Get the time interval in DATA PERIODS header
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Epw$interval()}\if{html}{\out{</div>}}
}

\subsection{Details}{
\verb{$interval()} returns a single positive integer of how many records
of weather data exist in one hour.
}

\subsection{Returns}{
A single integer.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
epw$interval()
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Epw-period"></a>}}
\if{latex}{\out{\hypertarget{method-Epw-period}{}}}
\subsection{Method \code{period()}}{
Get and modify data period meta data in DATA PERIODS header
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Epw$period(period, name, start_day_of_week)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{period}}{A positive integer vector identifying the data period
indexes.}

\item{\code{name}}{A character vector used as new names for specified data
periods. Should have the same length as \code{index}.}

\item{\code{start_day_of_week}}{A character vector or an integer vector used
as the new start days of week of specified data periods.
Should have the same length as \code{index}.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$period()} takes a data period index, a new period name and start
day of week specification, and uses that input to replace the data
period's name and start day of week. If no input is given, data
periods in current \code{Epw} is returned.
}

\subsection{Returns}{
A \link[data.table:data.table]{data.table} with 5 columns:
\itemize{
\item \code{index}: Integer type. The index of data period.
\item \code{name}: Character type. The name of data period.
\item \code{start_day_of_week}: Character type. The start day of week of data period.
\item \code{start_day}: Date (EpwDate) type. The start day of data period.
\item \code{end_day}: Date (EpwDate) type. The end day of data period.
}
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# modify data period name
epw$period(1, name = "test")

# change start day of week
epw$period(1, start_day_of_week = 3)
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Epw-missing_code"></a>}}
\if{latex}{\out{\hypertarget{method-Epw-missing_code}{}}}
\subsection{Method \code{missing_code()}}{
Get missing code for weather data variables
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Epw$missing_code()}\if{html}{\out{</div>}}
}

\subsection{Details}{
\verb{$missing_code()} returns a list of 29 elements containing the value
used as missing value identifier for all weather data.
}

\subsection{Returns}{
A named list of 29 elements.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
epw$missing_code()
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Epw-initial_missing_value"></a>}}
\if{latex}{\out{\hypertarget{method-Epw-initial_missing_value}{}}}
\subsection{Method \code{initial_missing_value()}}{
Get initial value for missing data of weather data variables
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Epw$initial_missing_value()}\if{html}{\out{</div>}}
}

\subsection{Details}{
\verb{$initial_missing_value()} returns a list of 16 elements containing
the initial value used to replace missing values for corresponding
weather data.
}

\subsection{Returns}{
A named list of 16 elements.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
epw$initial_missing_value()
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Epw-range_exist"></a>}}
\if{latex}{\out{\hypertarget{method-Epw-range_exist}{}}}
\subsection{Method \code{range_exist()}}{
Get value ranges for existing values of weather data variables
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Epw$range_exist()}\if{html}{\out{</div>}}
}

\subsection{Details}{
\verb{$range_exist()} returns a list of 28 elements containing the range
each numeric weather data should fall in. Any values out of this
range are treated as missing.
}

\subsection{Returns}{
A named list of 28 elements.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
epw$range_exist()
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Epw-range_valid"></a>}}
\if{latex}{\out{\hypertarget{method-Epw-range_valid}{}}}
\subsection{Method \code{range_valid()}}{
Get value ranges for valid values of weather data variables
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Epw$range_valid()}\if{html}{\out{</div>}}
}

\subsection{Details}{
\verb{$range_valid()} returns a list of 28 elements containing the range
each numeric weather data should fall in. Any values out of this
range are treated as invalid.
}

\subsection{Returns}{
A named list of 28 elements.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
epw$range_valid()
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Epw-fill_action"></a>}}
\if{latex}{\out{\hypertarget{method-Epw-fill_action}{}}}
\subsection{Method \code{fill_action()}}{
Get fill actions for abnormal values of weather data variables
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Epw$fill_action(type = c("missing", "out_of_range"))}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{type}}{What abnormal type of actions to return. Should be one of
\code{"missing"} and \code{"out_of_range"}. Default: \code{"missing"}.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$fill_action()} returns a list containing \code{action}s that EnergyPlus
will perform when certain abnormal data found for corresponding
weather data. There are 3 types of actions in total:
\itemize{
\item \code{do_nothing}: All abnormal values are left as they are.
\item \code{use_zero}: All abnormal values are reset to zeros.
\item \code{use_previous}: The first abnormal values of variables will be set to the
initial missing values. All after are set to previous valid one.
}
}

\subsection{Returns}{
A named list.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
epw$fill_action("missing")

epw$fill_action("out_of_range")
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Epw-data"></a>}}
\if{latex}{\out{\hypertarget{method-Epw-data}{}}}
\subsection{Method \code{data()}}{
Get weather data
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Epw$data(
  period = 1L,
  start_year = NULL,
  align_wday = TRUE,
  tz = "UTC",
  update = FALSE,
  line = FALSE
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{period}}{A single positive integer identifying the data period
index. Data periods information can be obtained using
\href{../../eplusr/html/Epw.html#method-period}{\code{$period()}}
described above.}

\item{\code{start_year}}{A positive integer identifying the year of first
date time in specified data period. If \code{NULL}, the values in
the \code{year} column are used as years of \code{datetime} column.
Default: \code{NULL}.}

\item{\code{align_wday}}{Only applicable when \code{start_year} is \code{NULL}. If
\code{TRUE}, a year value is automatically calculated for specified
data period that compliance with the start day of week value
specified in \verb{DATA PERIODS} header.}

\item{\code{tz}}{A valid time zone to be assigned to the \code{datetime} column.
All valid time zone names can be obtained using
\code{OlsonNames()}. Default:\code{"UTC"}.}

\item{\code{update}}{If \code{TRUE}, the \code{year} column are updated according to
the newly created \code{datetime} column using \code{start_year}. If
\code{FALSE}, original year data in the \code{Epw} object is kept.
Default: \code{FALSE}.}

\item{\code{line}}{If \code{TRUE}, a column named \code{line} is prepended indicating
the line numbers where data occur in the actual EPW file.
Default: \code{FALSE}.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$data()} returns weather data of specific data period.

Usually, EPW file downloaded from \href{https://energyplus.net/}{EnergyPlus website}
contains TMY weather data. As years of weather data is not
consecutive, it may be more convenient to align the year values to be
consecutive, which will makes it possible to direct analyze and plot
weather data. The \code{start_year} argument in \verb{$data()} method can help
to achieve this. However, randomly setting the \code{year} may result in a
date time series that does not have the same start day of week as
specified in the \verb{DATA PERIODS} header.  eplusr provides a simple
solution for this. By setting \code{year} to \code{NULL} and \code{align_wday} to
\code{TRUE}, eplusr will calculate a year value (from current year
backwards) for each data period that compliance with the start day of
week restriction.

Note that if current data period contains AMY data and \code{start_year}
is given, a warning is given because the actual year values will be
overwritten by input \code{start_year}. An error is given if:
\itemize{
\item Using input \code{start_year} introduces invalid date time. This may
happen when weather data contains leap year but input \code{start_year}
is not a leap year.
\item Applying specified time zone specified using \code{tz} introduces
invalid date time.
}
}

\subsection{Returns}{
A \code{\link[data.table:data.table]{data.table::data.table()}} of 36 columns.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# get weather data
str(epw$data())

# get weather data but change the year to 2018
# the year column is not changed by default, only the returned datetime column
head(epw$data(start_year = 2018)$datetime)
str(epw$data(start_year = 2018)$year)
# you can update the year column too
head(epw$data(start_year = 2018, update = TRUE)$year)

# change the time zone of datetime column in the returned weather data
attributes(epw$data()$datetime)
attributes(epw$data(tz = "Etc/GMT+8")$datetime)
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Epw-abnormal_data"></a>}}
\if{latex}{\out{\hypertarget{method-Epw-abnormal_data}{}}}
\subsection{Method \code{abnormal_data()}}{
Get abnormal weather data
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Epw$abnormal_data(
  period = 1L,
  cols = NULL,
  keep_all = TRUE,
  type = c("both", "missing", "out_of_range")
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{period}}{A single positive integer identifying the data period
index. Data periods information can be obtained using
\href{../../eplusr/html/Epw.html#method-period}{\code{$period()}}
described above.}

\item{\code{cols}}{A character vector identifying what data columns, i.e.
all columns except \code{datetime}, \code{year}, \code{month}, \code{day}, \code{hour}
\code{minute}, and character columns, to search abnormal values. If
\code{NULL}, all data columns are used. Default: \code{NULL}.}

\item{\code{keep_all}}{If \code{TRUE}, all columns are returned. If \code{FALSE}, only
\code{line}, \code{datetime}, \code{year}, \code{month}, \code{day}, \code{hour} and
\code{minute}, together with columns specified in \code{cols} are
returned. Default: \code{TRUE}}

\item{\code{type}}{What abnormal type of data to return. Should be one of
\code{"all"}, \code{"missing"} and \code{"out_of_range"}. Default: \code{"all"}.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$abnormal_data()} returns abnormal data of specific data period.
Basically, there are 2 types of abnormal data in \code{Epw}, i.e. missing
values and out-of-range values. Sometimes, it may be useful to
extract and inspect those data especially when inserting measured
weather data. \verb{$abnormal_data()} does this.

In the returned \code{\link[data.table:data.table]{data.table::data.table()}}, a column named \code{line}
is created indicating the line numbers where abnormal data occur in
the actual EPW file.
}

\subsection{Returns}{
A \code{\link[data.table:data.table]{data.table::data.table()}}.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
epw$abnormal_data()

# only check if there are any abnormal values in air temperature and
# liquid precipitation rate
epw$abnormal_data(cols = c("dry_bulb_temperature", "liquid_precip_rate"))

# save as above, but only return date time columns plus those 2 columns
epw$abnormal_data(cols = c("dry_bulb_temperature", "liquid_precip_rate"),
    keep_all = FALSE
)

# same as above, but only check for missing values
epw$abnormal_data(cols = c("dry_bulb_temperature", "liquid_precip_rate"),
    type = "missing"
)

# same as above, but only check for out-of-range values
epw$abnormal_data(cols = c("dry_bulb_temperature", "liquid_precip_rate"),
    type = "out_of_range"
)
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Epw-redundant_data"></a>}}
\if{latex}{\out{\hypertarget{method-Epw-redundant_data}{}}}
\subsection{Method \code{redundant_data()}}{
Get redundant weather data
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Epw$redundant_data()}\if{html}{\out{</div>}}
}

\subsection{Details}{
\verb{$redundant_data()} returns weather data in \code{Epw} object that do not
belong to any data period. This data can be further removed using
\href{../../eplusr/html/Epw.html#method-purge}{\code{$purge()}}`
method described below.

In the returned \code{\link[data.table:data.table]{data.table::data.table()}}, a column named \code{line}
is created indicating the line numbers where redundant data occur in
the actual EPW file.
}

\subsection{Returns}{
A \code{\link[data.table:data.table]{data.table::data.table()}} of 37 columns.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
epw$redundant_data()
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Epw-make_na"></a>}}
\if{latex}{\out{\hypertarget{method-Epw-make_na}{}}}
\subsection{Method \code{make_na()}}{
Convert abnormal data into NAs
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Epw$make_na(missing = FALSE, out_of_range = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{missing}}{If \code{TRUE}, missing values are included. Default:
\code{FALSE}.}

\item{\code{out_of_range}}{If \code{TRUE}, out-of-range values are included.
Default: \code{FALSE}.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$make_na()} converts specified abnormal data into \code{NA}s in specified
data period. This makes it easier to find abnormal data directly
using \code{is.na()} instead of using
\href{../../eplusr/html/Epw.html#method-missing_code}{\code{$missing_code()}}

\verb{$make_na()} and
\href{../../eplusr/html/Epw.html#method-fill_abnormal}{\code{$fill_abnormal()}}
are reversible, i.e.
\verb{$make_na()} can be used to counteract the effects introduced by
\href{../../eplusr/html/Epw.html#method-make_na}{\code{$make_na()}},
and vise a versa.

\strong{Note} that \verb{$make_na} modify the weather data in-place, meaning
that the returned data from
\href{../../eplusr/html/Epw.html#method-data}{\code{$data()}}
and
\href{../../eplusr/html/Epw.html#method-abnormal_data}{\code{$abnormal_data()}}
may be different after calling \verb{$make_na()}.
}

\subsection{Returns}{
The modified \code{Epw} object itself, invisibly.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# turn all missing values into NAs
summary(epw$data()$liquid_precip_rate)
epw$make_na(missing = TRUE)
summary(epw$data()$liquid_precip_rate)
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Epw-fill_abnormal"></a>}}
\if{latex}{\out{\hypertarget{method-Epw-fill_abnormal}{}}}
\subsection{Method \code{fill_abnormal()}}{
Fill abnormal data using prescribed pattern
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Epw$fill_abnormal(missing = FALSE, out_of_range = FALSE, special = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{missing}}{If \code{TRUE}, missing values are included. Default:
\code{FALSE}.}

\item{\code{out_of_range}}{If \code{TRUE}, out-of-range values are included.
Default: \code{FALSE}.}

\item{\code{special}}{If \code{TRUE}, abnormal data are filled using
corresponding actions listed
\href{../../eplusr/html/Epw.html#method-fill_action}{\code{$fill_action()}}.
If \code{FALSE}, all abnormal data are fill with missing code
described in
\href{../../eplusr/html/Epw.html#method-missing_code}{\code{$missing_code()}}.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$fill_abnormal()} fills specified abnormal data using corresponding
actions listed in
\href{../../eplusr/html/Epw.html#method-fill_action}{\code{$fill_action()}}.
For what kinds of actions to be performed, please see
\href{../../eplusr/html/Epw.html#method-fill_action}{\code{$fill_action()}}.
method described above. Note that only if \code{special} is \code{TRUE},
special actions listed in \verb{$fill_action()} is performed. If \code{special}
is \code{FALSE}, all abnormal data, including both missing values and
out-of-range values, are filled with corresponding missing codes.

\href{../../eplusr/html/Epw.html#method-make_na}{\code{$make_na()}}
and \verb{$fill_abnormal()} are reversible, i.e.
\href{../../eplusr/html/Epw.html#method-make_na}{\code{$make_na()}}
can be used to counteract the effects introduced by
\verb{$fill_abnormal()}, and vise a versa.

\strong{Note} that \verb{$fill_abnormal} modify the weather data in-place,
meaning that the returned data from
\href{../../eplusr/html/Epw.html#method-data}{\code{$data()}}
and
\href{../../eplusr/html/Epw.html#method-abnormal_data}{\code{$abnormal_data()}}
may be different after calling \verb{$fill_abnormal()}.
}

\subsection{Returns}{
The modified \code{Epw} object itself, invisibly.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# turn all missing values into NAs
summary(epw$data()$liquid_precip_rate)
epw$fill_abnormal(missing = TRUE)
summary(epw$data()$liquid_precip_rate)
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Epw-add_unit"></a>}}
\if{latex}{\out{\hypertarget{method-Epw-add_unit}{}}}
\subsection{Method \code{add_unit()}}{
Add units to weather data variables
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Epw$add_unit()}\if{html}{\out{</div>}}
}

\subsection{Details}{
\verb{$add_unit()} assigns units to numeric weather data using
\code{\link[units:units]{units::set_units()}} if applicable.

\verb{$add_unit()}
and
\href{../../eplusr/html/Epw.html#method-drop_unit}{\code{$drop_unit()}}
are reversible, i.e.
\verb{$add_unit()}
can be used to counteract the effects introduced by
\href{../../eplusr/html/Epw.html#method-drop_unit}{\code{$drop_unit()}},
and vise a versa.

\strong{Note} that \verb{$add_unit} modify the weather data in-place,
meaning that the returned data from
\href{../../eplusr/html/Epw.html#method-data}{\code{$data()}}
and
\href{../../eplusr/html/Epw.html#method-abnormal_data}{\code{$abnormal_data()}}
may be different after calling \verb{$add_unit()}.
}

\subsection{Returns}{
The modified \code{Epw} object itself, invisibly.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# get weather data with units
epw$add_unit()
head(epw$data())

# with units specified, you can easily perform unit conversion using units
# package
t_dry_bulb <- epw$data()$dry_bulb_temperature
units(t_dry_bulb) <- with(units::ud_units, "kelvin")

head(t_dry_bulb)
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Epw-drop_unit"></a>}}
\if{latex}{\out{\hypertarget{method-Epw-drop_unit}{}}}
\subsection{Method \code{drop_unit()}}{
Remove units in weather data variables
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Epw$drop_unit()}\if{html}{\out{</div>}}
}

\subsection{Details}{
\verb{$drop_unit()} removes all units of numeric weather data.

\href{../../eplusr/html/Epw.html#method-add_unit}{\code{$add_unit()}}
and
\verb{$drop_unit()}
are reversible, i.e.
\href{../../eplusr/html/Epw.html#method-drop_unit}{\code{$add_unit()}}
can be used to counteract the effects introduced by
\verb{$drop_unit()},
and vise a versa.

\strong{Note} that \verb{$add_unit} modify the weather data in-place,
meaning that the returned data from
\href{../../eplusr/html/Epw.html#method-data}{\code{$data()}}
and
\href{../../eplusr/html/Epw.html#method-abnormal_data}{\code{$abnormal_data()}}
may be different after calling \verb{$add_unit()}.
}

\subsection{Returns}{
The modified \code{Epw} object itself, invisibly.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
epw$drop_unit()
epw$data()
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Epw-purge"></a>}}
\if{latex}{\out{\hypertarget{method-Epw-purge}{}}}
\subsection{Method \code{purge()}}{
Delete redundant weather data observations
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Epw$purge()}\if{html}{\out{</div>}}
}

\subsection{Details}{
\verb{$purge()} deletes weather data in \code{Epw} object that do not belong to
any data period.
}

\subsection{Returns}{
The modified \code{Epw} object itself, invisibly.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
epw$purge()
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Epw-add"></a>}}
\if{latex}{\out{\hypertarget{method-Epw-add}{}}}
\subsection{Method \code{add()}}{
Add a data period
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Epw$add(
  data,
  realyear = FALSE,
  name = NULL,
  start_day_of_week = NULL,
  after = 0L
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{data}}{A \code{\link[data.table:data.table]{data.table::data.table()}} of new weather data to add
or set. Validation is performed according to rules described
above.}

\item{\code{realyear}}{Whether input data is AMY data. Default: \code{FALSE}.}

\item{\code{name}}{A new string used as name of added or set data period.
Should not be the same as existing data period names. If
\code{NULL}, it is generated automatically in format \code{Data},
\code{Data_1} and etc., based on existing data period names.
Default: \code{NULL}}

\item{\code{start_day_of_week}}{A single integer or character specifying
start day of week of input data period. If \code{NULL}, Sunday is
used for TMY data and the actual start day of week is used for
AMY data.  Default: \code{NULL}.}

\item{\code{after}}{A single integer identifying the index of data period
where input new data period to be inserted after. IF \code{0},
input new data period will be the first data period. Default:
\code{0}.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$add()} adds a new data period into current \code{Epw} object at
specified position.

The validity of input data is checked before adding according to
rules following:
\itemize{
\item Column \code{datetime} exists and has type of \code{POSIXct}. Note that time
zone of input date time will be reset to \code{UTC}.
\item It assumes that input data is already sorted, i.e. no further
sorting is made during validation. This is because when input data
is TMY data, there is no way to properly sort input data rows only
using \code{datetime} column.
\item Number of data records per hour should be consistent across input
data.
\item Input number of data records per hour should be the same as
existing data periods.
\item The date time of input data should not overlap with existing data
periods.
\item Input data should have all 29 weather data columns with correct
types. The \code{year}, \code{month}, \code{day}, and \code{minute} column are not
compulsory. They will be created according to values in the
\code{datetime} column. Existing values will be overwritten.
}
}

\subsection{Returns}{
The modified \code{Epw} object itself, invisibly.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# will fail since date time in input data has already been covered by
# existing data period
try(epw$add(epw$data()), silent = TRUE)
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Epw-set"></a>}}
\if{latex}{\out{\hypertarget{method-Epw-set}{}}}
\subsection{Method \code{set()}}{
Replace a data period
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Epw$set(
  data,
  realyear = FALSE,
  name = NULL,
  start_day_of_week = NULL,
  period = 1L
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{data}}{A \code{\link[data.table:data.table]{data.table::data.table()}} of new weather data to add
or set. Validation is performed according to rules described
above.}

\item{\code{realyear}}{Whether input data is AMY data. Default: \code{FALSE}.}

\item{\code{name}}{A new string used as name of added or set data period.
Should not be the same as existing data period names. If
\code{NULL}, it is generated automatically in format \code{Data},
\code{Data_1} and etc., based on existing data period names.
Default: \code{NULL}}

\item{\code{start_day_of_week}}{A single integer or character specifying
start day of week of input data period. If \code{NULL}, Sunday is
used for TMY data and the actual start day of week is used for
AMY data.  Default: \code{NULL}.}

\item{\code{period}}{A single integer identifying the index of data period
to set.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$set()} replaces existing data period using input new weather data.

The validity of input data is checked before replacing according to
rules following:
\itemize{
\item Column \code{datetime} exists and has type of \code{POSIXct}. Note that time
zone of input date time will be reset to \code{UTC}.
\item It assumes that input data is already sorted, i.e. no further
sorting is made during validation. This is because when input data
is TMY data, there is no way to properly sort input data rows only
using \code{datetime} column.
\item Number of data records per hour should be consistent across input
data.
\item Input number of data records per hour should be the same as
existing data periods.
\item The date time of input data should not overlap with existing data
periods.
\item Input data should have all 29 weather data columns with right
types. The \code{year}, \code{month}, \code{day}, and \code{minute} column are not
compulsory. They will be created according to values in the
\code{datetime} column. Existing values will be overwritten.
}
}

\subsection{Returns}{
The modified \code{Epw} object itself, invisibly.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# change the weather data
epw$set(epw$data())
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Epw-del"></a>}}
\if{latex}{\out{\hypertarget{method-Epw-del}{}}}
\subsection{Method \code{del()}}{
Delete a data period
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Epw$del(period)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{period}}{A single integer identifying the index of data period
to set.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$del()} removes a specified data period. Note that an error will be
given if current \code{Epw} only contains one data period.
}

\subsection{Returns}{
The modified \code{Epw} object itself, invisibly.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Epw-is_unsaved"></a>}}
\if{latex}{\out{\hypertarget{method-Epw-is_unsaved}{}}}
\subsection{Method \code{is_unsaved()}}{
Check if there are unsaved changes in current \code{Epw}
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Epw$is_unsaved()}\if{html}{\out{</div>}}
}

\subsection{Details}{
\verb{$is_unsaved()} returns \code{TRUE} if there are modifications on the
\code{Epw} object since it was read or since last time it was saved, and
returns \code{FALSE} otherwise.
}

\subsection{Returns}{
A single logical value of \code{TRUE} or \code{FALSE}.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
epw$is_unsaved()
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Epw-save"></a>}}
\if{latex}{\out{\hypertarget{method-Epw-save}{}}}
\subsection{Method \code{save()}}{
Save \code{Epw} object as an EPW file
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Epw$save(path = NULL, overwrite = FALSE, purge = FALSE, format_digit = TRUE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{path}}{A path where to save the weather file. If \code{NULL}, the
path of the weather file itself is used. Default: \code{NULL}.}

\item{\code{overwrite}}{Whether to overwrite the file if it already exists.
Default is \code{FALSE}.}

\item{\code{purge}}{Whether to remove redundant data when saving. Default:
\code{FALSE}.}

\item{\code{format_digit}}{Whether to remove trailing digits in weather
data. Default: \code{TRUE}.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
\verb{$save()} saves current \code{Epw} to an EPW file. Note that if missing
values and out-of-range values are converted to \code{NA}s using
\href{../../eplusr/html/Epw.html#method-make_na}{\code{$make_na()}},
they will be filled with corresponding missing codes during saving.
}

\subsection{Returns}{
A length-one character vector, invisibly.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
# save the weather file
epw$save(file.path(tempdir(), "weather.epw"), overwrite = TRUE)
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Epw-print"></a>}}
\if{latex}{\out{\hypertarget{method-Epw-print}{}}}
\subsection{Method \code{print()}}{
Print \code{Idf} object
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Epw$print()}\if{html}{\out{</div>}}
}

\subsection{Details}{
\verb{$print()} prints the \code{Epw} object, including location, elevation,
data source, WMO station, leap year indicator, interval and data
periods.
}

\subsection{Returns}{
The \code{Epw} object itself, invisibly.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\dontrun{
epw$print()
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-Epw-clone"></a>}}
\if{latex}{\out{\hypertarget{method-Epw-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Epw$clone(deep = TRUE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
