%  File ergm/man/ergm.exact.Rd
%  Part of the statnet package, http://statnetproject.org
%
%  This software is distributed under the GPL-3 license.  It is free,
%  open source, and has the attribution requirements (GPL Section 7) in
%    http://statnetproject.org/attribution
%
%  Copyright 2011 the statnet development team
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
\name{ergm.exact}
\alias{ergm.exact}
\title{Calculate the exact loglikelihood for an ERGM}
\usage{
ergm.exact (eta, formula, statmat=NULL, weights=NULL, ...)
}
\description{
\code{ergm.exact} calculates the exact loglikelihood, evaluated at \code{eta}, 
  for the 
 \code{statnet} exponential-family random graph model represented by
 \code{formula}.
}
\arguments{
 \item{eta}{vector of canonical parameter values at which the loglikelihood 
 should be evaluated.}
 \item{formula}{an \R \code{link{formula}} object of the form
   \code{y ~ <model terms>},
   where \code{y} is a network object or a matrix that can be
   coerced to a \code{\link[network]{network}} object.  For the details 
   on the possible
   \code{<model terms>}, see \code{\link{ergm-terms}}.  To create a
   \code{\link[network]{network}} object in \R, use the 
   \code{network()} function,
   then add nodal attributes to it using the \code{\%v\%}
   operator if necessary.}
 \item{statmat}{if NULL, call \code{\link{ergm.allstats}} to generate all 
 possible graph statistics for the networks in this model.}
 \item{weights}{In case \code{statmat} is not \code{NULL}, this should be the
  vector of counts corresponding to the rows of \code{statmat}. If
  \code{statmat} is \code{NULL}, this is generated by the call to 
  \code{\link{ergm.allstats}}.}
 \item{\dots}{further arguments; not currently used.}
}

\details{
 \code{ergm.exact} should only be used for small networks, since the number of
 possible networks grows extremely fast with the number of nodes.
 An error results if it is used on a directed network of more than 6 nodes or 
 an undirected network of more than 8 nodes; use \code{force=TRUE} to override
 this error.  
 
 In case this function is to be called repeatedly, for instance by an 
 optimization routine, it is preferable to call \code{\link{ergm.allstats}}
 first, then pass \code{statmat} and \code{weights} explicitly to avoid
 repeatedly calculating these objects.
}

\value{
Returns the value of the
exact loglikelihood, evaluated at \code{eta}, 
  for the 
 \code{statnet} exponential-family random graph model represented by
 \code{formula}.
}
\seealso{\code{\link{ergm.allstats}}}
\examples{
# Count by brute force all the edge statistics possible for a 7-node 
# undirected network
mynw <- network(matrix(0,7,7),dir=FALSE)
unix.time(a <- ergm.allstats(mynw~edges))

# Summarize results
rbind(t(a$statmat),a$weights)

# Each value of a$weights is equal to 21-choose-k, 
# where k is the corresponding statistic (and 21 is 
# the number of dyads in an 7-node undirected network).  
# Here's a check of that fact:
as.vector(a$weights - choose(21, t(a$statmat)))

# Simple ergm.exact outpuf for this network.  
# We know that the loglikelihood for my empty 7-node network
# should simply be -21*log(1+exp(eta)), so we may check that
# the following two values agree:
-21*log(1+exp(.1234)) 
ergm.exact(.1234, mynw~edges, statmat=a$statmat, weights=a$weights)
}
\keyword{models}
