% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/aux_picknetworkparallel.R
\name{aux_picknetworkparallel}
\alias{aux_picknetworkparallel}
\title{Pick seismic events with a sensor network approach, parallel version}
\usage{
aux_picknetworkparallel(
  start,
  stop,
  res,
  buffer,
  station,
  component,
  dir,
  f,
  envelope = TRUE,
  sta,
  lta,
  on,
  off,
  freeze,
  dur_min,
  dur_max,
  n_common,
  t_common,
  t_pause,
  cpu,
  ...
)
}
\arguments{
\item{start}{\code{POSIXct} value, start time for event detection analysis. 
If a character string (or vector) is provided, the function will try to 
convert it to POSIXct.}

\item{stop}{\code{POSIXct} value, start time for event detection analysis. 
If a character string (or vector) is provided, the function will try to 
convert it to POSIXct.}

\item{res}{\code{Numeric} value, time resolution of the snippets used to 
detect events, in seconds. A useful value would be one hour 
(\code{res = 3600}).}

\item{buffer}{\code{Numeric} vector, of length two. Size of the two-sided 
buffer to account for signal pre-processing effects, in seconds. The first 
value is used for the left side, the second for the right side. Events that 
happen to be detected during the buffer time will be removed from the output
data set.}

\item{station}{\code{Character} vector, seismic station IDs to be processed.}

\item{component}{\code{Character} value or vector, seismic component to 
be used. Information can be restricted to a single character, denoting the 
last character of the station component code, i.e. \code{"Z"} would use the 
first and last component of \code{c("BHZ", "HHE", "HHZ")}. If omitted, 
the function will use \code{"BHZ"} by default.}

\item{dir}{\code{Character} value, path to directory that contains the 
seismic files. See \code{read_data} for details and constraints on the 
data structure.}

\item{f}{\code{Numeric} Numeric vector of length two, cutoff frequencies 
(Hz) for the bandpass filter to be applied. If omitted, no filtering will 
be performed.}

\item{envelope}{\code{Logical} value, option to calculate signal envelopes
before detecting events. Default is \code{TRUE}.}

\item{sta}{\code{Numeric} value, duration of short-term window. Attention, 
this definition differs from the one in \code{pick_stalta}!}

\item{lta}{\code{Numeric} value, duration of long-term window. Attention, 
this definition differs from the one in \code{pick_stalta}!}

\item{on}{\code{Numeric} value, threshold value for event onset.}

\item{off}{\code{Numeric} value, threshold value for event end.}

\item{freeze}{\code{Logical} value, option to freeze lta value at start of 
an event. Useful to avoid self-adjustment of lta for long-duration events.}

\item{dur_min}{\code{Numeric} value, minimum duration of an event (s). Picks 
with a STA-LTA duration below this threshold will be discarded.}

\item{dur_max}{\code{Numeric} value, maximum duration of an event (s). Picks 
with a STA-LTA duration longer than this threshold will be discarded.}

\item{n_common}{\code{Numeric} value, minimum number of stations at which 
an event has to be registered.}

\item{t_common}{\code{Numeric} value, time (s) within which the picks of an 
event need to be registered by seismic stations.}

\item{t_pause}{\code{Numeric} value, time period after the onset of an event 
during which no other event is allowed to happen.}

\item{cpu}{\code{Numeric} value, fraction of CPUs to use. If omitted, 
only one CPU will be used.}

\item{\dots}{Further arguments passed to the function, i.e. keywords for 
the signal deconvolution part. See details for further information.}
}
\value{
\code{data frame}, detected events (start time, duration in 
seconds)
}
\description{
The function allows detection of discrete seismic events using the STA-LTA
picker approach and including several stations of a seismic network. It 
will pick events at all input stations and identify events picked by more 
than a user defined minimum number of stations, within a given time window.
It further allows rejecting events that are too short or too long, or occur 
to short after a previous event. The function can be used to process long 
data archives organised in a consistent file structure. This is the parallel 
computation version of the routine. For the single CPU version, allowing to 
show verbose progress information, use \code{aux_picknetwork()}.
}
\details{
The data will be imported time slice by time slice. Optionally, a signal 
deconvolution can be done. The data will be filtered to isolate the 
frequency range of interest, and tapered to account for edge effects. The 
taper size is automatically set be the two-sided buffer (see below). 
Optionally, a signal envelope can be calculated as part of the preprocessing
work. For all successfully imported and preprocessed signal time snippets, 
events will be detected by the STL-LTA approach. Events from all input 
signals will be checked for a minium and maximum allowed event duration, set 
as thresholds by the user. All remaining events are checked for the number 
of stations of the network that did consistently detect an event. That means
an event must be found in each signal within a user defined time window, a 
window that corresponds to the maximum travel time of a seismic signal 
through the network. The minimum number of stations that detect the event 
can be set by the user. In addition, events can be rejected if they occur 
during a user defined ban time after the onset of a previous event, which 
assumes that no subsequent events are allowed during the implementation of 
long duration events happening.   

The time period to process is defined by \code{start}, \code{end} and the 
length of time snippets \code{res} (s). In addition, a left sided and 
right sided buffer can and should be added \code{buffer = c(60, 60)} (s), 
to account for artefacts during signal preprocessing and events that 
occur across the time snippets. The \code{start} and \code{end} time should
be provided in \code{POSIXct} format. If it is provided as text string, the 
function will try to convert to POSIXct assuming the time zone is UTC.

There must be at least two seismic stations. The seismic components 
\code{component} must be a vector of the same length as \code{station} and 
contain the corresponding information for each station. 

Deconvolution of the input signals can be done, although usually this is
not essential for just picking events if the filter frequencies are in the 
flat response range of the sensors. If wanted, one needs to provide the 
additional vectors (all of the same length as \code{station}) \code{sensor}, 
\code{logger} and optionally \code{gain} (otherwise assumed to be \code{1}). 
See \code{signal_deconvolve()} for further information on the deconvolution 
routine and the supported keywords for sensors and loggers. 

STA-LTA picking is usually done on signal envelopes \code{enevelope = TRUE}. 
However, for array seismology or other specific cases it might be useful to 
work with the waveforms directly, instead of with their envelopes. 

The STA-LTA routine requires information on the length of the STA window, 
the LTA window, the on-threshold to define the start of an event, the 
off-threshold to define the end of an event, and the option to freeze the 
STA-LTA ratio at the onset of and event. See the documentation of the 
function \code{pick_stalta()} for further information. 

Events that last shorter than \code{dur_min} or longer than \code{dur_max} 
are rejected. This criterion is applied before the test of how many stations
have commonly detected an event. Hence, the minimum and maximum duration 
criteria need to be fulfilled by an event for all seismic stations, or at 
least as many as defined as minimum number of stations \code{n_common}.

A valid event should be detected by more than one station. At least two 
stations (\code{n_common = 2}) should detect it to confirm it is not just a
local effect like rain drop impacts or animal activity. At least three 
stations (\code{n_common = 3}) are needed to locate an event. Detection at
more stations increases the validity of an event as proper signal. However, 
the likelihood of stations not operating correctly or that some stations 
may be too far to detect a smaller event limits the number of stations that 
are be set as (\code{n_common}).

An event that happens just outside a seismic network will generate seismic 
waves that travel through the network and will require a given time for 
that. The time is set by the longest inter-station distance and the apparent 
seismic wave velocity. For example, an event who's signal propagated with 
1000 m/s through a network with a maximum aperture of 1000 m will need up to 
1 s. Hence, the parameter  \code{t_common} should be set to \code{1}, in 
this case. This parameter is good to reject events that show a longer travel
time, e.g. because they are pressure waves that travel through the 
atmosphere, like helicopter signals or lightning strikes.
}
\examples{


picks <- aux_picknetwork(start = "2017-04-09 01:00:00 UTC",
                         stop = "2017-04-09 02:00:00 UTC",
                         res = 1800,
                         buffer = c(90, 90),
                         station = c("RUEG1","RUEG2"),
                         component = rep("BHZ", 2),
                         dir = paste0(path.package("eseis"), "/extdata/"),
                         f = c(0.1, 0.2),
                         envelope = TRUE,
                         sta = 0.5,
                         lta = 300,
                         on = 3,
                         off = 1,
                         freeze = TRUE,
                         dur_min = 2,
                         dur_max = 90,
                         n_common = 2,
                         t_common = 5,
                         t_pause = 30)
                         
print(picks)

}
\author{
Michael Dietze
}
\keyword{eseis}
