#' esFinal
#
#' @description esFinal generates the final ESM dataset.
#
#' @param esDf a data.frame. A single ESM dataset. It must contain the 2 columns that hold the date-time object for when an ESM questionnaire was started and finished, respectively.
#
#' @param maxRows a numeric value. The number of data lines (per participant)  in the final event sampling dataset; must be equal for all participants. If no number is entered the maximum number across all participants is used.
#
#' @param RELEVANTVN_ES a list. This list is generated by function \code{\link{setES}} and it is extended once either by function \code{\link{genDateTime}} or by function \code{\link{splitDateTime}}.
#
#' @param RELEVANTINFO_ES a list. This list is generated by function \code{\link{setES}}.
#
#' @details The empty rows will either denote ESM questionnaires that were missed by the participant or it will denote fillers, i.e. rows of empty data to fill up the number of rows to be equal across all participants The number of maximum rows per participant either are computed by searching the actual maximum number of questionnaires started by the participant, or by what the user defines to be the maximum number of questionnaires.
#
#' @return \code{esDf} with empty rows of data added and with 2 additional columns MISSED and FILLER. MISSED refers to questionnaires that should have been answered by the participants but weren't. FILLER refers to empty rows of data in order for all participants to have equally many rows of data. See \strong{Details} for more information.
#
#' @examples
#' # o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o
#' # Prerequisites in order to execute esFinal. Start ------------------
#' # RELEVANTINFO_ES is delivered with the package
#' # Use example list delivered with the package
#' RELEVANTVN_ES <- RELEVANTVN_ESext
#' # tbsqDf is a raw ESM dataset, delivered with the package.
#' # Prerequisites in order to execute esFinal. End --------------------
#' # -------------------------------------------------------
#' # Run function 28 of 28; see esmprep functions' hierarchy.
#' # -------------------------------------------------------
#' # tbsqDf is the result of function 'computeTimeBetween'.
#' esDfFin <- esFinal(tbsqDf, RELEVANTINFO_ES, RELEVANTVN_ES)
#' # o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o
#
#' @seealso Exemplary code (fully executable) in the documentation of \code{\link{esmprep}} (function 28 of 28).
#
#' @export
#
esFinal <- function(esDf, RELEVANTINFO_ES=NULL, RELEVANTVN_ES=NULL, maxRows=NULL) {

    if(!is.data.frame(esDf)) {
        stop("Argument 'esDf' must be of class data.frame.")
    }

    variablesUsedIn_esFinal <- c("CV_ES", "CV_ESDAY", "ID",
                                 "PROMPT",
                                 RELEVANTVN_ES[["ES_START_DATETIME"]],
                                 RELEVANTVN_ES[["ES_END_DATETIME"]])

    if(any(is.na(match(variablesUsedIn_esFinal,	 names(esDf))))) {
        stop(paste0("In order to compute the time lag the variable ",
                    variablesUsedIn_esFinal,
                    " must be part of the data.frame that is passed to this function.\n\n"))
    }
    
    
    # Error handling function for all set-up lists generated by setES and setREF.
    # Both lists RELEVANTVN_ES and RELEVANTVN_REF get extended either by function
    # genDateTime or by function splitDateTime!
    SETUPLISTCheck(RELEVANTINFO_ES=RELEVANTINFO_ES,
    			   RELEVANTVN_ES=RELEVANTVN_ES,
    			   RELEVANTVN_REF=NULL)
    

	# If the user doesn't set a maximum number of questionnaires.
    if(is.null(maxRows)) {
    		# Compute the actual maximum number of questionnaires
        maxRows <- max(esDf[,"CV_ES"])
    # If maxRows is not a number or of it is not an integer number.
    } else if(!is.numeric(maxRows) || maxRows%%1 > 0) {
    		stop("The argument 'maxRows' must be an integer number.")
    	# If the number of maxRows is greater than the actual maximum.
    } else if(maxRows > max(esDf[,"CV_ES"])) {
    		stop(paste0("The maximum number of ESM questionnaires set for all participants cannot be greater than the actual maximum number of ESM questionnaires. The actual maximum number is: ", max(esDf[,"CV_ES"]), "."))
    }

    ids <- as.character(unique(esDf[,"ID"]))

    finalizeDf <- data.frame(ID=rep(ids, each = maxRows))
    finalizeDf[,"CV_ES"] <- rep(1:maxRows, times = length(ids))

    # Merge (use default sort = TRUE)
    esFinal <- merge(finalizeDf, esDf, all = TRUE)
    
    	filler <- rep(0, times=nrow(esFinal))
	
	# This for-loop inserts coherent values into the variables CV_ESDAY
	# and PROMPT, which by now are NA (because of inserting) empty rows
	# whenever a questionnaire is either a missing or a filler.
    for(i in 1:length(ids)) {
		
        idx_i <- which(esFinal[,"ID"] == ids[i])
        emptyLines_i <- is.na(esFinal[idx_i, "CV_ESDAY"])
        idx_emptyLines_i <- which(emptyLines_i == TRUE)
        len_emptyLines_i <- length(idx_emptyLines_i)
		
        # If person i has filled out EVERY questionnaire there is no need to
        # fill up anything in the variable 'CV_ESDAY'
        if(len_emptyLines_i == 0) {

            next

            # Task 1: Else fill up any NAs in the variable 'CV_ESDAY' and 'PROMPT'.
        } else {
        	
            for(j in idx_emptyLines_i) {
				
				# If 1 line prior to THIS line j of participant i is NOT the
				# last prompt of the day.
                if(esFinal[ (idx_i [j] - 1) , "PROMPT"] != RELEVANTINFO_ES[["MAXPROMPT"]]) {
					
					# CV_ESDAY must NOT change (remain within same ESM-day)
                    esFinal [ idx_i [j] , "CV_ESDAY"] <-
                        esFinal [ (idx_i [j] - 1) , "CV_ESDAY"]
					
					# Increase variable PROMPT by 1 (all erroneous lines
					# must have been removed by now).
                    esFinal [ idx_i [j] , "PROMPT"] <-
                        esFinal [ (idx_i [j] - 1) , "PROMPT"] + 1
				
				# Else: THIS line j of participant i is NOT the last prompt
				# of the day. Therefore ...
                } else {
					
					# Increase CV_ESDAY by 1 (start a new ESM-day)
                    esFinal [ idx_i [j] , "CV_ESDAY"] <-
                        esFinal [ (idx_i [j] - 1) , "CV_ESDAY"] + 1
					
					# Set variable PROMPT to 1. In the full ESM dataset
					# all prompts for each participant must be present.
                    esFinal [ idx_i [j] , "PROMPT"] <- 1
                }
            }
            
            # Task 2: Register the fillers. A filler is an empty questionnaire
            #		  at the end of all the questionnaires of of one participant
            #		  that fills up questionnaires so that all participants have
            #		  equally many questionnaires.
            
            # Set all lines that have to be filled for participant i to FALSE
           	idxFill_i <- rep(FALSE, times=length(idx_i))
            
            # If the last empty line of data is not the last line of all data
            # for participant i or if there is no empty line of data at all,
            # next participant.
            if(idx_emptyLines_i[len_emptyLines_i] != maxRows | len_emptyLines_i == 0) {
            	next
            
            # If only 1 questionnaire (the last one!) is a filler.
            } else if(length(idx_emptyLines_i) == 1 && idx_emptyLines_i == length(idx_i)) {
            	filler_i <- 1
            	# Insert TRUE in the last line, which is a filler.
            	idxFill_i[length(idx_i)] <- TRUE
            
            # If there are at least 2 fillers and no missing questionnaires.
            } else if(all(abs(diff(rev(idx_emptyLines_i)))==1)) {
            	# filler_i: Number of questionnaires at the end of participant i
            	#			that are fillers.
            	filler_i <- max(which(abs(diff(rev(idx_emptyLines_i))) == 1)) + 1
            	# Insert TRUE in all lines that are fillers.
            	idxFill_i[(length(idx_i)-filler_i):length(idx_i)] <- TRUE
            
            # Else fillers must be registered but missing questionnaires
            # must be ignored, for they are NOT fillers.
            } else {
            	# filler_i: Number of questionnaires at the end of participant i
            	#			that are fillers.
            	filler_i <- which(abs(diff(rev(idx_emptyLines_i))) > 1)[1] - 1
            	# Insert TRUE in all lines that are fillers.
            	idxFill_i[(length(idx_i)-filler_i):length(idx_i)] <- TRUE
            }
            
            # In variable filler change 0 to 1 for participant i.
            filler[idx_i] [idxFill_i] <- 1
        }
    }
    
    # Task 3: Register the missing questionnaires, which must
    #		  be mutually exclusive with the fillers.
    missed <- ifelse(is.na(esFinal[,"ES_MULT"]) & filler == 0, 1, 0)
    
    esFinal[,"MISSED"] <- missed
    esFinal[,"FILLER"] <- filler
    
    cat("!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!\n")
    cat("Don't forget to write the final ESM dataset on your computer's hard disk!\n")
    cat("!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!\n")
    
    return(esFinal)
}
