#' The opinionated "glass box" `eyeris` pipeline
#'
#' This `glassbox` function (in contrast to a "black box" function where you run
#' it and get a result but have no (or little) idea as to how you got from input
#' to output) has a few primary benefits over calling each exported function
#' from `eyeris` separately.
#'
#' First, this `glassbox` function provides a highly opinionated prescription of
#' steps and starting parameters we believe any pupillometry researcher should
#' use as their defaults when preprocessing pupillometry data.
#'
#' Second, and not mutually exclusive from the first point, using this function
#' should ideally reduce the probability of accidental mishaps when
#' "reimplementing" the steps from the preprocessing pipeline both within and
#' across projects. We hope to streamline the process in such a way that you
#' could collect a pupillometry dataset and within a few minutes assess the
#' quality of those data while simultaneously running a full preprocessing
#' pipeline in 1-ish line of code!
#'
#' Third, `glassbox` provides an "interactive" framework where you can evaluate
#' the consequences of the parameters within each step on your data in real
#' time, facilitating a fairly easy-to-use workflow for parameter optimization
#' on your particular dataset. This process essentially takes each of the
#' opinionated steps and provides a pre-/post-plot of the timeseries data for
#' each step so you can adjust parameters and re-run the pipeline until you are
#' satisfied with the choices of your paramters and their consequences on your
#' pupil timeseries data.
#'
#' @param file An SR Research EyeLink `.asc` file generated by the official
#' EyeLink `edf2asc` command.
#' @param interactive_preview A flag to indicate whether to run the `glassbox`
#' pipeline autonomously all the way through (set to `FALSE` by default), or to
#' interactively provide a visualization after each pipeline step, where you
#' must also indicate "(y)es" or "(n)o" to either proceed or cancel the
#' current `glassbox` pipeline operation (set to `TRUE`).
#' @param preview_n Number of random example "epochs" to generate for
#' previewing the effect of each preprocessing step on the pupil timeseries.
#' @param preview_duration Time in seconds of each randomly selected preview.
#' @param preview_window The start and stop raw timestamps used to subset the
#' preprocessed data from each step of the `eyeris` workflow for visualization.
#' Defaults to NULL, meaning random epochs as defined by `preview_n` and
#' `preview_duration` will be plotted. To override the random epochs, set
#' `preview_window` here to a vector with relative start and stop times (in
#' seconds), for example -- `c(5,6)` -- to indicate the raw data from 5-6 secs
#' on data that were recorded at 1000 Hz). Note, the start/stop time values
#' indicated here are in seconds because `eyeris` automatically computes the
#' indices for the supplied range of seconds using the `$info$sample.rate`
#' metadata in the `eyeris` S3 class object.
#' @param verbose A logical flag to indicate whether to print status messages to
#' the console. Defaults to `TRUE`. Set to `FALSE` to suppress messages about
#' the current processing step and run silently.
#' @param ... Additional arguments to override the default, prescribed settings.
#' @param confirm **(Deprecated)** Use `interactive_preview` instead.
#' @param num_previews **(Deprecated)** Use `preview_n` instead.
#' @param detrend_data **(Deprecated)** A flag to indicate whether to run the
#' `detrend` step (set to `FALSE` by default). Detrending your pupil timeseries
#' can have unintended consequences; we thus recommend that users understand the
#' implications of detrending -- in addition to whether detrending is
#' appropriate for the research design and question(s) -- before using this
#' function.
#' @param skip_detransient **(Deprecated)** A flag to indicate whether to skip
#' the `detransient` step (set to `FALSE` by default). In most cases, this
#' should remain `FALSE`. For a more detailed description about likely edge
#' cases that would prompt you to set this to `TRUE`, see the docs for
#' [eyeris::detransient()].
#'
#' @return Preprocessed pupil data contained within an object of class `eyeris`.
#'
#' @seealso [lifecycle::deprecate_warn()]
#'
#' @examples
#' demo_data <- eyelink_asc_demo_dataset()
#'
#' # (1) examples using the default prescribed parameters and pipeline recipe
#'
#' ## (a) run an automated pipeline with no real-time inspection of parameters
#' output <- eyeris::glassbox(demo_data)
#'
#' plot(
#'   output,
#'   steps = c(1, 5),
#'   preview_window = c(0, max(output$timeseries$block_1$time_secs)),
#'   seed = 0
#' )
#'
#' ## (b) run a interactive workflow (with confirmation prompts after each step)
#' \donttest{
#' output <- eyeris::glassbox(demo_data, interactive_preview = TRUE, seed = 0)
#' }
#'
#' # (2) examples of overriding the default parameters
#' output <- eyeris::glassbox(
#'   demo_data,
#'   interactive_preview = FALSE, # TRUE to visualize each step in real-time
#'   deblink = list(extend = 40),
#'   lpfilt = list(plot_freqz = TRUE) # overrides verbose parameter
#' )
#'
#' plot(output, seed = 0)
#'
#' # (3) examples of disabling certain steps
#' output <- eyeris::glassbox(
#'   demo_data,
#'   detransient = FALSE,
#'   detrend = FALSE,
#'   zscore = FALSE
#' )
#'
#' plot(output, seed = 0)
#'
#' @export
glassbox <- function(file,
                     interactive_preview = FALSE,
                     preview_n = 3,
                     preview_duration = 5,
                     preview_window = NULL,
                     verbose = TRUE,
                     ...,
                     confirm = deprecated(),
                     num_previews = deprecated(),
                     detrend_data = deprecated(),
                     skip_detransient = deprecated()) {
  # handle deprecated parameters
  if (is_present(confirm)) {
    deprecate_warn("1.1.0",
                   "glassbox(confirm)",
                   "glassbox(interactive_preview)")
    interactive_preview <- confirm
  }

  if (is_present(num_previews)) {
    deprecate_warn("1.1.0",
                   "glassbox(num_previews)",
                   "glassbox(preview_n)")
    preview_n <- num_previews
  }

  if (is_present(detrend_data)) {
    deprecate_warn(
      "1.1.0",
      "glassbox(detrend_data)",
      details = paste("The `detrend_data` argument is no longer used",
                      "and will be ignored.")
    )

    detrend_data <- NULL
  }

  if (is_present(skip_detransient)) {
    deprecate_warn(
      "1.1.0",
      "glassbox(skip_detransient)",
      details = paste("The `skip_detransient` argument is no longer used",
                      "and will be ignored.")
    )

    skip_detransient <- NULL
  }

  # the default parameters
  default_params <- list(
    load_asc = list(block = "auto"),
    deblink = list(extend = 50),
    detransient = list(n = 16, mad_thresh = NULL),
    interpolate = TRUE,
    lpfilt = list(wp = 4, ws = 8, rp = 1, rs = 35, plot_freqz = verbose),
    detrend = FALSE,
    zscore = TRUE
  )

  # override defaults
  params <- utils::modifyList(default_params, list(...))

  # guard params that accept lists in the event a boolean is supplied
  if ("load_asc" %in% names(list(...)) && isTRUE(list(...)$load_asc)) {
    cli::cli_alert_warning(
      paste(
        "[ WARN ] - `load_asc` expects a list of args (not a boolean)...",
        "using default: `list(block = \"auto\")`"
      )
    )
    params$load_asc <- default_params$load_asc
  }

  if ("deblink" %in% names(list(...)) && isTRUE(list(...)$deblink)) {
    cli::cli_alert_warning(
      paste(
        "[ WARN ] - `deblink` expects a list of args (not a boolean)...",
        "using default: `list(extend = 50)`"
      )
    )
    params$deblink <- default_params$deblink
  }

  if ("detransient" %in% names(list(...)) && isTRUE(list(...)$detransient)) {
    cli::cli_alert_warning(
      paste(
        "[ WARN ] - `detransient` expects a list of args (not a boolean)...",
        "using default: `list(n = 16, mad_thresh = NULL)`"
      )
    )
    params$detransient <- default_params$detransient
  }

  if ("lpfilt" %in% names(list(...)) && isTRUE(list(...)$lpfilt)) {
    cli::cli_alert_warning(paste(
      "[ WARN ] - `lpfilt` expects a list of args (not a boolean)...",
      "using default:",
      "`list(wp = 4, ws = 8, rp = 1, rs = 35, plot_freqz = verbose)`"
    ))
    params$lpfilt <- default_params$lpfilt
  }

  # evaluate which steps of pipeline to run
  which_steps <- evaluate_pipeline_step_params(params)

  # eyeris workflow data structure
  pipeline <- list(
    load_asc = function(data, params) {
      if (which_steps[["load_asc"]]) {
        eyeris::load_asc(data, block = params$load_asc$block)
      } else {
        stop("No data loaded... the glassbox pipeline cannot proceed.")
      }
    },
    deblink = function(data, params) {
      if (which_steps[["deblink"]]) {
        eyeris::deblink(data, extend = params$deblink$extend)
      } else {
        data
      }
    },
    detransient = function(data, params) {
      if (which_steps[["detransient"]]) {
        eyeris::detransient(data, n = params$detransient$n)
      } else {
        data
      }
    },
    interpolate = function(data, params) {
      if (which_steps[["interpolate"]]) {
        eyeris::interpolate(data, verbose = verbose)
      } else {
        data
      }
    },
    lpfilt = function(data, params) {
      if (which_steps[["lpfilt"]]) {
        eyeris::lpfilt(data,
          wp = params$lpfilt$wp,
          ws = params$lpfilt$ws,
          rp = params$lpfilt$rp,
          rs = params$lpfilt$rs,
          plot_freqz = params$lpfilt$plot_freqz
        )
      } else {
        data
      }
    },
    detrend = function(data, params) {
      if (which_steps[["detrend"]]) {
        eyeris::detrend(data)
      } else {
        data
      }
    },
    zscore = function(data, params) {
      if (which_steps[["zscore"]]) {
        eyeris::zscore(data)
      } else {
        data
      }
    }
  )

  seed <- params$seed
  step_counter <- 1
  only_linear_trend <- FALSE
  next_step <- c()

  for (step_name in names(pipeline)) {
    action <- "Running "
    skip_plot <- FALSE

    if (!which_steps[[step_name]]) {
      action <- "Skipping "
      step_counter <- step_counter - 1
      skip_plot <- TRUE
    } else {
      if (step_name == "detrend") {
        only_linear_trend <- TRUE
      }
    }

    if (verbose) {
      if (action == "Running ") {
        cli::cli_alert_success(
          paste0("[  OK  ] - ", action, "eyeris::", step_name, "()")
        )
      } else {
        cli::cli_alert_warning(
          paste0("[ SKIP ] - ", action, "eyeris::", step_name, "()")
        )
      }
    }

    step_to_run <- pipeline[[step_name]]
    err_thrown <- FALSE

    file <- tryCatch(
      {
        step_to_run(file, params)
      },
      error = function(e) {
        if (!which_steps[["interpolate"]] && which_steps[["detrend"]]) {
          cli::cli_alert_danger(
            paste0(
              "[ WARN ] - ", "Because missing pupil samples were not ",
              "interpolated, there is a mismatch in the number of samples ",
              "in the detrended data. Please set `interpolate` to `TRUE` ",
              "before detrending data OR disable detrending by setting ",
              "`detrend` to `FALSE`."
            )
          )
        }

        if (verbose) {
          cli::cli_alert_info(
            paste0(
              "[ INFO ] - ", "Skipping eyeris::", step_name, "(): ",
              e$message
            )
          )
        }
        err_thrown <<- TRUE
        step_counter <<- step_counter - 1
        file
      }
    )

    pupil_steps <- grep("^pupil_",
      colnames(file$timeseries$block_1),
      value = TRUE
    )

    if (interactive_preview) {
      if (!err_thrown) {
        if (!skip_plot) {
          if (step_counter + 1 <= length(names(pipeline))) {
            next_step <- c(next_step, pupil_steps[step_counter])
          } else {
            next_step <- NULL
          }

          for (block_name in names(file$timeseries)) {
            bn <- get_block_numbers(block_name)
            if (is.null(seed)) {
              seed <- rlang::`%||%`(seed, sample.int(.Machine$integer.max, 1))
            }
            withr::with_seed(
              seed,
              {
                plot(
                  file,
                  steps = step_counter,
                  preview_n = preview_n,
                  seed = seed,
                  preview_duration = preview_duration,
                  preview_window = preview_window,
                  only_linear_trend = only_linear_trend,
                  next_step = next_step,
                  block = bn
                )
              }
            )
          }

          if (step_name == "detrend") {
            # reset linear trend flags
            only_linear_trend <- FALSE
          }
        }
        if (step_name != "zscore") {
          if (!prompt_user()) {
            if (verbose) {
              cli::cli_alert_info(
                paste(
                  "Process cancelled after running the", step_name, "step.",
                  "Adjust your parameters and re-run!\n"
                )
              )
            }

            break
          }
        }
      }
    }

    step_counter <- step_counter + 1
  }

  return(file)
}

prompt_user <- function() {
  resp <- readline(prompt = "Continue? [Yes/No]: ")
  tolower(resp) == "yes" | tolower(resp) == "y"
}

evaluate_pipeline_step_params <- function(params) {
  sapply(params, function(x) {
    if (is.logical(x)) {
      isTRUE(x)
    } else {
      !identical(x, FALSE)
    }
  })
}
