% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/compute_annual_trends.R
\name{compute_annual_trends}
\alias{compute_annual_trends}
\title{Calculate prewhitened nonlinear annual trends on streamflow data}
\usage{
compute_annual_trends(
  data,
  dates = Date,
  values = Value,
  groups = STATION_NUMBER,
  station_number,
  zyp_method,
  basin_area,
  water_year_start = 1,
  start_year,
  end_year,
  exclude_years,
  annual_percentiles = c(10, 90),
  monthly_percentiles = c(10, 20),
  stats_days = 1,
  stats_align = "right",
  lowflow_days = c(1, 3, 7, 30),
  lowflow_align = "right",
  timing_percent = c(25, 33, 50, 75),
  normal_percentiles = c(25, 75),
  ignore_missing = FALSE,
  include_plots = TRUE,
  zyp_alpha
)
}
\arguments{
\item{data}{Data frame of daily data that contains columns of dates, flow values, and (optional) groups (e.g. station numbers).
Leave blank if using \code{station_number} argument.}

\item{dates}{Name of column in \code{data} that contains dates formatted YYYY-MM-DD. Only required if dates column name is not 
'Date' (default). Leave blank if using \code{station_number} argument.}

\item{values}{Name of column in \code{data} that contains numeric flow values, in units of cubic metres per second.
Only required if values column name is not 'Value' (default). Leave blank if using \code{station_number} argument.}

\item{groups}{Name of column in \code{data} that contains unique identifiers for different data sets, if applicable. Only required 
if groups column name is not 'STATION_NUMBER'. Function will automatically group by a column named 'STATION_NUMBER' if 
present. Remove the 'STATION_NUMBER' column beforehand to remove this grouping. Leave blank if using \code{station_number} 
argument.}

\item{station_number}{Character string vector of seven digit Water Survey of Canada station numbers (e.g. \code{"08NM116"}) of
which to extract daily streamflow data from a HYDAT database. Requires \code{tidyhydat} package and a HYDAT database.
Leave blank if using \code{data} argument.}

\item{zyp_method}{Character string identifying the prewhitened trend method to use from \code{zyp}, either \code{'zhang'}
or \code{'yuepilon'}. Required.}

\item{basin_area}{Upstream drainage basin area, in square kilometres, to apply to observations. Three options:

(1) Leave blank if \code{groups} is STATION_NUMBER with HYDAT station numbers to extract basin areas from HYDAT.

(2) A single numeric value to apply to all observations.

(3) List each basin area for each group/station in groups (can override HYDAT value if listed) as such \code{c("08NM116" = 795, 
"08NM242" = 10)}. If group is not listed the HYDAT area will be applied if it exists, otherwise it will be \code{NA}.}

\item{water_year_start}{Numeric value indicating the month (\code{1} through \code{12}) of the start of water year for
analysis. Default \code{1}.}

\item{start_year}{Numeric value of the first year to consider for analysis. Leave blank to use the first year of the source data.}

\item{end_year}{Numeric value of the last year to consider for analysis. Leave blank to use the last year of the source data.}

\item{exclude_years}{Numeric vector of years to exclude from analysis. Leave blank to include all years.}

\item{annual_percentiles}{Numeric vector of percentiles to calculate annually. Set to \code{NA} if none required. Used for
\code{calc_annual_stats()} function. Default \code{c(10,90)}.}

\item{monthly_percentiles}{Numeric vector of percentiles to calculate monthly for each year. Set to \code{NA} if none required. 
Used for \code{calc_monthly_stats()} function. Default \code{c(10,20)}.}

\item{stats_days}{Numeric vector of the number of days to apply a rolling mean on basic stats. Default \code{c(1)}.
Used for \code{calc_annual_stats()} and \code{calc_monthly_stats()} functions.}

\item{stats_align}{Character string identifying the direction of the rolling mean on basic stats from the specified date, either by 
the first (\code{'left'}), last (\code{'right'}), or middle (\code{'center'}) day of the rolling n-day group of observations.
Default \code{'right'}. Used for \code{calc_annual_stats()}, \code{calc_monthly_stats()}, and \code{calc_annual_outside_normal()}
functions.}

\item{lowflow_days}{Numeric vector of the number of days to apply a rolling mean on low flow stats. Default \code{c(1,3,7,30)}.
Used for \code{calc_lowflow_stats()} function.}

\item{lowflow_align}{Character string identifying the direction of the rolling mean on low flow stats from the specified date,
either by the first (\code{'left'}), last (\code{'right'}), or middle (\code{'center'}) day of the rolling n-day group of 
observations. Default \code{'right'}. Used for \code{calc_lowflow_stats()} function.}

\item{timing_percent}{Numeric vector of percents of annual total flows to determine dates. Used for \code{calc_annual_flow_timing()}
function. Default \code{c(25,33.3,50,75)}.}

\item{normal_percentiles}{Numeric vector of two values, lower and upper percentiles, respectively indicating the limits of the 
normal range. Default \code{c(25,75)}.}

\item{ignore_missing}{Logical value indicating whether dates with missing values should be included in the calculation. If
\code{TRUE} then a statistic will be calculated regardless of missing dates. If \code{FALSE} then only those statistics from
time periods with no missing dates will be returned. Default \code{FALSE}.}

\item{include_plots}{Logical value indicating if annual trending plots should be included. Default \code{TRUE}.}

\item{zyp_alpha}{Numeric value of the significance level (ex. \code{0.05}) of when to plot a trend line. Leave blank for no line.}
}
\value{
A list of tibbles and optional plots from the trending analysis including:
  \item{Annual_Trends_Data}{a tibble of the annual statistics used for trending}
  \item{Annual_Trends_Results}{a tibble of the results of the zyp trending analysis}
  \item{Annual_*}{each ggplot2 object for each annual trended statistic}
}
\description{
Calculates prewhitened nonlinear trends on annual streamflow data. Uses the
   \href{https://CRAN.R-project.org/package=zyp}{\code{zyp}} package to calculate trends. Review \code{zyp} for more information
   Calculates statistics from all values, unless specified. Returns a list of tibbles and plots.
   All annual statistics calculated using the \code{calc_all_annual_stats()} function which uses the following 
   \code{fasstr} functions:
\itemize{
 \item{\code{calc_annual_stats()}}
 \item{\code{calc_annual_lowflows()}}
 \item{\code{calc_annual_cumulative_stats()}}
 \item{\code{calc_annual_flow_timing()}}
 \item{\code{calc_monthly_stats()}}
 \item{\code{calc_annual_outside_normal()}}
 }
}
\examples{
\dontrun{

# Working examples:

# Compute trends statistics using a data frame and data argument with defaults
flow_data <- tidyhydat::hy_daily_flows(station_number = "08NM116")
trends <- compute_annual_trends(data = flow_data,
                                zyp_method = "yuepilon")

# Compute trends statistics using station_number with defaults
trends <- compute_annual_trends(station_number = "08NM116",
                                zyp_method = "yuepilon")
                      
# Compute trends statistics and plot a trend line if the significance is less than 0.05
trends <- compute_annual_trends(station_number = "08NM116",
                                zyp_method = "yuepilon",
                                zyp_alpha = 0.05)
                      
# Compute trends statistics and do not plot the results
trends <- compute_annual_trends(station_number = "08NM116",
                                zyp_method = "yuepilon",
                                include_plots = FALSE)

}                
}
\references{
References from zyp package:
\itemize{
 \item{Wang, X.L. and Swail, V.R., 2001. Changes in extreme wave heights in northern hemisphere oceans and 
       related atmospheric circulation regimes. Journal of Climate, 14: 2204-2221.}
 \item{Yue, S., P. Pilon, B. Phinney and G. Cavadias, 2002. The influence of autocorrelation on the ability
       to detect trend in hydrological series. Hydrological Processes, 16: 1807-1829.}
 \item{Zhang, X., Vincent, L.A., Hogg, W.D. and Niitsoo, A., 2000. Temperature and Precipitation Trends in
       Canada during the 20th Century. Atmosphere-Ocean 38(3): 395-429.}
 \item{Sen, P.K., 1968. Estimates of the Regression Coefficient Based on Kendall's Tau. Journal of the 
       American Statistical Association Vol. 63, No. 324: 1379-1389.}
       }
}
\seealso{
\code{\link[zyp]{zyp-package}}, 
         \code{\link{calc_all_annual_stats}}
}
