% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{qrdowndate}
\alias{qrdowndate}
\title{Fast downdating of the QR factorization}
\usage{
qrdowndate(Q, R, k, m = NULL, type = NULL, fast = NULL, complete = NULL)
}
\arguments{
\item{Q}{a \eqn{n\times n} matrix.}

\item{R}{a \eqn{n\times p} upper triangular matrix.}

\item{k}{position where the columns or the rows are removed.}

\item{m}{number of columns or rows to be removed. Default is \eqn{m=1}.}

\item{type}{either 'row' of 'column', for deleting rows or columns. Default is 'column'.}

\item{fast}{fast mode: disable to check whether the provided matrices are valid inputs. Default is FALSE.}

\item{complete}{logical expression of length 1. Indicates whether an arbitrary orthogonal completion of the \eqn{Q} matrix is to be made, or whether the \eqn{R} matrix is to be completed by binding zero-value rows beneath the square upper triangle.}
}
\value{
A named list containing \describe{
\item{Q}{the updated Q matrix.}
\item{R}{the updated R matrix.}
}
}
\description{
qrdowndate provides the update of the QR factorization after the deletion of \eqn{m>1} rows or columns to the matrix \eqn{X\in\mathbb{R}^{n\times p}} with \eqn{n>p}. The QR factorization of the matrix \eqn{X\in\mathbb{R}^{n\times p}} returns the matrices \eqn{Q\in\mathbb{R}^{n\times n}} and \eqn{R\in\mathbb{R}^{n\times p}} such that \eqn{X=QR}. The \eqn{Q} and \eqn{R} matrices are factorized as \eqn{Q=\begin{bmatrix}Q_1&Q_2\end{bmatrix}} and \eqn{R=\begin{bmatrix}R_1\\R_2\end{bmatrix}}, with \eqn{Q_1\in\mathbb{R}^{n\times p}}, \eqn{Q_2\in\mathbb{R}^{n\times (n-p)}} such that \eqn{Q_1^{\top}Q_2=Q_2^\top Q_1=0} and \eqn{R_1\in\mathbb{R}^{p\times p}} upper triangular matrix and \eqn{R_2\in\mathbb{R}^{(n-p)\times p}}. qrupdate accepts in input the matrices \eqn{Q} and either the complete matrix \eqn{R} or the reduced one, \eqn{R_1}. See Golub and Van Loan (2013) for further details on the method.
}
\examples{
## Remove one column
## generate sample data
set.seed(10)
n      <- 10
p      <- 6
X      <- matrix(rnorm(n * p, 1), n, p)

## get the initial QR factorization
output <- fastQR::qr(X, type = "householder",
                     nb = NULL,
                     complete = TRUE)
Q      <- output$Q
R      <- output$R

## select the column to be deleted
## from X and update X
k  <- 2
X1 <- X[, -k]

## downdate the QR decomposition
out <- fastQR::qrdowndate(Q = Q, R = R,
                          k = k, m = 1,
                          type = "column",
                          fast = FALSE,
                          complete = TRUE)

## check
round(out$Q \%*\% out$R - X1, 5)
max(abs(out$Q \%*\% out$R - X1))

## Remove m columns
## generate sample data
set.seed(10)
n      <- 10
p      <- 6
X      <- matrix(rnorm(n * p, 1), n, p)

## get the initial QR factorization
output <- fastQR::qr(X, type = "householder",
                     nb = NULL,
                     complete = TRUE)
Q      <- output$Q
R      <- output$R

## select the column to be deleted from X
## and update X
m  <- 2
k  <- 2
X1 <- X[, -c(k,k+m-1)]

## downdate the QR decomposition
out <- fastQR::qrdowndate(Q = Q, R = R,
                          k = k, m = 2,
                          type = "column",
                          fast = TRUE,
                          complete = FALSE)

## check
round(out$Q \%*\% out$R - X1, 5)
max(abs(out$Q \%*\% out$R - X1))

## Remove one row
## generate sample data
set.seed(10)
n      <- 10
p      <- 6
X      <- matrix(rnorm(n * p, 1), n, p)

## get the initial QR factorization
output <- fastQR::qr(X, type = "householder",
                     nb = NULL,
                     complete = TRUE)
Q      <- output$Q
R      <- output$R

## select the row to be deleted from X and update X
k  <- 5
X1 <- X[-k,]

## downdate the QR decomposition
out <- fastQR::qrdowndate(Q = Q, R = R,
                          k = k, m = 1,
                          type = "row",
                          fast = FALSE,
                          complete = TRUE)

## check
round(out$Q \%*\% out$R - X1, 5)
max(abs(out$Q \%*\% out$R - X1))

## Remove m rows
## generate sample data
set.seed(10)
n      <- 10
p      <- 6
X      <- matrix(rnorm(n * p, 1), n, p)

## get the initial QR factorization
output <- fastQR::qr(X, type = "householder",
                     nb = NULL,
                     complete = TRUE)
Q      <- output$Q
R      <- output$R

## select the rows to be deleted from X and update X
k  <- 5
m  <- 2
X1 <- X[-c(k,k+1),]

## downdate the QR decomposition
out <- fastQR::qrdowndate(Q = Q, R = R,
                          k = k, m = m,
                          type = "row",
                          fast = FALSE,
                          complete = TRUE)

## check
round(out$Q \%*\% out$R - X1, 5)
max(abs(out$Q \%*\% out$R - X1))

}
\references{
\insertRef{golub_van_loan.2013}{fastQR}

\insertRef{bjorck.2015}{fastQR}

\insertRef{bjorck.2024}{fastQR}

\insertRef{bernardi_etal.2024}{fastQR}
}
