% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/directed_dcsbm.R
\name{directed_dcsbm}
\alias{directed_dcsbm}
\title{Create a directed degree corrected stochastic blockmodel object}
\usage{
directed_dcsbm(
  n = NULL,
  theta_in = NULL,
  theta_out = NULL,
  k_in = NULL,
  k_out = NULL,
  B = NULL,
  ...,
  pi_in = rep(1/k_in, k_in),
  pi_out = rep(1/k_out, k_out),
  sort_nodes = TRUE,
  force_identifiability = TRUE,
  poisson_edges = TRUE,
  allow_self_loops = TRUE
)
}
\arguments{
\item{n}{(degree heterogeneity) The number of nodes in the blockmodel.
Use when you don't want to specify the degree-heterogeneity
parameters \code{theta_in} and \code{theta_out} by hand. When \code{n} is specified,
\code{theta_in} and \code{theta_out} are randomly generated from
a \code{LogNormal(2, 1)} distribution.
This is subject to change, and may not be reproducible.
\code{n} defaults to \code{NULL}. You must specify either \code{n}
or \code{theta_in} and \code{theta_out} together, but not both.}

\item{theta_in}{(degree heterogeneity) A numeric vector
explicitly specifying the degree heterogeneity
parameters. This implicitly determines the number of nodes
in the resulting graph, i.e. it will have \code{length(theta_in)} nodes.
Must be positive. Setting to a vector of ones recovers
a stochastic blockmodel without degree correction.
Defaults to \code{NULL}. You must specify either \code{n}
or \code{theta_in} and \code{theta_out} together, but not both.}

\item{theta_out}{(degree heterogeneity) A numeric vector
explicitly specifying the degree heterogeneity
parameters. This implicitly determines the number of nodes
in the resulting graph, i.e. it will have \code{length(theta)} nodes.
Must be positive. Setting to a vector of ones recovers
a stochastic blockmodel without degree correction.
Defaults to \code{NULL}. You must specify either \code{n}
or \code{theta_in} and \code{theta_out} together, but not both.}

\item{k_in}{(mixing matrix) The number of blocks in the blockmodel.
Use when you don't want to specify the mixing-matrix by hand.
When \code{k_in} is specified, the elements of \code{B} are drawn
randomly from a \code{Uniform(0, 1)} distribution.
This is subject to change, and may not be reproducible.
\code{k_in} defaults to \code{NULL}. You must specify either \code{k_in} and
\code{k_out} together, or \code{B}. You may specify all three at once, in which
case \code{k_in} is only used to set \code{pi_in} (when \code{pi_in} is
left at its default argument value).}

\item{k_out}{(mixing matrix) The number of blocks in the blockmodel.
Use when you don't want to specify the mixing-matrix by hand.
When \code{k_out} is specified, the elements of \code{B} are drawn
randomly from a \code{Uniform(0, 1)} distribution.
This is subject to change, and may not be reproducible.
\code{k_out} defaults to \code{NULL}. You may specify all three at once, in which
case \code{k_out} is only used to set \code{pi_out} (when \code{pi_out} is
left at its default argument value).}

\item{B}{(mixing matrix) A \code{k_in} by \code{k_out} matrix of block connection
probabilities. The probability that a node in block \code{i} connects
to a node in community \code{j} is \code{Poisson(B[i, j])}.
\code{matrix} and \code{Matrix} objects are both acceptable.
Defaults to \code{NULL}. You must specify either \code{k_in} and
\code{k_out} together, or \code{B}, but not both.}

\item{...}{
  Arguments passed on to \code{\link[=directed_factor_model]{directed_factor_model}}
  \describe{
    \item{\code{expected_in_degree}}{If specified, the desired expected in degree
of the graph. Specifying \code{expected_in_degree} simply rescales \code{S}
to achieve this. Defaults to \code{NULL}. Specify only one of
\code{expected_in_degree}, \code{expected_out_degree}, and \code{expected_density}.}
    \item{\code{expected_out_degree}}{If specified, the desired expected out degree
of the graph. Specifying \code{expected_out_degree} simply rescales \code{S}
to achieve this. Defaults to \code{NULL}. Specify only one of
\code{expected_in_degree}, \code{expected_out_degree}, and \code{expected_density}.}
    \item{\code{expected_density}}{If specified, the desired expected density
of the graph. Specifying \code{expected_density} simply rescales \code{S}
to achieve this. Defaults to \code{NULL}. Specify only one of
\code{expected_in_degree}, \code{expected_out_degree}, and \code{expected_density}.}
  }}

\item{pi_in}{(relative block probabilities) Relative block
probabilities. Must be positive, but do not need to sum
to one, as they will be normalized internally.
Must match the rows of \code{B}, or \code{k_in}. Defaults to
\code{rep(1 / k_in, k_in)}, or a balanced incoming blocks.}

\item{pi_out}{(relative block probabilities) Relative block
probabilities. Must be positive, but do not need to sum
to one, as they will be normalized internally.
Must match the columns of \code{B}, or \code{k_out}. Defaults to
\code{rep(1 / k_out, k_out)}, or a balanced outgoing blocks.}

\item{sort_nodes}{Logical indicating whether or not to sort the nodes
so that they are grouped by block. Useful for plotting.
Defaults to \code{TRUE}.}

\item{force_identifiability}{Logical indicating whether or not to
normalize \code{theta_in} such that it sums to one within each incoming
block and \code{theta_out} such that it sums to one within each outgoing
block. Defaults to \code{TRUE}.}

\item{poisson_edges}{Logical indicating whether or not
multiple edges are allowed to form between a pair of
nodes. Defaults to \code{TRUE}. When \code{FALSE}, sampling proceeds
as usual, and duplicate edges are removed afterwards. Further,
when \code{FALSE}, we assume that \code{S} specifies a desired between-factor
connection probability, and back-transform this \code{S} to the
appropriate Poisson intensity parameter to approximate Bernoulli
factor connection probabilities. See Section 2.3 of Rohe et al. (2017)
for some additional details.}

\item{allow_self_loops}{Logical indicating whether or not
nodes should be allowed to form edges with themselves.
Defaults to \code{TRUE}. When \code{FALSE}, sampling proceeds allowing
self-loops, and these are then removed after the fact.}
}
\value{
A \code{directed_dcsbm} S3 object, a subclass of the
\code{\link[=directed_factor_model]{directed_factor_model()}} with the following additional
fields:
\itemize{
\item \code{theta_in}: A numeric vector of incoming community
degree-heterogeneity parameters.
\item \code{theta_out}: A numeric vector of outgoing community
degree-heterogeneity parameters.
\item \code{z_in}: The incoming community memberships of each node,
as a \code{\link[=factor]{factor()}}. The factor will have \code{k_in} levels,
where \code{k_in} is the number of incoming
communities in the stochastic blockmodel. There will not
always necessarily be observed nodes in each community.
\item \code{z_out}: The outgoing community memberships of each node,
as a \code{\link[=factor]{factor()}}. The factor will have \code{k_out} levels,
where \code{k_out} is the number of outgoing
communities in the stochastic blockmodel. There will not
always necessarily be observed nodes in each community.
\item \code{pi_in}: Sampling probabilities for each incoming
community.
\item \code{pi_out}: Sampling probabilities for each outgoing
community.
\item \code{sorted}: Logical indicating where nodes are arranged by
block (and additionally by degree heterogeneity parameter)
within each block.
}
}
\description{
To specify a degree-corrected stochastic blockmodel, you must specify
the degree-heterogeneity parameters (via \code{n_in} or \code{theta_in}, and
\code{n_out} or \code{theta_out}), the mixing matrix
(via \code{k_in} and \code{k_out}, or \code{B}), and the relative block
probabilities (optional, via \code{p_in} and \code{pi_out}).
We provide defaults for most of these
options to enable rapid exploration, or you can invest the effort
for more control over the model parameters. We \strong{strongly recommend}
setting the \code{expected_in_degree}, \code{expected_out_degree},
or \code{expected_density} argument
to avoid large memory allocations associated with
sampling large, dense graphs.
}
\section{Generative Model}{
There are two levels of randomness in a directed degree-corrected
stochastic blockmodel. First, we randomly chose a incoming
block membership and an outgoing block membership
for each node in the blockmodel. This is
handled by \code{directed_dcsbm()}. Then, given these block memberships,
we randomly sample edges between nodes. This second
operation is handled by \code{\link[=sample_edgelist]{sample_edgelist()}},
\code{\link[=sample_sparse]{sample_sparse()}}, \code{\link[=sample_igraph]{sample_igraph()}} and
\code{\link[=sample_tidygraph]{sample_tidygraph()}}, depending on your desired
graph representation.
\subsection{Block memberships}{

Let \eqn{x} represent the incoming block membership of a node
and \eqn{y} represent the outgoing block membership of a node.
To generate \eqn{x} we sample from a categorical
distribution with parameter \eqn{\pi_in}.
To generate \eqn{y} we sample from a categorical
distribution with parameter \eqn{\pi_out}.
Block memberships are independent across nodes. Incoming and outgoing
block memberships of the same node are also independent.
}

\subsection{Degree heterogeneity}{

In addition to block membership, the DCSBM also
nodes to have different propensities for incoming and
outgoing edge formation.
We represent the propensity to form incoming edges for a
given node by a positive number \eqn{\theta_in}.
We represent the propensity to form outgoing edges for a
given node by a positive number \eqn{\theta_out}.
Typically the \eqn{\theta_in} (and \eqn{theta_out}) across all nodes are
constrained to sum to one for identifiability purposes,
but this doesn't really matter during sampling.
}

\subsection{Edge formulation}{

Once we know the block memberships \eqn{x} and \eqn{y}
and the degree  heterogeneity parameters \eqn{\theta_{in}} and
\eqn{\theta_{out}}, we need one more
ingredient, which is the baseline intensity of connections
between nodes in block \code{i} and block \code{j}. Then each edge forms
independently according to a Poisson distribution with
parameters

\deqn{
  \lambda = \theta_{in} * B_{x, y} * \theta_{out}.
}
}
}

\examples{

set.seed(27)

B <- matrix(0.2, nrow = 5, ncol = 8)
diag(B) <- 0.9

ddcsbm <- directed_dcsbm(
  n = 1000,
  B = B,
  k_in = 5,
  k_out = 8,
  expected_density = 0.01
)

ddcsbm

population_svd <- svds(ddcsbm)

}
\seealso{
Other stochastic block models: 
\code{\link{dcsbm}()},
\code{\link{mmsbm}()},
\code{\link{overlapping_sbm}()},
\code{\link{planted_partition}()},
\code{\link{sbm}()}

Other directed graphs: 
\code{\link{directed_erdos_renyi}()}
}
\concept{directed graphs}
\concept{stochastic block models}
