% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/functional_autocorrelation.R
\name{obtain_autocorrelation}
\alias{obtain_autocorrelation}
\title{Estimate the autocorrelation function of the series}
\usage{
obtain_autocorrelation(
  Y,
  v = seq(from = 0, to = 1, length.out = ncol(Y)),
  nlags
)
}
\arguments{
\item{Y}{Matrix containing the discretized values
of the functional time series. The dimension of the
matrix is \eqn{(n x m)}, where \eqn{n} is the
number of curves and \eqn{m} is the number of points
observed in each curve.}

\item{v}{Discretization points of the curves, by default
\code{seq(from = 0, to = 1, length.out = 100)}.}

\item{nlags}{Number of lagged covariance operators
of the functional time series that will be used
to estimate the autocorrelation function.}
}
\value{
Return a list with the lagged autocorrelation
functions estimated from the data. Each function is given
by a \eqn{(m x m)} matrix, where \eqn{m} is the
number of points observed in each curve.
}
\description{
Obtain the empirical autocorrelation function for
lags \eqn{= 0,...,}\code{nlags} of the functional time
series. Given \eqn{Y_{1},...,Y_{T}} a functional time
series, the sample autocovariance functions
\eqn{\hat{C}_{h}(u,v)} are given by:
\deqn{\hat{C}_{h}(u,v) =  \frac{1}{T} \sum_{i=1}^{T-h}(Y_{i}(u) - \overline{T}_{T}(u))(Y_{i+h}(v) - \overline{Y}_{T}(v))}
where
\eqn{ \overline{Y}_{T}(u) = \frac{1}{T} \sum_{i = 1}^{T} Y_{i}(t)}
denotes the sample mean function. By normalizing these
functions using the normalizing factor
\eqn{\int\hat{C}_{0}(u,u)du}, the range of the
autocovariance functions becomes \eqn{(0,1)}; thus
defining the autocorrelation functions of the series
}
\examples{
# Example 1

N <- 100
v <- seq(from = 0, to = 1, length.out = 10)
sig <- 2
bbridge <- simulate_iid_brownian_bridge(N, v, sig)
nlags <- 1
lagged_autocor <- obtain_autocorrelation(Y = bbridge,
                                        nlags = nlags)
image(x = v, y = v, z = lagged_autocor$Lag0)

\donttest{
# Example 2
require(fields)
N <- 500
v <- seq(from = 0, to = 1, length.out = 50)
sig <- 2
bbridge <- simulate_iid_brownian_bridge(N, v, sig)
nlags <- 4
lagged_autocov <- obtain_autocovariance(Y = bbridge,
                                        nlags = nlags)
lagged_autocor <- obtain_autocorrelation(Y = bbridge,
                                         v = v,
                                         nlags = nlags)

opar <- par(no.readonly = TRUE)
par(mfrow = c(1,2))
z_lims <- range(lagged_autocov$Lag1)
colors <- heat.colors(12)
image.plot(x = v, 
           y = v,
           z = lagged_autocov$Lag1,
           legend.width = 2,
           zlim = z_lims,
           col = colors,
           xlab = "u",
           ylab = "v",
           main = "Autocovariance")
z_lims <- range(lagged_autocor$Lag1)
image.plot(x = v, 
           y = v,
           z = lagged_autocor$Lag1,
           legend.width = 2,
           zlim = z_lims,
           col = colors,
           xlab = "u",
           ylab = "v",
           main = "Autocorrelation")
par(opar)
}
}
