#' MFL players library
#'
#' A cached table of MFL players. Will store in memory for each session!
#' (via memoise in zzz.R)
#'
#' @param conn optionally, pass in a conn object generated by ff_connect to receive league-specific custom players
#'
#' @examples
#' \donttest{
#' player_list <- mfl_players()
#' dplyr::sample_n(player_list, 5)
#' }
#'
#' @return a dataframe containing all ~2000+ players in the MFL database
#'
#' @export

mfl_players <- function(conn = NULL) {
  if (!is.null(conn) && class(conn) != "mfl_conn") {
    stop("conn must be generated by 'mfl_connect()' and have type 'mfl_conn'")
  }

  if (is.null(conn)) {
    conn <- mfl_connect(.fn_choose_season())
  }

  conn %>%
    mfl_getendpoint("players", DETAILS = 1) %>%
    purrr::pluck("content", "players", "player") %>%
    tibble::tibble() %>%
    tidyr::unnest_wider(1) %>%
    dplyr::mutate_at("birthdate", ~ as.numeric(.x) %>%
      lubridate::as_datetime() %>%
      lubridate::as_date()) %>%
    dplyr::mutate("age" = round(as.numeric(Sys.Date() - .data$birthdate) / 365.25, 1)) %>%
    dplyr::select(
      "player_id" = .data$id,
      "player_name" = .data$name,
      "pos" = .data$position,
      .data$age,
      .data$team,
      .data$status,
      dplyr::starts_with("draft_"),
      dplyr::ends_with("_id"),
      dplyr::everything()
    )
}
