% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/btd_foot.R
\name{btd_foot}
\alias{btd_foot}
\title{Bayesian Bradley-Terry-Davidson Model}
\usage{
btd_foot(
  data,
  dynamic_rank = FALSE,
  home_effect = FALSE,
  prior_par = list(logStrength = normal(0, 3), logTie = normal(0, 0.3), home = normal(0,
    5)),
  rank_measure = "median",
  ...
)
}
\arguments{
\item{data}{A data frame containing the observations with columns:
\itemize{
  \item \code{periods}: Time point of each observation (integer >= 1).
  \item \code{home_team}: Home team's name (character string).
  \item \code{away_team}: Away team's name (character string).
  \item \code{match_outcome}: Outcome (1 if home team beats away team, 2 for tie, and 3 if away team beats home team).
}
The data frame must not contain missing values.}

\item{dynamic_rank}{A logical value indicating whether a dynamic ranking model is used (default is \code{FALSE}).}

\item{home_effect}{A logical value indicating the inclusion of a home effect in the model. (default is \code{FALSE}).}

\item{prior_par}{A list specifying the prior distributions for the parameters of interest, using the \code{normal} function:
\itemize{
  \item \code{logStrength}: Prior for the team log-strengths. Default is \code{normal(0, 3)}.
  \item \code{logTie}: Prior for the tie parameter. Default is \code{normal(0, 0.3)}.
  \item \code{home}: Prior for the home effect (\code{home}). Applicable only if \code{home_effect = TRUE}. Default is \code{normal(0, 5)}.
}
Only normal priors are allowed for this model.}

\item{rank_measure}{A character string specifying the method used to summarize the posterior distributions of the team strengths. Options are:
\itemize{
  \item \code{"median"}: Uses the median of the posterior samples (default).
  \item \code{"mean"}: Uses the mean of the posterior samples.
  \item \code{"map"}: Uses the Maximum A Posteriori estimate, calculated as the mode of the posterior distribution.
}}

\item{...}{Additional arguments passed to \code{\link[rstan]{stan}} (e.g., \code{iter}, \code{chains}, \code{control}).}
}
\value{
A list of class \code{"btdFoot"} containing:
  \itemize{
    \item \code{fit}: The fitted \code{stanfit} object returned by \code{\link[rstan]{stan}}.
    \item \code{rank}: A data frame with the rankings, including columns:
      \itemize{
        \item \code{periods}: The time period.
        \item \code{team}: The team name.
        \item \code{rank_points}: The estimated strength of the team based on the chosen \code{rank_measure}.
      }
    \item \code{data}: The input data.
    \item \code{stan_data}: The data list prepared for Stan.
    \item \code{stan_code}: The path to the Stan model code used.
    \item \code{stan_args}: The optional parameters passed to (\code{...}).
    \item \code{rank_measure}: The method used to compute the rankings.
  }
}
\description{
Fits a Bayesian Bradley-Terry-Davidson model using Stan. Supports both static and dynamic ranking models, allowing for the estimation of team strengths over time.
}
\examples{
\dontrun{

library(dplyr)

data("italy")

italy_2020_2021 <- italy \%>\%
  dplyr::select(Season, home, visitor, hgoal, vgoal) \%>\%
  dplyr::filter(Season == "2020" | Season == "2021") \%>\%
  dplyr::mutate(match_outcome = dplyr::case_when(
    hgoal > vgoal ~ 1,        # Home team wins
    hgoal == vgoal ~ 2,       # Draw
    hgoal < vgoal ~ 3         # Away team wins
  )) \%>\%
  dplyr::mutate(periods = dplyr::case_when(
    dplyr::row_number() <= 190 ~ 1,
    dplyr::row_number() <= 380 ~ 2,
    dplyr::row_number() <= 570 ~ 3,
    TRUE ~ 4
  )) \%>\%  # Assign periods based on match number
  dplyr::select(periods, home_team = home,
   away_team = visitor, match_outcome)

# Dynamic Ranking Example with Median Rank Measure
fit_result_dyn <- btd_foot(
  data = italy_2020_2021,
  dynamic_rank = TRUE,
  home_effect = TRUE,
  prior_par = list(
    logStrength = normal(0, 10),
    logTie = normal(0, 5),
    home = normal(0, 5)
  ),
  rank_measure = "median",
  iter = 1000,
  cores = 2,
  chains = 2
)

print(fit_result_dyn)

print(fit_result_dyn, pars = c("logStrength", "home"), teams = c("AC Milan", "AS Roma"))

# Static Ranking Example with MAP Rank Measure
fit_result_stat <- btd_foot(
  data = italy_2020_2021,
  dynamic_rank = FALSE,
  prior_par = list(
    logStrength = normal(0, 10),
    logTie = normal(0, 5),
    home = normal(0, 5)
  ),
  rank_measure = "map",
  iter = 1000,
  chains = 2
)

print(fit_result_stat)
}
}
\author{
Roberto Macrì Demartino \email{roberto.macridemartino@phd.unipd.it}.
}
