% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{t_running_sum}
\alias{t_running_sum}
\alias{t_running_mean}
\title{Compute sums or means over a sliding time window.}
\usage{
t_running_sum(v, time = NULL, time_deltas = NULL, window = NULL,
  wts = NULL, lb_time = NULL, na_rm = FALSE, min_df = 0L,
  restart_period = 10000L, variable_win = FALSE, wts_as_delta = TRUE,
  check_wts = FALSE)

t_running_mean(v, time = NULL, time_deltas = NULL, window = NULL,
  wts = NULL, lb_time = NULL, na_rm = FALSE, min_df = 0L,
  restart_period = 10000L, variable_win = FALSE, wts_as_delta = TRUE,
  check_wts = FALSE)
}
\arguments{
\item{v}{a vector.}

\item{time}{an optional vector of the timestamps of \code{v}. If given, must be
the same length as \code{v}. If not given, we try to infer it by summing the
\code{time_deltas}.}

\item{time_deltas}{an optional vector of the deltas of timestamps. If given, must be
the same length as \code{v}. If not given, and \code{wts} are given and \code{wts_as_delta} is true,
we take the \code{wts} as the time deltas.  The deltas must be positive. We sum them to arrive
at the times.}

\item{window}{the window size, in time units. if given as finite integer or double, passed through.
If \code{NULL}, \code{NA_integer_}, \code{NA_real_} or \code{Inf} are given, 
 and \code{variable_win} is true, then we infer the window from the lookback times: the
 first window is infinite, but the remaining is the deltas between lookback times.
 If \code{variable_win} is false, then these undefined values are equivalent to an
 infinite window.
 If negative, an error will be thrown.}

\item{wts}{an optional vector of weights. Weights are \sQuote{replication}
weights, meaning a value of 2 is shorthand for having two observations
with the corresponding \code{v} value. If \code{NULL}, corresponds to
equal unit weights, the default. Note that weights are typically only meaningfully defined
up to a multiplicative constant, meaning the units of weights are
immaterial, with the exception that methods which check for minimum df will,
in the weighted case, check against the sum of weights. For this reason,
weights less than 1 could cause \code{NA} to be returned unexpectedly due
to the minimum condition. When weights are \code{NA}, the same rules for checking \code{v}
are applied. That is, the observation will not contribute to the moment
if the weight is \code{NA} when \code{na_rm} is true. When there is no
checking, an \code{NA} value will cause the output to be \code{NA}.}

\item{lb_time}{a vector of the times from which lookback will be performed. The output should
be the same size as this vector. If not given, defaults to \code{time}.}

\item{na_rm}{whether to remove NA, false by default.}

\item{min_df}{the minimum df to return a value, otherwise \code{NaN} is returned,
only for the means computation.
This can be used to prevent moments from being computed on too few observations.
Defaults to zero, meaning no restriction.}

\item{restart_period}{the recompute period. because subtraction of elements can cause
loss of precision, the computation of moments is restarted periodically based on 
this parameter. Larger values mean fewer restarts and faster, though potentially less 
accurate results. Unlike in the computation of even order moments, loss of precision
is unlikely to be disastrous, so the default value is rather large.}

\item{variable_win}{if true, and the \code{window} is not a concrete number,
the computation window becomes the time between lookback times.}

\item{wts_as_delta}{if true and the \code{time} and \code{time_deltas} are not
given, but \code{wts} are given, we take \code{wts} as the \code{time_deltas}.}

\item{check_wts}{a boolean for whether the code shall check for negative
weights, and throw an error when they are found. Default false for speed.}
}
\value{
A vector the same size as the lookback times.
}
\description{
Compute the mean or sum over 
an infinite or finite sliding time window, returning a vector the same size as the lookback
times.
}
\details{
Computes the mean or sum of the elements, using a Kahan's Compensated Summation Algorithm,
a numerically robust one-pass method.

Given the length \eqn{n} vector \eqn{x}, we output matrix \eqn{M} where
\eqn{M_{i,1}}{M_i,1} is the sum or mean 
of some elements \eqn{x_i} defined by the sliding time window.
Barring \code{NA} or \code{NaN}, this is over a window of time width \code{window}.
}
\note{
The moment computations provided by fromo are 
numerically robust, but will often \emph{not} provide the
same results as the 'standard' implementations,
due to differences in roundoff. We make every attempt to balance
speed and robustness. User assumes all risk from using
the fromo package.

Note that when weights are given, they are treated as replication weights.
This can have subtle effects on computations which require minimum
degrees of freedom, since the sum of weights will be compared to
that minimum, not the number of data points. Weight values
(much) less than 1 can cause computations to return \code{NA}
somewhat unexpectedly due to this condition, while values greater
than one might cause the computation to spuriously return a value
with little precision.
}
\section{Time Windowing }{


This function supports time (or other counter) based running computation. 
Here the input are the data \eqn{x_i}, and optional weights vectors, \eqn{w_i}, defaulting to 1,
and a vector of time indices, \eqn{t_i} of the same length as \eqn{x}. The
times must be non-decreasing:
\deqn{t_1 \le t_2 \le \ldots}{t_1 <= t_2 <= ...}
It is assumed that \eqn{t_0 = -\infty}.
The window, \eqn{W} is now a time-based window. 
An optional set of \emph{lookback times} are also given, \eqn{b_j}, which
may have different length than the \eqn{x} and \eqn{w}. 
The output will correspond to the lookback times, and should be the same
length. The \eqn{j}th output is computed over indices \eqn{i} such that
\deqn{b_j - W < t_i \le b_j.}

For comparison functions (like Z-score, rescaling, centering), which compare
values of \eqn{x_i} to local moments, the lookbacks may not be given, but
a lookahead \eqn{L} is admitted. In this case, the \eqn{j}th output is computed over
indices \eqn{i} such that
\deqn{t_j - W + L < t_i \le t_j + L.}

If the times are not given, \sQuote{deltas} may be given instead. If
\eqn{\delta_i} are the deltas, then we compute the times as
\deqn{t_i = \sum_{1 \le j \le i} \delta_j.}
The deltas must be the same length as \eqn{x}.  
If times and deltas are not given, but weights are given and the \sQuote{weights as deltas}
flag is set true, then the weights are used as the deltas.

Some times it makes sense to have the computational window be the space
between lookback times. That is, the \eqn{j}th output is to be
computed over indices \eqn{i} such that
\deqn{b_{j-1} - W < t_i \le b_j.}
This can be achieved by setting the \sQuote{variable window} flag true
and setting the window to null. This will not make much sense if
the lookback times are equal to the times, since each moment computation
is over a set of a single index, and most moments are underdefined.
}

\examples{
x <- rnorm(1e5)
xs <- t_running_sum(x,time=seq_along(x),window=10)
xm <- t_running_mean(x,time=cumsum(runif(length(x))),window=7.3)

}
\references{
Terriberry, T. "Computing Higher-Order Moments Online."
\url{http://people.xiph.org/~tterribe/notes/homs.html}

J. Bennett, et. al., "Numerically Stable, Single-Pass, 
Parallel Statistics Algorithms," Proceedings of IEEE
International Conference on Cluster Computing, 2009.
\url{https://www.semanticscholar.org/paper/Numerically-stable-single-pass-parallel-statistics-Bennett-Grout/a83ed72a5ba86622d5eb6395299b46d51c901265}

Cook, J. D. "Accurately computing running variance."
\url{http://www.johndcook.com/standard_deviation.html}

Cook, J. D. "Comparing three methods of computing 
standard deviation."
\url{http://www.johndcook.com/blog/2008/09/26/comparing-three-methods-of-computing-standard-deviation}

Kahan, W. "Further remarks on reducing truncation errors,"
Communications of the ACM, 8 (1), 1965.
\url{https://doi.org/10.1145/363707.363723}

Wikipedia contributors "Kahan summation algorithm," 
Wikipedia, The Free Encyclopedia, 
\url{https://en.wikipedia.org/w/index.php?title=Kahan_summation_algorithm&oldid=777164752}
(accessed May 31, 2017).
}
\author{
Steven E. Pav \email{shabbychef@gmail.com}
}
