% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{running_apx_quantiles}
\alias{running_apx_quantiles}
\alias{running_apx_median}
\title{Compute approximate quantiles over a sliding window}
\usage{
running_apx_quantiles(
  v,
  p,
  window = NULL,
  wts = NULL,
  max_order = 5L,
  na_rm = FALSE,
  min_df = 0L,
  used_df = 0,
  restart_period = 100L,
  check_wts = FALSE,
  normalize_wts = TRUE,
  check_negative_moments = TRUE
)

running_apx_median(
  v,
  window = NULL,
  wts = NULL,
  max_order = 5L,
  na_rm = FALSE,
  min_df = 0L,
  used_df = 0,
  restart_period = 100L,
  check_wts = FALSE,
  normalize_wts = TRUE,
  check_negative_moments = TRUE
)
}
\arguments{
\item{v}{a vector}

\item{p}{the probability points at which to compute the quantiles. Should be in the range (0,1).}

\item{window}{the window size. if given as finite integer or double, passed through.
If \code{NULL}, \code{NA_integer_}, \code{NA_real_} or \code{Inf} are given, equivalent
to an infinite window size. If negative, an error will be thrown.}

\item{wts}{an optional vector of weights. Weights are \sQuote{replication}
weights, meaning a value of 2 is shorthand for having two observations
with the corresponding \code{v} value. If \code{NULL}, corresponds to
equal unit weights, the default. Note that weights are typically only meaningfully defined
up to a multiplicative constant, meaning the units of weights are
immaterial, with the exception that methods which check for minimum df will,
in the weighted case, check against the sum of weights. For this reason,
weights less than 1 could cause \code{NA} to be returned unexpectedly due
to the minimum condition. When weights are \code{NA}, the same rules for checking \code{v}
are applied. That is, the observation will not contribute to the moment
if the weight is \code{NA} when \code{na_rm} is true. When there is no
checking, an \code{NA} value will cause the output to be \code{NA}.}

\item{max_order}{the maximum order of the centered moment to be computed.}

\item{na_rm}{whether to remove NA, false by default.}

\item{min_df}{the minimum df to return a value, otherwise \code{NaN} is returned.
This can be used to prevent moments from being computed on too few observations.
Defaults to zero, meaning no restriction.}

\item{used_df}{the number of degrees of freedom consumed, used in the denominator
of the centered moments computation. These are subtracted from the number of
observations.}

\item{restart_period}{the recompute period. because subtraction of elements can cause
loss of precision, the computation of moments is restarted periodically based on 
this parameter. Larger values mean fewer restarts and faster, though less accurate
results.}

\item{check_wts}{a boolean for whether the code shall check for negative
weights, and throw an error when they are found. Default false for speed.}

\item{normalize_wts}{a boolean for whether the weights should be
renormalized to have a mean value of 1. This mean is computed over elements
which contribute to the moments, so if \code{na_rm} is set, that means non-NA
elements of \code{wts} that correspond to non-NA elements of the data
vector.}

\item{check_negative_moments}{a boolean flag. Normal computation of running
moments can result in negative estimates of even order moments due to loss of
numerical precision. With this flag active, the computation checks for negative
even order moments and restarts the computation when one is detected. This
should eliminate the possibility of negative even order moments. The
downside is the speed hit of checking on every output step. Note also the
code checks for negative moments of every even order tracked, even if they
are not output; that is if the kurtosis, say, is being computed, and a
negative variance is detected, then the computation is restarted.
Defaults to \code{TRUE} to avoid negative even moments. Set to \code{FALSE}
only if you know what you are doing.}
}
\value{
A matrix, with one row for each element of \code{x}, and one column for each element of \code{q}.
}
\description{
Computes cumulants up to some given order, then employs the Cornish-Fisher approximation
to compute approximate quantiles using a Gaussian basis.
}
\details{
Computes the cumulants, then approximates quantiles using AS269 of Lee & Lin.
}
\note{
The current implementation is not as space-efficient as it could be, as it first computes
the cumulants for each row, then performs the Cornish-Fisher approximation on a row-by-row
basis. In the future, this computation may be moved earlier into the pipeline to be more
space efficient. File an issue if the memory footprint is an issue for you.

The moment computations provided by fromo are 
numerically robust, but will often \emph{not} provide the
same results as the 'standard' implementations,
due to differences in roundoff. We make every attempt to balance
speed and robustness. User assumes all risk from using
the fromo package.

Note that when weights are given, they are treated as replication weights.
This can have subtle effects on computations which require minimum
degrees of freedom, since the sum of weights will be compared to
that minimum, not the number of data points. Weight values
(much) less than 1 can cause computations to return \code{NA}
somewhat unexpectedly due to this condition, while values greater
than one might cause the computation to spuriously return a value
with little precision.
}
\examples{
x <- rnorm(1e5)
xq <- running_apx_quantiles(x,c(0.1,0.25,0.5,0.75,0.9))
xm <- running_apx_median(x)

}
\references{
Terriberry, T. "Computing Higher-Order Moments Online."
\url{https://web.archive.org/web/20140423031833/http://people.xiph.org/~tterribe/notes/homs.html}

J. Bennett, et. al., "Numerically Stable, Single-Pass, 
Parallel Statistics Algorithms," Proceedings of IEEE
International Conference on Cluster Computing, 2009.
\doi{10.1109/CLUSTR.2009.5289161}

Cook, J. D. "Accurately computing running variance."
\url{https://www.johndcook.com/standard_deviation/}

Cook, J. D. "Comparing three methods of computing 
standard deviation."
\url{https://www.johndcook.com/blog/2008/09/26/comparing-three-methods-of-computing-standard-deviation/}
}
\seealso{
\code{\link{t_running_apx_quantiles}}, \code{\link{running_cumulants}}, \code{PDQutils::qapx_cf}, \code{PDQutils::AS269}.
}
\author{
Steven E. Pav \email{shabbychef@gmail.com}
}
