% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/funDataClass.R
\docType{class}
\name{multiFunData-class}
\alias{multiFunData}
\alias{multiFunData,ANY-method}
\alias{multiFunData-class}
\title{A class for multivariate functional data}
\usage{
\S4method{multiFunData}{ANY}(...)
}
\arguments{
\item{...}{A list of funData objects or several funData objects passed as one
argument, each. See Details.}
}
\description{
The \code{multiFunData} class represents multivariate functional data on 
(potentially) different domains, i.e. a multivariate functional data object 
is a vector of (univariate) functional data objects, just as a vector in 
\eqn{\mathrm{IR}^n}{IR^n} is a vector of \eqn{n}{n} scalars. In this implementation, a 
\code{multiFunData} object is represented as a list of univariate 
\code{funData} objects, see Details.
}
\details{
A \code{multiFunData} object is represented as a list of univariate 
\code{funData} objects, each having a \code{argvals} and \code{X} slot, 
representing the x-values and the observed y-values (see the 
\code{\link{funData}} class). When constructing a \code{multiFunData} object,
the  elements can be supplied as a list of \code{funData} objects or can be
passed directly as arguments to the constructor function.

Most functions implemented for the \code{\link{funData}} class are also 
implemented for \code{multiFunData} objects. In most cases, they simply apply
the corresponding univariate method to each element of the multivariate 
object and return it as a vector (if the result of the univariate function is
scalar, such as \code{\link{dimSupp}}) or as a \code{multiFunData} object (if
the result of the univariate function is a \code{funData} object, such as 
\code{\link{extractObs}}).

The norm of a multivariate functional data \eqn{f = (f_1 , \ldots, f_p)}{f =
(f_1 , \ldots, f_p)} is defined as \deqn{||| f ||| := \left(\sum_{j=1}^p ||
f_j ||^2 \right) ^{1/2}.}{||| f ||| := ( \sum || f_j ||^2 )^{1/2}.}

A \code{funData} object can be coerced to a \code{multiFunData} object with
one element using \code{as.multiFunData(funDataObject).}
}
\section{Methods (by generic)}{
\itemize{
\item \code{multiFunData}: Constructor for multivariate functional data
objects.
}}
\examples{
### Creating a multifunData object with 2 observations on the same domain
# Univariate elements
x <- 1:5
f1 <- funData(x, rbind(x, x+1))
f2 <- funData(x,rbind(x^2, sin(x)))
# Basic
m1 <- new("multiFunData", list(f1,f2))
# Using the constructor, passing the elements as list
m2 <- multiFunData(list(f1,f2))
# Using the constructor, passing the elements directly
m3 <- multiFunData(f1,f2)
# Test if all the same
all.equal(m1,m2)
all.equal(m1,m3)
# Display multiFunData object in the console
m3

### Creating a multifunData object with 2 observations on different domains (both 1D)
# A new element
y <- 1:3
g1 <- funData(y, rbind(3*y, y+4))
# Create the multiFunData object
m4 <- multiFunData(f1,g1)
# Display multiFunData object in the console
m4

### Creating a multifunData object with 2 observations on different domains (1D and 2D)
# A new element
y <- 1:3; z <- 1:4
g2 <- funData(list(y,z), array(rnorm(24), dim = c(2,3,4)))
# Create the multiFunData object
m5 <- multiFunData(f1,g2)
# Display multiFunData object in the console
m5

### A more realistic object
# element 1
x <- seq(0,2*pi, 0.01)
f1 <- funData(x, outer(seq(0.75, 1.25, length.out = 6), sin(x)))
# element 2
y <- seq(-1,1, 0.01); z <- seq(-0.5, 0.5, 0.01)
X2 <- array(NA, c(6, length(y), length(z)))
for(i in 1:6) X2[i,,] <- outer(y, z, function(x,y){sin(i*pi*y)*cos(i*pi*z)})
f2 <- funData(list(y,z), X2)
# MultiFunData Object
m6 <- multiFunData(f1,f2)
# Display multiFunData object in the console for basic information
m6
# Use the plot function to get an impression of the data
\dontrun{plot(m6)} # m6 has 2D element, must specify one observation for plotting
plot(m6, obs = 1, main = c("1st element (obs 1)", "2nd element (obs 1)"))
plot(m6, obs = 6, main = c("1st element (obs 6)", "2nd element (obs 6)"))
}
\seealso{
\code{\linkS4class{funData}}
}

