% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/faMB.R
\name{faMB}
\alias{faMB}
\title{Multiple Battery Factor Analysis by Maximum Likelihood Methods}
\usage{
faMB(
  X = NULL,
  R = NULL,
  n = NULL,
  NB = NULL,
  NVB = NULL,
  numFactors = NULL,
  epsilon = 1e-06,
  rotate = "oblimin",
  rotateControl = NULL,
  PrintLevel = 0,
  Seed = 1
)
}
\arguments{
\item{X}{(Matrix) A raw data matrix (or data frame) structured in a subject 
(row) by variable (column) format. Defaults to \code{X = NULL}.}

\item{R}{(Matrix) A correlation matrix. Defaults to \code{R = NULL}.}

\item{n}{(Numeric) Sample size associated with either the raw data (X) or 
the correlation matrix (R). Defaults to \code{n = NULL}.}

\item{NB}{(Numeric) The number of batteries to analyze. In interbattery factor analysis NB = 2.}

\item{NVB}{(Vector) The number of variables in each battery. For example, 
analyzing three batteries including seven, four, and five variables 
(respectively) would be specified as \code{NVB = c(7, 4, 5)}.}

\item{numFactors}{(Numeric) The number of factors to extract for subsequent 
rotation.}

\item{epsilon}{(Numeric) The convergence threshold for the Gauss-Seidel iterator
when analyzing three or more batteries. Defaults to \code{epsilon = 1e-06}.}

\item{rotate}{(Character) Designate which rotation algorithm to apply. The 
following are available rotation options: "oblimin", "quartimin", "targetT", 
"targetQ", "oblimax", "entropy", "quartimax", "varimax", "simplimax", 
"bentlerT", "bentlerQ", "tandemI", "tandemII", "geominT", "geominQ", "cfT", 
"cfQ", "infomaxT", "infomaxQ", "mccammon", "bifactorT", "bifactorQ", and 
"none". Defaults to rotate = "oblimin". See \pkg{GPArotation} package for more 
details. Note that rotations ending in "T" and "Q" represent orthogonal and 
oblique rotations, respectively.}

\item{rotateControl}{(List) A list of control values to pass to the factor rotation algorithms.
\itemize{
  \item \strong{numberStarts}: (Numeric) The number of random (orthogonal) 
  starting configurations for the chosen rotation method (e.g., oblimin). The first
  rotation will always commence from the unrotated factors orientation.
  Defaults to numberStarts = 10. 
  \item \strong{gamma}: (Numeric) This is a tuning parameter (between 0 
  and 1, inclusive) for an oblimin rotation.  See the \pkg{GPArotation} 
  library's oblimin documentation for more details. Defaults to gamma = 0 
  (i.e., a quartimin rotation).
  \item \strong{delta}: (Numeric) This is a tuning parameter for the geomin
   rotation. It adds a small number (default = .01) to the squared factor 
   loadings before computing the geometric means in the discrepancy function.
  \item \strong{kappa}: (Numeric) The main parameterization of the 
  Crawford-Ferguson (CF) rotations (i.e., "cfT" and "cfQ" for orthogonal and 
  oblique CF rotation, respectively). Defaults to kappa = 0. 
  \item \strong{k}: (Numeric) A specific parameter of the simplimax rotation. 
  Defaults to k = the number of observed variables.
  \item \strong{standardize}: (Character) The standardization routine used 
  on the unrotated factor structure. The three options are "none", "Kaiser", 
  and "CM". Defaults to standardize = "none". 
  \itemize{
    \item \strong{"none"}: No standardization is applied to the unrotated 
    factor structure. 
    \item \strong{"Kaiser"}: Use a factor structure matrix that has been 
    normed by Kaiser's method (i.e., normalize all rows to have a unit length).
    \item \strong{"CM"}: Use a factor structure matrix that has been normed
     by the Cureton-Mulaik method.
  }
  \item \strong{epsilon}: (Numeric) The rotational convergence criterion to 
  use. Defaults to epsilon = 1e-5.
  \item \strong{power}: (Numeric) Raise factor loadings the the n-th power 
  in the \code{\link{promaxQ}} rotation. Defaults to power = 4.
  \item \strong{maxItr}: (Numeric) The maximum number of iterations for the 
  rotation algorithm. Defaults to maxItr = 15000.
}}

\item{PrintLevel}{(Numeric) When a value greater than zero is specified, 
\code{PrintLevel} prints the maximum change in communality estimates 
for each iteration of the Gauss-Seidel function. Note that Gauss-Seidel 
iteration is only called when three or more 
batteries are analyzed. Defaults to \code{PrintLevel = 0}.}

\item{Seed}{(Integer) Starting seed for the random number generator. 
Defaults to \code{Seed = 1}.}
}
\value{
The \code{faMB} function will produce abundant output in addition
to the rotated multiple battery factor pattern and factor correlation matrices. 

\itemize{
  \item \strong{loadings}: (Matrix) The (possibly) rotated multiple battery factor solution with the 
    lowest evaluated complexity value \emph{of the examined random starting configurations}. 
    It is not guaranteed to find the "true" global minimum. Note that multiple
     (or even all) local solutions can have the same discrepancy functions.
  \item \strong{Phi}: (Matrix) The factor correlations of the rotated factor 
    solution with the lowest evaluated discrepancy function (see Details).
  \item \strong{fit}: (Vector) A vector containing the following fit statistics:
  \itemize{
    \item \strong{ChiSq}: Chi-square goodness of fit value. 
    Note that, as recommended by Browne (1979), we apply Lawley's (1959) correction when computing the chi-square value when \code{NB = 2}.
    \item \strong{DF}: Degrees of freedom for the estimated model. 
    \item \strong{pvalue}: P-value associated with the above chi-square statistic.
    \item \strong{AIC}: Akaike's Information Criterion where a lower value indicates better fit. 
    \item \strong{BIC}: Bayesian Information Criterion where a lower value indicates better fit. 
    \item \strong{RMSEA}: Root mean squared error of approximation (Steiger & Lind, 1980).
  }
  \item \strong{R}: (Matrix) The \emph{sample} correlation matrix, 
  useful when raw data are supplied. 
  \item \strong{Rhat}: (Matrix) The \emph{reproduced} correlation matrix with communalities on the diagonal. 
  \item \strong{Resid}: (Matrix) A residual matrix (R - Rhat). 
  \item \strong{facIndeterminacy}: (Vector) A vector (with length equal to the number of factors)
  containing Guttman's (1955) index of factor indeterminacy for each factor. 
  \item \strong{localSolutions}: (List) A list (of length equal to the 
  \code{numberStarts} argument within \code{rotateControl}) containing all local solutions 
  in ascending order of their rotation complexity values (i.e., the first solution 
  is the "global" minimum). Each solution returns the following:
  \itemize{
     \item \strong{loadings}: (Matrix) the factor loadings, 
     \item \strong{Phi}: (Matrix) factor correlations, 
     \item \strong{RotationComplexityValue}: (Numeric) the complexity value of the rotation algorithm, 
     \item \strong{facIndeterminacy}: (Vector) A vector of factor indeterminacy indices for each common factor, and 
     \item \strong{RotationConverged}: (Logical) convergence status of the rotation algorithm. 
     }
  \item \strong{numLocalSets}: (Numeric) An integer indicating how many sets of local solutions
   with the same discrepancy value were obtained. 
  \item \strong{localSolutionSets}: (List) A list (of length equal to the 
  \code{numLocalSets}) that contains all local solutions with the same 
  rotation complexity value. Note that it is not guarenteed that all 
  solutions with the same complexity values have equivalent factor loading patterns. 
  \item \strong{rotate}: (Character) The chosen rotation algorithm.
  \item \strong{rotateControl}: (List) A list of the control parameters 
  passed to the rotation algorithm.
  \item \strong{unSpunSolution}: (List) A list of output parameters (e.g., loadings, Phi, etc) from 
  the rotated solution that was obtained by rotating directly from the unspun 
  (i.e., not multiplied by a random orthogonal transformation matrix) common 
  factor orientation. 
  \item \strong{Call}: (call) A copy of the function call.

}
}
\description{
\code{faMB} estimates multiple battery factor analysis using maximum 
likelihood estimation procedures described by Browne (1979, 1980). Unrotated
multiple battery solutions are rotated (using the \pkg{GPArotation} package) 
from a user-specified number of of random (orthogonal) starting configurations. 
Based on procedures analogous to those in the \code{\link{faMain}} function,
rotation complexity values of all solutions are ordered to determine
the number of local solutions and the "global" minimum solution (i.e., the 
minimized rotation complexity value from the finite number of solutions).
}
\examples{
# These examples reproduce published multiple battery analyses. 

# ----EXAMPLE 1: Browne, M. W. (1979)----
#
# Data originally reported in:
# Thurstone, L. L. & Thurstone, T. G. (1941). Factorial studies 
# of intelligence. Psychometric Monograph (2), Chicago: Univ. 
# Chicago Press.

## Load Thurstone & Thurstone's data used by Browne (1979)
data(Thurstone41)

Example1Output <-  faMB(R             = Thurstone41, 
                        n             = 710,
                        NB            = 2, 
                        NVB           = c(4,5), 
                        numFactors    = 2,
                        rotate        = "oblimin",
                        rotateControl = list(standardize = "Kaiser"))
                        
summary(Example1Output, PrintLevel = 2)                         

# ----EXAMPLE 2: Browne, M. W. (1980)----
# Data originally reported in:
# Jackson, D. N. & Singer, J. E. (1967). Judgments, items and 
# personality. Journal of Experimental Research in Personality, 20, 70-79.

## Load Jackson and Singer's dataset
data(Jackson67)



Example2Output <-  faMB(R             = Jackson67, 
                        n             = 480,
                        NB            = 5, 
                        NVB           = rep(4,5), 
                        numFactors    = 4,
                        rotate        = "varimax",
                        rotateControl = list(standardize = "Kaiser"),
                        PrintLevel    = 1)
                        
summary(Example2Output)                         



# ----EXAMPLE 3: Cudeck (1982)----
# Data originally reported by:
# Malmi, R. A., Underwood, B. J., & Carroll, J. B. (1979).
# The interrelationships among some associative learning tasks. 
# Bulletin of the Psychonomic Society, 13(3), 121-123. DOI: 10.3758/BF03335032 

## Load Malmi et al.'s dataset
data(Malmi79)

Example3Output <- faMB(R             = Malmi79, 
                       n             = 97,
                       NB            = 3, 
                       NVB           = c(3, 3, 6), 
                       numFactors    = 2,
                       rotate        = "oblimin",
                       rotateControl = list(standardize = "Kaiser"))
                       
summary(Example3Output)                        



# ----Example 4: Cudeck (1982)----
# Data originally reported by: 
# Boruch, R. F., Larkin, J. D., Wolins, L. and MacKinney, A. C. (1970). 
#  Alternative methods of analysis: Multitrait-multimethod data. Educational 
#  and Psychological Measurement, 30,833-853.

## Load Boruch et al.'s dataset
data(Boruch70)

Example4Output <- faMB(R             = Boruch70,
                       n             = 111,
                       NB            = 2,
                       NVB           = c(7,7),
                       numFactors    = 2,
                       rotate        = "oblimin",
                       rotateControl = list(standardize  = "Kaiser",
                                            numberStarts = 100))
                                            
summary(Example4Output, digits = 3)                                             

}
\references{
Boruch, R. F., Larkin, J. D., Wolins, L., & MacKinney, A. C. (1970). 
Alternative methods of analysis: Multitrait-multimethod data. \emph{Educational 
and Psychological Measurement, 30}(4), 833–853. 
https://doi.org/10.1177/0013164470030004055

Browne, M. W.  (1979).  The maximum-likelihood solution in inter-battery factor analysis. 
\emph{British Journal of Mathematical and Statistical Psychology, 32}(1), 75-86.

Browne, M. W.  (1980).  Factor analysis of multiple batteries by maximum likelihood.  
\emph{British Journal of Mathematical and Statistical Psychology, 33}(2), 184-199.

Browne, M. W. (2001). An overview of analytic rotation in 
exploratory factor analysis. \emph{Multivariate Behavioral Research, 36}(1), 111-150.

Browne, M. and Cudeck, R. (1992). Alternative ways of assessing model fit. 
\emph{Sociological Methods and Research, 21(2)}, 230-258.

Burnham, K. P. & Anderson, D. R.  (2004).  Multimodel inference: Understanding AIC and BIC in model selection.  
\emph{Sociological methods and research, 33}, 261-304.

Cudeck, R. (1982). Methods for estimating between-battery factors,
\emph{Multivariate Behavioral Research, 17}(1), 47-68. 10.1207/s15327906mbr1701_3

Cureton, E. E., & Mulaik, S. A. (1975). The weighted varimax 
rotation and the promax rotation. \emph{Psychometrika, 40}(2), 183-195.

Guttman, L. (1955). The determinacy of factor score matrices with 
implications for five other basic problems of common factor theory. 
\emph{British Journal of Statistical Psychology, 8}(2), 65-81.

Steiger, J. & Lind, J. (1980). Statistically based tests for the
 number of common factors. In \emph{Annual meeting of the Psychometric Society, 
 Iowa City, IA, volume 758}.

Tucker, L. R.  (1958).  An inter-battery method of factor analysis.  
\emph{Psychometrika, 23}(2), 111-136.
}
\seealso{
Other Factor Analysis Routines: 
\code{\link{BiFAD}()},
\code{\link{Box26}},
\code{\link{GenerateBoxData}()},
\code{\link{Ledermann}()},
\code{\link{SLi}()},
\code{\link{SchmidLeiman}()},
\code{\link{faAlign}()},
\code{\link{faEKC}()},
\code{\link{faIB}()},
\code{\link{faMain}()},
\code{\link{faScores}()},
\code{\link{faSort}()},
\code{\link{faStandardize}()},
\code{\link{faX}()},
\code{\link{fals}()},
\code{\link{fapa}()},
\code{\link{fareg}()},
\code{\link{orderFactors}()},
\code{\link{print.faMB}()},
\code{\link{print.faMain}()},
\code{\link{promaxQ}()},
\code{\link{summary.faMB}()},
\code{\link{summary.faMain}()}
}
\author{
\itemize{
  \item Niels G. Waller (nwaller@umn.edu)
  \item Casey Giordano (Giord023@umn.edu)
}
}
\concept{Factor Analysis Routines}
