% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gdalraster_proc.R
\name{combine}
\alias{combine}
\title{Raster overlay for unique combinations}
\usage{
combine(
  rasterfiles,
  var.names = NULL,
  bands = NULL,
  dstfile = NULL,
  fmt = NULL,
  dtName = "UInt32",
  options = NULL,
  quiet = FALSE
)
}
\arguments{
\item{rasterfiles}{Character vector of raster filenames to combine.}

\item{var.names}{Character vector of \code{length(rasterfiles)} containing
variable names for each raster layer. Defaults will be assigned if
\code{var.names} are omitted.}

\item{bands}{Numeric vector of \code{length(rasterfiles)} containing the band
number to use for each raster in \code{rasterfiles}. Band 1 will be used for
each input raster if \code{bands} are not specified.}

\item{dstfile}{Character. Optional output raster filename for writing the
per-pixel combination IDs. The output raster will be created (and
overwritten if it already exists).}

\item{fmt}{Character. Output raster format name (e.g., "GTiff" or "HFA").}

\item{dtName}{Character. Output raster data type name. Combination IDs are
sequential integers starting at 1. The data type for the output raster
should be large enough to accommodate the potential number of unique
combinations of the input values
(e.g., "UInt16" or the default "UInt32").}

\item{options}{Optional list of format-specific creation options in a
vector of "NAME=VALUE" pairs
(e.g., \code{options = c("COMPRESS=LZW")} to set LZW compression
during creation of a GTiff file).}

\item{quiet}{Logical scalar. If \code{TRUE}, progress bar and messages will be
suppressed. Defaults to \code{FALSE}.}
}
\value{
A data frame with column \code{cmbid} containing the combination IDs,
column \code{count} containing the pixel counts for each combination,
and \code{length(rasterfiles)} columns named \code{var.names} containing the integer
values comprising each unique combination.
}
\description{
\code{combine()} overlays multiple rasters so that a unique ID is assigned to
each unique combination of input values. The input raster layers
typically have integer data types (floating point will be coerced to
integer by truncation), and must have the same projection, extent and cell
size. Pixel counts for each unique combination are obtained, and
combination IDs are optionally written to an output raster.
}
\details{
To specify input raster layers that are bands of a multi-band
raster file, repeat the filename in \code{rasterfiles} and provide the
corresponding band numbers in \code{bands}. For example:
\preformatted{
rasterfiles <- c("multi-band.tif", "multi-band.tif", "other.tif")
bands <- c(4, 5, 1)
var.names <- c("multi_b4", "multi_b5", "other")
}

\code{\link[=rasterToVRT]{rasterToVRT()}} provides options for virtual clipping, resampling and pixel
alignment, which may be helpful here if the input rasters are not already
aligned on a common extent and cell size.

If an output raster of combination IDs is written, the user should verify
that the number of combinations obtained did not exceed the range of the
output data type. Combination IDs are sequential integers starting at 1.
Typical output data types are the unsigned types:
Byte (0 to 255), UInt16 (0 to 65,535) and UInt32 (the default, 0 to
4,294,967,295).
}
\examples{
evt_file <- system.file("extdata/storml_evt.tif", package="gdalraster")
evc_file <- system.file("extdata/storml_evc.tif", package="gdalraster")
evh_file <- system.file("extdata/storml_evh.tif", package="gdalraster")
rasterfiles <- c(evt_file, evc_file, evh_file)
var.names <- c("veg_type", "veg_cov", "veg_ht")
tbl <- combine(rasterfiles, var.names)
nrow(tbl)
tbl <- tbl[order(-tbl$count),]
head(tbl, n = 20)

# combine two bands from a multi-band file and write the combination IDs
# to an output raster
lcp_file <- system.file("extdata/storm_lake.lcp", package="gdalraster")
rasterfiles <- c(lcp_file, lcp_file)
bands <- c(4, 5)
var.names <- c("fbfm", "tree_cov")
cmb_file <- paste0(tempdir(), "/", "fbfm_cov_cmbid.tif")
opt <- c("COMPRESS=LZW")
tbl <- combine(rasterfiles, var.names, bands, cmb_file, options = opt)
head(tbl)
ds <- new(GDALRaster, cmb_file)
ds$info()
ds$close()
deleteDataset(cmb_file)
}
\seealso{
\code{\link[=CmbTable]{CmbTable-class}}, \code{\link[=GDALRaster]{GDALRaster-class}}, \code{\link[=calc]{calc()}},
\code{\link[=rasterToVRT]{rasterToVRT()}}

\code{\link[=buildRAT]{buildRAT()}} to compute a table of the unique pixel values and their counts
for a single raster layer
}
