# ===========================================================
#     _/_/_/   _/_/_/  _/_/_/_/    _/_/_/_/  _/_/_/   _/_/_/
#      _/    _/       _/             _/    _/    _/   _/   _/
#     _/    _/       _/_/_/_/       _/    _/    _/   _/_/_/
#    _/    _/       _/             _/    _/    _/   _/
# _/_/_/   _/_/_/  _/_/_/_/_/     _/     _/_/_/   _/_/
# ===========================================================
#
# GDS.all.r: the R interface of CoreArray library
#
# Copyright (C) 2012	Xiuwen Zheng
#
# This file is part of CoreArray.
#
# CoreArray is free software: you can redistribute it and/or modify it
# under the terms of the GNU Lesser General Public License Version 3 as
# published by the Free Software Foundation.
#
# CoreArray is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with CoreArray.
# If not, see <http://www.gnu.org/licenses/>.


##################################################################################
# File Operations
##################################################################################

#############################################################
# To create a new CoreArray Genomic Data Structure (GDS) file
#
createfn.gds <- function(fn)
{
	r <- .C("gdsCreateGDS", filename=as.character(fn), id=integer(1),
		root=integer(2), err=integer(1), NAOK=TRUE, PACKAGE="gdsfmt")
	if (r$err != 0)
	{
		stop(lasterr.gds())
		return(invisible(NULL))
	} else {
		class(r$root) <- "gdsn"
		r$readonly <- FALSE
		class(r) <- "gdsclass"
		return(r)
	}
}


#############################################################
# To open an existing file
#
openfn.gds <- function(fn, readonly=TRUE)
{
	r <- .C("gdsOpenGDS", filename=as.character(fn[1]), id=integer(1),
		root=integer(2), readonly=as.integer(readonly), err=integer(1),
		NAOK=TRUE, PACKAGE="gdsfmt")
	if (r$err != 0)
	{
		stop(lasterr.gds())
		return(invisible(NULL))
	} else {
		class(r$root) <- "gdsn"
		r$readonly <- readonly
		class(r) <- "gdsclass"
		return(r)
	}
}


#############################################################
# To close an open CoreArray Genomic Data Structure (GDS) file
#
closefn.gds <- function(gds)
{
	stopifnot(class(gds)=="gdsclass")
	r <- .C("gdsCloseGDS", as.integer(gds$id), err=integer(1), NAOK=TRUE, PACKAGE="gdsfmt")
	if (r$err != 0)
		stop(lasterr.gds())
	return(invisible(NULL))
}


#############################################################
# To write the data cached in memory to disk
#
sync.gds <- function(gds)
{
	stopifnot(class(gds)=="gdsclass")
	r <- .C("gdsSyncGDS", as.integer(gds$id), err=integer(1), NAOK=TRUE, PACKAGE="gdsfmt")
	if (r$err != 0)
		stop(lasterr.gds())
	return(invisible(NULL))
}


#############################################################
# To clean up fragments of a GDS file
#
cleanup.gds <- function(fn, verbose=TRUE)
{
	stopifnot(is.character(fn))

	r <- .C("gdsTidyUp", fn, as.logical(verbose), err=integer(1),
		NAOK=TRUE, PACKAGE="gdsfmt")
	if (r$err != 0) stop(lasterr.gds())

	return(invisible(NULL))
}






##################################################################################
# File Structure Operations
##################################################################################

#############################################################
# To get the number of child nodes for a specified node
#
cnt.gdsn <- function(node)
{
	stopifnot(class(node)=="gdsn")
	r <- .C("gdsNodeChildCnt", as.integer(node), cnt=as.integer(1), NAOK=TRUE,
		PACKAGE="gdsfmt")
	if (r$cnt < 0)
		stop(lasterr.gds())
	return(r$cnt)
}


#############################################################
# To get the variable name of a node
#
name.gdsn <- function(node, fullname=FALSE)
{
	stopifnot(class(node)=="gdsn")
	r <- .C("gdsNodeName", as.integer(node), name=character(1),
		as.integer(fullname), err=integer(1), NAOK=TRUE, PACKAGE="gdsfmt")
	if (r$err != 0) {
		stop(lasterr.gds())
		return(invisible(NULL))
	} else { return(r$name) }
}


#############################################################
# To rename a GDS node
#
rename.gdsn <- function(node, newname)
{
	stopifnot(class(node)=="gdsn")
	r <- .C("gdsRenameNode", as.integer(node), as.character(newname),
		err=integer(1), NAOK=TRUE, PACKAGE="gdsfmt")
	if (r$err != 0)
		stop(lasterr.gds())
	return(node)
}


#############################################################
# To get a list of names for the child nodes
#
ls.gdsn <- function(node)
{
	if (class(node)=="gdsclass")
		node <- node$root
	stopifnot(class(node)=="gdsn")
	cnt <- cnt.gdsn(node)
	r <- .C("gdsNodeEnumName", as.integer(node), names=character(cnt),
		err=integer(1), NAOK=TRUE, PACKAGE="gdsfmt")
	if (r$err != 0)
	{
		stop(lasterr.gds())
		return(invisible(NULL))
	} else
		return(r$names)
}


#############################################################
# To get a specified node
#
index.gdsn <- function(node, index, silent=FALSE)
{
	if (class(node)=="gdsclass")
		node <- node$root
	stopifnot(class(node)=="gdsn")
	stopifnot(is.character(index) | is.numeric(index))
	if (missing(index))
		return(node)
	cnt <- cnt.gdsn(node)
	if (is.character(index))
	{
		r <- .C("gdsNodeIndexEx", node=as.integer(node), as.character(index),
			as.integer(length(index)), err=integer(1), NAOK=TRUE, PACKAGE="gdsfmt")
	} else {
		r <- .C("gdsNodeIndex", node=as.integer(node), as.integer(index),
			as.integer(length(index)), err=integer(1), NAOK=TRUE, PACKAGE="gdsfmt")
	}
	if (r$err != 0)
	{
		if (!silent) stop(lasterr.gds())
		rv <- NULL
	} else {
		rv <- r$node
		class(rv) <- "gdsn"
	}
	return(rv)
}


#############################################################
# To get the descritpion of a specified node
#
objdesp.gdsn <- function(node)
{
	stopifnot(class(node)=="gdsn")
	cnt <- cnt.gdsn(node)
	r <- .C("gdsNodeObjDesp", as.integer(node), desp=character(1), name=character(1),
		sv=integer(1), arr=logical(1), dimcnt=integer(1), dimeach=integer(1024),
		maxlen=integer(1), cp=character(1), cpratio=double(1), err=integer(1),
		NAOK=TRUE, PACKAGE="gdsfmt")
	if (r$err == 0)
	{
		if (r$dimcnt > 0)
			d <- rev(r$dimeach[1:r$dimcnt])
		else
			d <- NULL
		rv <- list(desp = r$desp, name = r$name, svtype = r$sv, is.array = r$arr,
			dim = d, compress = r$cp, cpratio = r$cpratio)
		if (r$maxlen >= 0)
			rv$maxlen <- r$maxlen
	} else {
		stop(lasterr.gds())
	}
	return(rv)
}


#############################################################
# To add a GDS node
#
add.gdsn <- function(node, name, val=NULL, storage=storage.mode(val), valdim=NULL,
	compress=c("", "ZIP", "ZIP.fast", "ZIP.default", "ZIP.max"),
	closezip=FALSE)
{
	if (class(node)=="gdsclass")
		node <- node$root
	stopifnot(class(node)=="gdsn")

	if (missing(name))
		name <- paste("Item", cnt.gdsn(node)+1, sep="")

	# if val is factor
	if (is.factor(val))
	{
		tmp <- val
		val <- as.character(tmp)
		val[is.na(tmp)] <- ""; rm(tmp)
		storage <- "factor"
	} else if (is.character(val))
	{
		val[is.na(val)] <- ""
	}

	if (storage == "") storage <- "NULL"

	if (is.null(valdim))
	{
		if (!(storage %in% c("NULL", "list")))
		{
			val <- as.array(val)
			valdim <- dim(val)
		} else
			valdim <- c()
	} else {
		if (storage == "NULL") storage <- "integer"

		if (!is.null(val))
		{
			rv <- add.gdsn(node, name, val=val, storage=storage,
				compress=compress, closezip=FALSE)
			r <- .C("gdsObjSetDim", as.integer(rv), length(valdim), rev(as.integer(valdim)),
				err = integer(1), NAOK = TRUE, PACKAGE = "gdsfmt")
			if (r$err != 0) stop(lasterr.gds())
			return(rv)
		}
	}

	if (is.character(val))
		MaxLen <- max(nchar(val))
	else
		MaxLen <- 1

	r <- .C("gdsAddNode", node=as.integer(node), as.character(name),
		as.character(storage), as.character(compress[1]),
		length(valdim), as.integer(rev(valdim)), as.integer(MaxLen), is.null(val),
		err=integer(1), NAOK=TRUE, PACKAGE="gdsfmt")

	if (r$err == 0)
	{
		rv <- r$node; class(rv) <- "gdsn"
		if (!is.null(val))
		{
			if (!(storage %in% c("NULL", "list")))
			{
				append.gdsn(rv, val)
				if (compress[1] != "")
				{
					if (closezip) readmode.gdsn(rv)
				}
			}
		}

		if (storage == "list")
		{
			put.attr.gdsn(rv, "R.class", class(val))
			iNames <- names(val); iN <- 1
			for (v in val)
			{
				add.gdsn(rv, iNames[iN], v, compress=compress, closezip=closezip)
				iN <- iN + 1
			}
		} else if (storage == "logical") {
			put.attr.gdsn(rv, "R.logical")
		} else if (storage == "factor") {
			put.attr.gdsn(rv, "R.factor")
		}

		return(rv)
	} else {
		stop(lasterr.gds())
		return()
	}
}


#############################################################
# To add a GDS node with a file
#
addfile.gdsn <- function(node, name, filename,
	compress=c("ZIP", "ZIP.fast", "ZIP.default", "ZIP.max", ""))
{
	if (class(node)=="gdsclass")
		node <- node$root
	stopifnot(class(node)=="gdsn")
	stopifnot(is.character(filename))

	if (missing(name))
		name <- paste("Item", cnt.gdsn(node)+1, sep="")

	r <- .C("gdsAddFile", node=as.integer(node), as.character(name), as.character(filename[1]),
		as.character(compress[1]), err=integer(1), NAOK=TRUE, PACKAGE="gdsfmt")

	if (r$err != 0) stop(lasterr.gds())
	rv <- r$node; class(rv) <- "gdsn"
	return(rv)
}


#############################################################
# To delete a specified node
#
delete.gdsn <- function(node)
{
	stopifnot(class(node)=="gdsn")
	r <- .C("gdsDeleteNode", node=as.integer(node), err=integer(1),
		NAOK=TRUE, PACKAGE="gdsfmt")
	if (r$err != 0)
		stop(lasterr.gds())
	return(invisible(NULL))
}





##################################################################################
# Attribute
##################################################################################

#############################################################
# To add an attribute to a GDS node
#
put.attr.gdsn <- function(node, name, val=NULL)
{
	stopifnot(class(node)=="gdsn")
	name <- as.character(name)
	if (name != "")
	{
		storage <- switch( storage.mode(val),
			"NULL" = 0, "integer" = 1, "double" = 2, "character" = 3, "logical" = 4,
			stop("Unsupported type!") )
		if (is.null(val)) val <- integer(0)
		r <- .C("gdsPutAttr", as.integer(node), name, as.integer(storage),
			val, err=integer(1), NAOK=TRUE, PACKAGE="gdsfmt")
		if (r$err != 0)
			stop(lasterr.gds())
		return(invisible(NULL))
	} else
		stop("'attribute' needs a name!")
}


#############################################################
# To get the attributes of a CoreArray GDS node
#
get.attr.gdsn <- function(node)
{
	stopifnot(class(node)=="gdsn")
	r <- .C("gdsAttrCnt", as.integer(node), Cnt=integer(1), err=integer(1),
		NAOK=TRUE, PACKAGE="gdsfmt")
	if (r$err != 0) stop(lasterr.gds())

	if (r$Cnt > 0)
	{
		rv <- vector("list", r$Cnt)
		r1 <- .C("gdsAttrType", as.integer(node), rtype=integer(r$Cnt), err=integer(1),
			NAOK=TRUE, PACKAGE="gdsfmt")
		if (r1$err != 0) stop(lasterr.gds())

		rn <- character(r$Cnt)
		for (i in 1:r$Cnt)
		{
			rt <- r1$rtype[i]
			r2 <- .C( "gdsGetAttr", as.integer(node), as.integer(i), rt,
				data=switch(rt+1, integer(0), integer(1), double(1), character(1), logical(1)),
				name=character(1), err=integer(1), NAOK=TRUE, PACKAGE="gdsfmt")
			if (r2$err == 0)
			{
				if (rt == 0) r2$data <- NULL
				if (!is.null(r2$data)) rv[[i]] <- r2$data
				rn[i] <- r2$name
			} else
				stop(lasterr.gds())
		}
		names(rv) <- rn
	} else
		rv <- NULL
	return(rv)
}


#############################################################
# To remove an attribute from a CoreArray GDS node
#
delete.attr.gdsn <- function(node, name)
{
	stopifnot(class(node)=="gdsn")
	r <- .C("gdsDeleteAttr", as.integer(node), as.character(name),
		err=integer(1), NAOK=TRUE, PACKAGE="gdsfmt")
	if (r$err != 0)
		stop(lasterr.gds())
	return(invisible(NULL))
}





##################################################################################
# Data Operations
##################################################################################

#############################################################
# To modify the data compression mode of data field
#
compression.gdsn <- function(node,
	compress=c("", "ZIP", "ZIP.fast", "ZIP.default", "ZIP.max") )
{
	stopifnot(class(node)=="gdsn")
	r <- .C("gdsObjCompress", as.integer(node), as.character(compress[1]),
		err=integer(1), NAOK=TRUE, PACKAGE="gdsfmt")
	if (r$err != 0)
		stop(lasterr.gds())
	return(node)
}


#############################################################
# Get into read mode of compression
#
readmode.gdsn <- function(node)
{
	stopifnot(class(node)=="gdsn")
	r <- .C("gdsObjPackClose", as.integer(node), err=integer(1),
		NAOK=TRUE, PACKAGE="gdsfmt")
	if (r$err != 0)
		stop(lasterr.gds())
	return(node)
}


#############################################################
# To set the new dimension of the data field for a GDS node
#
setdim.gdsn <- function(node, valdim)
{
	stopifnot(class(node)=="gdsn")
	r <- .C("gdsObjSetDim", as.integer(node), length(valdim),
		rev(as.integer(valdim)), err=integer(1), NAOK=TRUE, PACKAGE="gdsfmt")
	if (r$err != 0)
		stop(lasterr.gds())
	return(node)
}


#############################################################
# Append data to a specified variable
#
append.gdsn <- function(node, val, check=TRUE)
{
	stopifnot(class(node)=="gdsn")
	if (is.character(val))
	{
		val[is.na(val)] <- ""
	}
	r <- switch( storage.mode(val),
		"integer" =
			.C( "gdsObjAppend", as.integer(node), as.integer(1), as.integer(val),
					length(val), CntWarn=integer(1), err=integer(1), NAOK=TRUE, PACKAGE="gdsfmt"),
		"double" =
			.C( "gdsObjAppend", as.integer(node), as.integer(2), as.double(val),
					length(val), CntWarn=integer(1), err=integer(1), NAOK=TRUE, PACKAGE="gdsfmt"),
		"numeric" =
			.C( "gdsObjAppend", as.integer(node), as.integer(2), as.double(val),
					length(val), CntWarn=integer(1), err=integer(1), NAOK=TRUE, PACKAGE="gdsfmt"),
		"character" =
			.C( "gdsObjAppend", as.integer(node), as.integer(3), as.character(val),
					length(val), CntWarn=integer(1), err=integer(1), NAOK=TRUE, PACKAGE="gdsfmt"),
		"logical" =
			.C( "gdsObjAppend", as.integer(node), as.integer(1), as.integer(val),
					length(val), CntWarn=integer(1), err=integer(1), NAOK=TRUE, PACKAGE="gdsfmt"),
		stop("only support integer, double and character.") )
	if (r$err == 0)
	{
		if (r$CntWarn != 0)
			warning("No a complete sub-dataset.");
	} else {
		stop(lasterr.gds())
	}
	return(invisible(NULL))
}


#############################################################
# Read data field of a GDS node
#
read.gdsn <- function(node, start, count)
{
	stopifnot(class(node)=="gdsn")

	if (missing(start))
	{
		if (missing(count))
		{
			rvattr <- get.attr.gdsn(node)
			rvclass <- rvattr$R.class
			if (!is.null(rvclass))
			{
				cnt <- cnt.gdsn(node)
				r <- vector("list", cnt)
				if (cnt > 0)
				{
					for (i in 1:cnt)
					{
						n <- index.gdsn(node, i)
						if (!is.null(d <- read.gdsn(n)))
							r[[i]] <- d
						names(r)[i] <- name.gdsn(n)
					}
				}
				if (rvclass == "data.frame")
					r <- as.data.frame(r, stringsAsFactors=FALSE)
				if (!(rvclass %in% c("list", "data.frame")))
					class(r) <- rvclass
				return(r)
			}

			r <- .C("gdsxObjDesp", as.integer(node), cnt=as.integer(-1),
				rstart=as.integer(rep(1, 256)), rcount=as.integer(rep(-1, 256)),
				total=integer(1), rtype=integer(1), err=integer(1),
				NAOK=TRUE, PACKAGE="gdsfmt")

			rfactor <- ("R.factor" %in% names(rvattr))

		} else {
			stop("start is missing!")
		}
	} else if (missing(count)) {
			stop("count is missing")
		} else {

			stopifnot(length(start)==length(count))

			rvattr <- get.attr.gdsn(node)
			rfactor <- ("R.factor" %in% names(rvattr))

			r <- .C("gdsxObjDesp", as.integer(node), cnt=as.integer(length(start)),
				rstart=rev(as.integer(start)), rcount=rev(as.integer(count)),
				total=integer(1), rtype=integer(1), err=integer(1),
				NAOK=TRUE, PACKAGE="gdsfmt")
		}

	if (r$err == 0)
	{
		r <- .C("gdsObjReadData", as.integer(node), cnt=as.integer(r$cnt),
			as.integer(r$rstart), count=as.integer(r$rcount), as.integer(r$rtype),
			data=switch(r$rtype+1, integer(0), integer(r$total), double(r$total),
				character(r$total), logical(r$total)),
			err=integer(1), NAOK=TRUE, PACKAGE="gdsfmt")

		if (rfactor)
		{
			s <- r$data
			r$data <- factor(s)
			r$data[s==""] <- NA
		}

		if (r$err == 0)
		{
			if (r$cnt == 2)
			{
				if ((r$count[1] == 1) || (r$count[2] == 1))
					return(r$data)
				else
					return(array(r$data, dim=rev(r$count[1:r$cnt])))
			} else if (r$cnt <= 1)
				return(r$data)
			else
				return(array(r$data, dim=rev(r$count[1:r$cnt])))
		}
	}
	stop(lasterr.gds())
	return(invisible(NULL))
}


#############################################################
# Read data field of a GDS node
#
readex.gdsn <- function(node, sel=NULL)
{
	stopifnot(class(node)=="gdsn")
	stopifnot(is.null(sel) | is.logical(sel) | is.list(sel))

	if (!is.null(sel))
	{
		if (is.vector(sel) & !is.list(sel))
			sel <- list(d1=sel)
		DimSel <- sapply(sel, function(x) {
				if (!is.logical(x)) stop("The element of `sel' should be a logical variable.")
				length(x)
			})

		# check
		n <- objdesp.gdsn(node)
		if (length(n$dim) != length(DimSel))
			stop("The dimension of selection is not correct.")
		if (any(n$dim != DimSel))
			stop("The dimension of selection is not correct.")

		rvattr <- get.attr.gdsn(node)
		rfactor <- ("R.factor" %in% names(rvattr))
		DimCnt <- sapply(sel, function(x) sum(x, na.rm=TRUE))
		totalcnt <- prod(DimCnt)

		# check
		rd <- .C("gdsxObjType", as.integer(node), rtype=integer(1), err=integer(1),
			NAOK=TRUE, PACKAGE="gdsfmt")
		if (rd$err != 0) stop(lasterr.gds())

		# no selection
		if (totalcnt <= 0)
			return(switch(rd$rtype, integer(0), double(0), character(0), logical(0)))

		# reading
		r <- .C("gdsObjReadExData", as.integer(node),
			as.logical(unlist(sel)), as.integer(rd$rtype),
			data=switch(rd$rtype+1, integer(0), integer(totalcnt), double(totalcnt),
				character(totalcnt), logical(totalcnt)),
			err=integer(1), NAOK=TRUE, PACKAGE="gdsfmt")

		if (rfactor)
		{
			s <- r$data
			r$data <- factor(s)
			r$data[s==""] <- NA
		}

		if (r$err == 0)
		{
			if (length(DimCnt) <= 1)
				return(r$data)
			else
				return(array(r$data, dim=DimCnt))
		} else {
			stop(lasterr.gds())
		}
	} else {
		return(read.gdsn(node))
	}
}


#############################################################
# Apply functions over array margins of a GDS node
#
apply.gdsn <- function(node, margin=2, row.sel=NULL, col.sel=NULL,
	buf.size=1024, simplify=TRUE, FUN, ...)
{
	# check
	stopifnot(class(node)=="gdsn")
	stopifnot(margin %in% c(1, 2))
	stopifnot(is.null(row.sel) | is.logical(row.sel))
	stopifnot(is.null(col.sel) | is.logical(col.sel))
	stopifnot(is.logical(simplify))

	desp <- objdesp.gdsn(node)
	if (!desp$is.array)
		stop("`node' should be array-type.")
	if (length(desp$dim) != 2)
		stop("`node' should be a matrix")
	if (is.logical(row.sel))
	{
		if (length(row.sel) != desp$dim[1])
			stop("Invalid length of `row.sel'.")
		if (sum(row.sel, na.rm=TRUE) <= 0)
			return(NULL)
	}
	if (is.logical(col.sel))
	{
		if (length(col.sel) != desp$dim[2])
			stop("Invalid length of `col.sel'.")
		if (sum(col.sel, na.rm=TRUE) <= 0)
			return(NULL)
	}

	# initialize ...
	rd <- .C("gdsxObjType", as.integer(node), rtype=integer(1), err=integer(1),
		NAOK=TRUE, PACKAGE="gdsfmt")
	if (rd$err != 0) stop(lasterr.gds())
	FUN <- match.fun(FUN)

	# for-loop
	if (margin == 1)
	{
		# for each row

		if (is.logical(row.sel))
		{
			rowidx <- which(row.sel)
			totalcnt <- length(rowidx)
		} else {
			rowidx <- seq_len(desp$dim[1])
			totalcnt <- desp$dim[1]
		}
		if (is.logical(col.sel))
		{
			eachcnt <- sum(col.sel, na.rm=TRUE)
		} else {
			eachcnt <- desp$dim[2]
			col.sel <- rep(TRUE, eachcnt)
		}

		ans <- vector(mode="list", length=totalcnt)
		rsel <- .C("gdsLongBool2CBOOL", col.sel, CBOOL=integer(length(col.sel)),
			length(col.sel), NAOK=TRUE, PACKAGE="gdsfmt")

		# for-loop
		idx <- 1
		while (idx <= length(rowidx))
		{
			cnt <- length(rowidx) - idx + 1
			if (cnt > buf.size) cnt <- buf.size

			r <- .C("gdsApplyRow", as.integer(node), as.integer(idx),
				as.integer(rowidx), as.integer(cnt), rsel$CBOOL, as.integer(rd$rtype),
				data=switch(rd$rtype+1, integer(0),
					matrix(as.integer(0), nrow=cnt, ncol=eachcnt),
					matrix(as.double(0), nrow=cnt, ncol=eachcnt),
					matrix(as.character(0), nrow=cnt, ncol=eachcnt),
					matrix(as.logical(0), nrow=cnt, ncol=eachcnt)),
				err=integer(1), NAOK=TRUE, PACKAGE="gdsfmt")
			if (r$err != 0) stop(lasterr.gds())

			# call
			for (i in seq.int(idx, idx+cnt-1))
				ans[[i]] <- FUN(r$data[i-idx+1, ], ...)

			idx <- idx + cnt
		}
	} else {
		# for each column

		if (is.logical(col.sel))
		{
			colidx <- which(col.sel)
			totalcnt <- length(colidx)
		} else {
			colidx <- seq_len(desp$dim[2])
			totalcnt <- desp$dim[2]
		}
		if (is.logical(row.sel))
		{
			eachcnt <- sum(row.sel, na.rm=TRUE)
		} else {
			eachcnt <- desp$dim[1]
			row.sel <- rep(TRUE, eachcnt)
		}

		ans <- vector(mode="list", length=totalcnt)
		rsel <- .C("gdsLongBool2CBOOL", row.sel, CBOOL=integer(length(row.sel)),
			length(row.sel), NAOK=TRUE, PACKAGE="gdsfmt")

		# for-loop
		idx <- 1
		while (idx <= length(colidx))
		{
			cnt <- length(colidx) - idx + 1
			if (cnt > buf.size) cnt <- buf.size

			r <- .C("gdsApplyCol", as.integer(node), as.integer(idx),
				as.integer(colidx), as.integer(cnt), rsel$CBOOL, as.integer(rd$rtype),
				data=switch(rd$rtype+1, integer(0),
					matrix(as.integer(0), nrow=eachcnt, ncol=cnt),
					matrix(as.double(0), nrow=eachcnt, ncol=cnt),
					matrix(as.character(0), nrow=eachcnt, ncol=cnt),
					matrix(as.logical(0), nrow=eachcnt, ncol=cnt)),
				err=integer(1), NAOK=TRUE, PACKAGE="gdsfmt")
			if (r$err != 0) stop(lasterr.gds())

			# call
			for (i in seq.int(idx, idx+cnt-1))
				ans[[i]] <- FUN(r$data[, i-idx+1], ...)

			idx <- idx + cnt
		}
	}

	# return
	if ((length(ans) > 0) && simplify)
		simplify2array(ans, higher=TRUE)
	else
		ans
}


#############################################################
# Apply functions over array margins of a list of GDS nodes
#
applylt.gdsn <- function(nodes, margin=2, row.sel=NULL, col.sel=NULL,
	buf.size=1024, simplify=TRUE, FUN, ...)
{
	# check
	stopifnot(is.list(nodes))
	stopifnot(length(nodes) > 0)
	stopifnot(margin %in% c(1, 2))
	stopifnot(is.logical(simplify))

	if (margin == 1)
	{
		# enumerate each row

		stopifnot(is.null(row.sel) | is.logical(row.sel))
		stopifnot(is.null(col.sel) | is.list(col.sel))
		if (is.list(col.sel))
			stopifnot(length(col.sel) == length(nodes))

		if (is.logical(row.sel))
		{
			num.row <- sum(row.sel, na.rm=TRUE)
		} else {
			desp <- objdesp.gdsn(nodes[[1]])
			num.row <- desp$dim[1]
		}
		if (num.row <= 0) return(NULL)

		for (i in 1:length(nodes))
		{
			if (class(nodes[[i]]) != "gdsn")
				stop(sprintf("`nodes[%d]' should be a `gdsn' object!", i))

			desp <- objdesp.gdsn(nodes[[i]])
			if (!desp$is.array)
				stop(sprintf("`nodes[%d]' should be array-type!", i))
			if (!(length(desp$dim) %in% c(1,2)))
				stop(sprintf("`nodes[%d]' should be a matrix or vector!", i))
			if (is.logical(row.sel))
			{
				if (length(row.sel) != desp$dim[1])
					stop(sprintf("The row dimension of `nodes[%d]' is invalid!", i))
			} else {
				if (num.row != desp$dim[1])
					stop(sprintf("The row dimension of `nodes[%d]' is invalid!", i))
			}
			if (is.list(col.sel))
			{
				sel <- col.sel[[i]]
				if (!is.null(sel) & !is.logical(sel))
					stop(sprintf("`col.sel[%d]' should be NULL or a logical vector!", i))
				if (length(desp$dim) > 1)
				{
					if (is.logical(sel) & length(sel)!= desp$dim[2])
						stop(sprintf("Invalid length of `col.sel[%d]'!", i))
				}
			}
		}

	} else {
		# enumerate each column

		stopifnot(is.null(col.sel) | is.logical(col.sel))
		stopifnot(is.null(row.sel) | is.list(row.sel))
		if (is.list(row.sel))
			stopifnot(length(row.sel) == length(nodes))

		if (is.logical(col.sel))
		{
			num.col <- sum(col.sel, na.rm=TRUE)
		} else {
			desp <- objdesp.gdsn(nodes[[1]])
			if (length(desp$dim) > 1)
			{
				num.col <- desp$dim[2]
			} else {
				num.col <- desp$dim[1]
			}
		}
		if (num.col <= 0) return(NULL)

		for (i in 1:length(nodes))
		{
			if (class(nodes[[i]]) != "gdsn")
				stop(sprintf("`nodes[%d]' should be a `gdsn' object!", i))

			desp <- objdesp.gdsn(nodes[[i]])
			if (!desp$is.array)
				stop(sprintf("`nodes[%d]' should be array-type!", i))
			if (!(length(desp$dim) %in% c(1,2)))
				stop(sprintf("`nodes[%d]' should be a matrix or vector!", i))
			if (is.logical(col.sel))
			{
				if (length(desp$dim) > 1)
				{
					n.tmp <- desp$dim[2]
				} else {
					n.tmp <- desp$dim[1]
				}
				if (length(col.sel) != n.tmp)
					stop(sprintf("The column dimension of `nodes[%d]' is invalid!", i))
			} else {
				if (length(desp$dim) > 1)
				{
					n.tmp <- desp$dim[2]
				} else {
					n.tmp <- desp$dim[1]
				}
				if (num.col != n.tmp)
					stop(sprintf("The column dimension of `nodes[%d]' is invalid!", i))
			}
			if (is.list(row.sel))
			{
				sel <- row.sel[[i]]
				if (!is.null(sel) & !is.logical(sel))
					stop(sprintf("`row.sel[%d]' should be NULL or a logical vector!", i))
				if (length(desp$dim) > 1)
				{
					if (is.logical(sel) & length(sel)!= desp$dim[1])
						stop(sprintf("Invalid length of `row.sel[%d]'!", i))
				}
			}
		}
	}

	# initialize function
	FUN <- match.fun(FUN)

	# initialize type
	rtype <- integer(length(nodes))
	for (i in 1:length(nodes))
	{
		r <- .C("gdsxObjType", as.integer(nodes[[i]]), rtype=integer(1), err=integer(1),
			NAOK=TRUE, PACKAGE="gdsfmt")
		if (r$err != 0) stop(lasterr.gds())
		rtype[i] <- r$rtype
	}

	# for-loop
	if (margin == 1)
	{
		# for each row

		if (is.logical(row.sel))
		{
			rowidx <- which(row.sel)
		} else {
			rowidx <- seq_len(num.row)
		}
		ans <- vector(mode="list", length=num.row)

		eachcnt <- integer(length(nodes))
		sel.CBOOL <- vector("list", length(nodes))
		for (i in 1:length(nodes))
		{
			desp <- objdesp.gdsn(nodes[[i]])
			sel <- col.sel[[i]]
			if (is.logical(sel))
			{
				eachcnt[i] <- sum(sel, na.rm=TRUE)
			} else {
				if (length(desp$dim) > 1)
				{
					eachcnt[i] <- desp$dim[2]
				} else {
					eachcnt[i] <- 1
				}
				sel <- rep(TRUE, eachcnt[i])
			}
			r <- .C("gdsLongBool2CBOOL", sel, CBOOL=integer(length(sel)),
				length(sel), NAOK=TRUE, PACKAGE="gdsfmt")
			sel.CBOOL[[i]] <- r$CBOOL
		}

		dat.list <- vector("list", length(nodes))
		dat <- vector("list", length(nodes))
		names(dat) <- names(nodes)

		# for-loop
		idx <- 1
		while (idx <= length(rowidx))
		{
			cnt <- length(rowidx) - idx + 1
			if (cnt > buf.size) cnt <- buf.size

			for (i in 1:length(nodes))
			{
				r <- .C("gdsApplyRow", as.integer(nodes[[i]]), as.integer(idx),
					as.integer(rowidx), as.integer(cnt), sel.CBOOL[[i]], rtype[i],
					data=switch(rtype[i]+1, integer(0),
						matrix(as.integer(0), nrow=cnt, ncol=eachcnt[i]),
						matrix(as.double(0), nrow=cnt, ncol=eachcnt[i]),
						matrix(as.character(0), nrow=cnt, ncol=eachcnt[i]),
						matrix(as.logical(0), nrow=cnt, ncol=eachcnt[i])),
					err=integer(1), NAOK=TRUE, PACKAGE="gdsfmt")
				if (r$err != 0) stop(lasterr.gds())
				dat.list[[i]] <- r$data
			}

			# call
			for (i in seq.int(idx, idx+cnt-1))
			{
				for (k in 1:length(nodes))
					dat[[k]] <- dat.list[[k]][i-idx+1, ]
				ans[[i]] <- FUN(dat, ...)
			}

			idx <- idx + cnt
		}
	} else {
		# for each column

		if (is.logical(col.sel))
		{
			colidx <- which(col.sel)
		} else {
			colidx <- seq_len(num.col)
		}
		ans <- vector(mode="list", length=num.col)

		eachcnt <- integer(length(nodes))
		sel.CBOOL <- vector("list", length(nodes))
		for (i in 1:length(nodes))
		{
			desp <- objdesp.gdsn(nodes[[i]])
			sel <- row.sel[[i]]
			if (is.logical(sel))
			{
				eachcnt[i] <- sum(sel, na.rm=TRUE)
			} else {
				if (length(desp$dim) > 1)
				{
					eachcnt[i] <- desp$dim[1]
				} else {
					eachcnt[i] <- 1
				}
				sel <- rep(TRUE, eachcnt[i])
			}
			r <- .C("gdsLongBool2CBOOL", sel, CBOOL=integer(length(sel)),
				length(sel), NAOK=TRUE, PACKAGE="gdsfmt")
			sel.CBOOL[[i]] <- r$CBOOL
		}

		dat.list <- vector("list", length(nodes))
		dat <- vector("list", length(nodes))
		names(dat) <- names(nodes)

		# for-loop
		idx <- 1
		while (idx <= length(colidx))
		{
			cnt <- length(colidx) - idx + 1
			if (cnt > buf.size) cnt <- buf.size

			for (i in 1:length(nodes))
			{
				r <- .C("gdsApplyCol", as.integer(nodes[[i]]), as.integer(idx),
					as.integer(colidx), as.integer(cnt), sel.CBOOL[[i]], rtype[i],
					data=switch(rtype[i]+1, integer(0),
						matrix(as.integer(0), nrow=eachcnt[i], ncol=cnt),
						matrix(as.double(0), nrow=eachcnt[i], ncol=cnt),
						matrix(as.character(0), nrow=eachcnt[i], ncol=cnt),
						matrix(as.logical(0), nrow=eachcnt[i], ncol=cnt)),
					err=integer(1), NAOK=TRUE, PACKAGE="gdsfmt")
				if (r$err != 0) stop(lasterr.gds())
				dat.list[[i]] <- r$data
			}

			# call
			for (i in seq.int(idx, idx+cnt-1))
			{
				for (k in 1:length(nodes))
					dat[[k]] <- dat.list[[k]][, i-idx+1]
				ans[[i]] <- FUN(dat, ...)
			}

			idx <- idx + cnt
		}
	}

	# return
	if ((length(ans) > 0) && simplify)
		simplify2array(ans, higher=TRUE)
	else
		ans
}


#############################################################
# Apply functions over array margins of a GDS node via clusters
#
clusterApply.gdsn <- function(cl, gds.fn, node, margin=2,
	row.sel=NULL, col.sel=NULL, buf.size=1024, simplify=TRUE, FUN, ...)
{
	# library
    if (!require(parallel))
    {
	    if (!require(snow))
			stop("the `parallel' or `snow' package is needed.")
    }

	# check
	stopifnot(inherits(cl, "cluster"))
	stopifnot(is.character(gds.fn))
	stopifnot(is.character(node) | is.numeric(node))
	stopifnot(margin %in% c(1, 2))
	stopifnot(is.null(row.sel) | is.logical(row.sel))
	stopifnot(is.null(col.sel) | is.logical(col.sel))
	stopifnot(is.logical(simplify))

	gds <- openfn.gds(gds.fn)
	ifopen <- TRUE
	on.exit({ if (ifopen) closefn.gds(gds) })

	nd <- index.gdsn(gds, node, TRUE)
	if (is.null(nd))
	{
		closefn.gds(gds)
		stop(sprintf("There is no node \"%s\" in the specified gds file.",
			paste(node, collapse="/")))
	}
	desp <- objdesp.gdsn(nd)

	if (!desp$is.array)
		stop("`node' should be array-type.")
	if (length(desp$dim) != 2)
		stop("`node' should be a matrix")
	if (is.logical(row.sel))
	{
		if (length(row.sel) != desp$dim[1])
			stop("Invalid length of `row.sel'.")
		if (sum(row.sel, na.rm=TRUE) <= 0)
			return(NULL)
	}
	if (is.logical(col.sel))
	{
		if (length(col.sel) != desp$dim[2])
			stop("Invalid length of `col.sel'.")
		if (sum(col.sel, na.rm=TRUE) <= 0)
			return(NULL)
	}

	# initialize ...
	if (margin == 1)
	{
		# enumerate each row
		if (is.logical(row.sel))
		{
			sel <- which(row.sel)
		} else {
			sel <- seq_len(desp$dim[1])
		}
		n <- desp$dim[1]
	} else {
		# enumerate each column
		if (is.logical(col.sel))
		{
			sel <- which(col.sel)
		} else {
			sel <- seq_len(desp$dim[2])
		}
		n <- desp$dim[2]
	}

	if (length(cl) > 1)
	{
		ifopen <- FALSE
		closefn.gds(gds)

		clseq <- splitIndices(length(sel), length(cl))
		for (i in 1:length(cl))
		{
			val <- rep(FALSE, n)
			val[ sel[ clseq[[i]] ] ] <- TRUE
			clseq[[i]] <- val
		}

		if (margin == 1)
		{
			# enumerate each row
			ans <- clusterApply(cl, clseq, fun =
				function(sel, gds.fn, node, col.sel, buf.size, FUN, ...) {

					library(gdsfmt)
					# open the file
					gds <- openfn.gds(gds.fn)
					nd <- index.gdsn(gds, node)
					# apply
					rv <- apply.gdsn(node=nd, margin=1, row.sel=sel, col.sel=col.sel,
						buf.size=buf.size, simplify=FALSE, FUN=FUN, ...)
					# close
					closefn.gds(gds)
					rv

				}, gds.fn=gds.fn, node=node,
					col.sel=col.sel, buf.size=buf.size, FUN=FUN, ...)
		} else {
			# enumerate each column
			ans <- clusterApply(cl, clseq, fun =
				function(sel, gds.fn, node, row.sel, buf.size, FUN, ...) {

					library(gdsfmt)
					# open the file
					gds <- openfn.gds(gds.fn)
					nd <- index.gdsn(gds, node)
					# apply
					rv <- apply.gdsn(node=nd, margin=2, row.sel=row.sel, col.sel=sel,
						buf.size=buf.size, simplify=FALSE, FUN=FUN, ...)
					# close
					closefn.gds(gds)
					rv

				}, gds.fn=gds.fn, node=node,
					row.sel=row.sel, buf.size=buf.size, FUN=FUN, ...)
		}

		# return
		ans <- unlist(ans, recursive=FALSE, use.names=FALSE)
		if ((length(ans) > 0) && simplify)
			simplify2array(ans, higher=TRUE)
		else
			ans
	} else{
		apply.gdsn(nd, margin, row.sel, col.sel, buf.size, simplify, FUN, ...)
	}
}


#############################################################
# Apply functions over array margins of a list of GDS nodes
#
clusterApplylt.gdsn <- function(cl, gds.fn, nodes, margin=2,
	row.sel=NULL, col.sel=NULL, buf.size=1024, simplify=TRUE, FUN, ...)
{
	# library
    if (!require(parallel))
    {
	    if (!require(snow))
			stop("the `parallel' or `snow' package is needed.")
    }

	# check
	stopifnot(inherits(cl, "cluster"))
	stopifnot(is.character(gds.fn))
	stopifnot(is.list(nodes))
	stopifnot(length(nodes) > 0)
	stopifnot(margin %in% c(1, 2))
	stopifnot(is.logical(simplify))

	ifopen <- TRUE
	gds <- openfn.gds(gds.fn)
	on.exit({ if (ifopen) closefn.gds(gds) })

	nd.nodes <- vector("list", length(nodes))
	for (i in 1:length(nodes))
	{
		v <- index.gdsn(gds, nodes[[i]], silent=TRUE)
		nd.nodes[[i]] <- v
		if (is.null(v))
		{
			stop(sprintf("There is no node \"%s\" in the specified gds file.",
				paste(nodes[[i]], collapse="/")))
		}
	}

	if (margin == 1)
	{
		# enumerate each row

		stopifnot(is.null(row.sel) | is.logical(row.sel))
		stopifnot(is.null(col.sel) | is.list(col.sel))
		if (is.list(col.sel))
			stopifnot(length(col.sel) == length(nodes))

		if (is.logical(row.sel))
		{
			num.row <- sum(row.sel, na.rm=TRUE)
			xsel <- which(row.sel)
			n.total <- length(row.sel)
		} else {
			desp <- objdesp.gdsn(nd.nodes[[1]])
			num.row <- desp$dim[1]
			xsel <- seq_len(desp$dim[1])
			n.total <- desp$dim[1]
		}
		if (num.row <= 0) return(NULL)

		for (i in 1:length(nodes))
		{
			desp <- objdesp.gdsn(nd.nodes[[i]])
			if (!desp$is.array)
				stop(sprintf("`nodes[%d]' should be array-type!", i))
			if (!(length(desp$dim) %in% c(1,2)))
				stop(sprintf("`nodes[%d]' should be a matrix or vector!", i))
			if (is.logical(row.sel))
			{
				if (length(row.sel) != desp$dim[1])
					stop(sprintf("The row dimension of `nodes[%d]' is invalid!", i))
			} else {
				if (num.row != desp$dim[1])
					stop(sprintf("The row dimension of `nodes[%d]' is invalid!", i))
			}
			if (is.list(col.sel))
			{
				sel <- col.sel[[i]]
				if (!is.null(sel) & !is.logical(sel))
					stop(sprintf("`col.sel[%d]' should be NULL or a logical vector!", i))
				if (length(desp$dim) > 1)
				{
					if (is.logical(sel) & length(sel)!= desp$dim[2])
						stop(sprintf("Invalid length of `col.sel[%d]'!", i))
				}
			}
		}

	} else {
		# enumerate each column

		stopifnot(is.null(col.sel) | is.logical(col.sel))
		stopifnot(is.null(row.sel) | is.list(row.sel))
		if (is.list(row.sel))
			stopifnot(length(row.sel) == length(nodes))

		if (is.logical(col.sel))
		{
			num.col <- sum(col.sel, na.rm=TRUE)
			xsel <- which(col.sel)
			n.total <- length(col.sel)
		} else {
			desp <- objdesp.gdsn(nd.nodes[[1]])
			if (length(desp$dim) > 1)
			{
				num.col <- desp$dim[2]
			} else {
				num.col <- desp$dim[1]
			}
			xsel <- seq_len(num.col)
			n.total <- num.col
		}
		if (num.col <= 0) return(NULL)

		for (i in 1:length(nodes))
		{
			desp <- objdesp.gdsn(nd.nodes[[i]])
			if (!desp$is.array)
				stop(sprintf("`nodes[%d]' should be array-type!", i))
			if (!(length(desp$dim) %in% c(1,2)))
				stop(sprintf("`nodes[%d]' should be a matrix or vector!", i))
			if (is.logical(col.sel))
			{
				if (length(desp$dim) > 1)
				{
					n.tmp <- desp$dim[2]
				} else {
					n.tmp <- desp$dim[1]
				}
				if (length(col.sel) != n.tmp)
					stop(sprintf("The column dimension of `nodes[%d]' is invalid!", i))
			} else {
				if (length(desp$dim) > 1)
				{
					n.tmp <- desp$dim[2]
				} else {
					n.tmp <- desp$dim[1]
				}
				if (num.col != n.tmp)
					stop(sprintf("The column dimension of `nodes[%d]' is invalid!", i))
			}
			if (is.list(row.sel))
			{
				sel <- row.sel[[i]]
				if (!is.null(sel) & !is.logical(sel))
					stop(sprintf("`row.sel[%d]' should be NULL or a logical vector!", i))
				if (length(desp$dim) > 1)
				{
					if (is.logical(sel) & length(sel)!= desp$dim[1])
						stop(sprintf("Invalid length of `row.sel[%d]'!", i))
				}
			}
		}
	}

	# run ...
	if (length(cl) > 1)
	{
		ifopen <- FALSE
		closefn.gds(gds)

		clseq <- splitIndices(length(xsel), length(cl))
		for (i in 1:length(cl))
		{
			val <- rep(FALSE, n.total)
			val[ xsel[ clseq[[i]] ] ] <- TRUE
			clseq[[i]] <- val
		}

		if (margin == 1)
		{
			# enumerate each row
			ans <- clusterApply(cl, clseq, fun =
				function(sel, gds.fn, nodes, col.sel, buf.size, FUN, ...) {

					library(gdsfmt)
					# open the file
					gds <- openfn.gds(gds.fn)
					nd.nodes <- vector("list", length(nodes))
					for (i in 1:length(nodes))
						nd.nodes[[i]] <- index.gdsn(gds, nodes[[i]])
					# apply
					rv <- applylt.gdsn(nd.nodes, margin=1, row.sel=sel, col.sel=col.sel,
						buf.size=buf.size, simplify=FALSE, FUN=FUN, ...)
					# close
					closefn.gds(gds)
					rv

				}, gds.fn=gds.fn, nodes=nodes,
					col.sel=col.sel, buf.size=buf.size, FUN=FUN, ...)
		} else {
			# enumerate each column
			ans <- clusterApply(cl, clseq, fun =
				function(sel, gds.fn, nodes, row.sel, buf.size, FUN, ...) {

					library(gdsfmt)
					# open the file
					gds <- openfn.gds(gds.fn)
					nd.nodes <- vector("list", length(nodes))
					for (i in 1:length(nodes))
						nd.nodes[[i]] <- index.gdsn(gds, nodes[[i]])
					# apply
					rv <- applylt.gdsn(nd.nodes, margin=2, row.sel=row.sel, col.sel=sel,
						buf.size=buf.size, simplify=FALSE, FUN=FUN, ...)
					# close
					closefn.gds(gds)
					rv

				}, gds.fn=gds.fn, nodes=nodes,
					row.sel=row.sel, buf.size=buf.size, FUN=FUN, ...)
		}

		# return
		ans <- unlist(ans, recursive=FALSE, use.names=FALSE)
		if ((length(ans) > 0) && simplify)
			simplify2array(ans, higher=TRUE)
		else
			ans
	} else{
		applylt.gdsn(nd.nodes, margin, row.sel, col.sel,
			buf.size, simplify, FUN, ...)
	}
}


#############################################################
# Write data to a GDS node
#
write.gdsn <- function(node, val, start, count)
{
	stopifnot(class(node)=="gdsn")
	stopifnot(!missing(val))

	if (missing(start))
	{
		if (missing(count))
		{
			if (is.character(val)) val[is.na(val)] <- ""

			rt <- switch( storage.mode(val),
				"integer" = 1, "double" = 2, "character" = 3, "logical" = 1,
				stop("only support integer, double and character.") )
			val <- as.array(val); dimval <- dim(val)
			r <- .C("gdsObjWriteAll", as.integer(node), length(dimval),
				rev(as.integer(dimval)), as.integer(rt),
				switch(rt, as.integer(val), as.double(val), as.character(val)),
				err=integer(1), NAOK=TRUE, PACKAGE="gdsfmt")
			if (r$err != 0)
				stop(lasterr.gds())
			return(invisible(NULL))

		} else {
			stop("start is missing!")
		}
	} else if (missing(count)) {
		stop("count is missing")
	} else {

		stopifnot(length(start)==length(count))
		r <- .C("gdsxObjDesp", as.integer(node), cnt=as.integer(length(start)),
			rstart=as.integer(rev(start)), rcount=as.integer(rev(count)),
			total=integer(1), rtype=integer(1), err=integer(1), NAOK=TRUE,
			PACKAGE="gdsfmt")

		if (r$err == 0)
		{
			if (r$total != length(val))
				stop(paste("the length of val ", length(val),
					" is not equal to count(", r$total, ").", sep=""))
			if (is.character(val)) val[is.na(val)] <- ""

			r <- .C("gdsObjWriteData", as.integer(node), as.integer(r$cnt),
				as.integer(r$rstart), as.integer(r$rcount), as.integer(r$rtype),
				switch(r$rtype, as.integer(val), as.double(val),
					as.character(val), as.integer(val)),
				err=integer(1), NAOK=TRUE, PACKAGE="gdsfmt")
			if (r$err != 0)
				stop(lasterr.gds())

		} else
			stop(lasterr.gds())
	}
	return(invisible(NULL))
}


#############################################################
# To get a file from a stream container
#
getfile.gdsn <- function(node, out.filename)
{
	stopifnot(class(node)=="gdsn")
	stopifnot(is.character(out.filename))

	r <- .C("gdsGetFile", as.integer(node), out.filename, err=integer(1),
		NAOK=TRUE, PACKAGE="gdsfmt")
	if (r$err != 0) stop(lasterr.gds())

	return(invisible(NULL))
}





##################################################################################
# Error function
##################################################################################

#############################################################
# Return the last error
#
lasterr.gds <- function()
{
	r <- .C("gdsLastErrGDS", s=character(1), NAOK=TRUE, PACKAGE="gdsfmt")
	r$s
}





##################################################################################
# R Generic functions
##################################################################################

print.gdsclass <- function(x, ...)
{
	enum <- function(node, space, level)
	{
		n <- objdesp.gdsn(node)
		cnt <- cnt.gdsn(node)
		cat(space, "+ ", name.gdsn(node), "	[ ", n$desp, " ", sep="")
		cat(n$dim, sep="x")
		if (n$compress!="") cat("", n$compress)
		if (is.finite(n$cpratio))
			cat(sprintf("(%0.2f%%)", 100*n$cpratio))
		if (length(get.attr.gdsn(node)) > 0)
			cat(" ] *\n")
		else
			cat(" ]\n")

		r <- .C("gdsNodeEnumPtr", as.integer(node), id=integer(cnt*2),
			err=integer(1), NAOK=TRUE, PACKAGE="gdsfmt")
		if ((r$err == 0) & (cnt > 0))
		{
			for (i in 0:(cnt-1))
			{
				m <- r$id[(2*i+1):(2*i+2)]
				class(m) <- "gdsn"
				if (level==1)
					s <- paste("|--", space, sep="")
				else
					s <- paste("|  ", space, sep="")
				enum(m, s, level+1)
			}
		}
	}

	stopifnot(class(x)=="gdsclass")
	rv <- .C("gdsFileValid", x$id, valid=integer(1), NAOK=TRUE, PACKAGE="gdsfmt")
	if (rv$valid == 0)
	{
		cat("The GDS file has been closed.\n")
	} else {
		cat("file name: ", x$filename, "\n\n", sep="");
		enum(x$root, "", 1)
	}
}

print.gdsn <- function(x, expand=TRUE, ...)
{
	enum <- function(node, space, level, expand, fullname)
	{
		n <- objdesp.gdsn(node)
		cnt <- cnt.gdsn(node)
		cat(space, "+ ", name.gdsn(node, fullname), "	[ ", n$desp, " ", sep="")
		cat(n$dim, sep="x")
		if (n$compress != "") cat("", n$compress)
		if (is.finite(n$cpratio))
			cat(sprintf("(%0.2f%%)", 100*n$cpratio))
		if (length(get.attr.gdsn(node)) > 0)
			cat(" ] *\n")
		else
			cat(" ]\n")

		if (expand)
		{
			r <- .C("gdsNodeEnumPtr", as.integer(node), id=integer(cnt*2),
				err=integer(1), NAOK=TRUE, PACKAGE="gdsfmt")
			if ((r$err == 0) & (cnt > 0))
			{
				for (i in 0:(cnt-1))
				{
					m <- r$id[(2*i+1):(2*i+2)]
					class(m) <- "gdsn"
					if (level==1)
						s <- paste("|--", space, sep="")
					else
						s <- paste("|  ", space, sep="")
					enum(m, s, level+1, TRUE, FALSE)
				}
			}
		}
	}

	stopifnot(class(x)=="gdsn")
	rv <- .C("gdsNodeValid", as.integer(x), valid=integer(1), NAOK=TRUE, PACKAGE="gdsfmt")
	if (rv$valid == 0)
	{
		cat("The GDS file has been closed.\n")
	} else {
		enum(x, "", 1, expand, TRUE)
	}
}




##################################################################################
# R internal functions
##################################################################################

.onAttach <- function(lib, pkg)
{
	library.dynam("gdsfmt", pkg, lib)
	TRUE
}
