% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/geom-table.R
\name{geom_table}
\alias{geom_table}
\alias{geom_table_npc}
\title{Inset tables}
\usage{
geom_table(
  mapping = NULL,
  data = NULL,
  stat = "identity",
  position = "identity",
  ...,
  table.theme = NULL,
  table.rownames = FALSE,
  table.colnames = TRUE,
  table.hjust = 0.5,
  parse = FALSE,
  na.rm = FALSE,
  show.legend = FALSE,
  inherit.aes = FALSE
)

geom_table_npc(
  mapping = NULL,
  data = NULL,
  stat = "identity",
  position = "identity",
  ...,
  table.theme = NULL,
  table.rownames = FALSE,
  table.colnames = TRUE,
  table.hjust = 0.5,
  parse = FALSE,
  na.rm = FALSE,
  show.legend = FALSE,
  inherit.aes = FALSE
)
}
\arguments{
\item{mapping}{The aesthetic mapping, usually constructed with
\code{\link[ggplot2]{aes}} or \code{\link[ggplot2]{aes_}}. Only needs
to be set at the layer level if you are overriding the plot defaults.}

\item{data}{A layer specific data set - only needed if you want to override
the plot defaults.}

\item{stat}{The statistical transformation to use on the data for this layer,
as a string.}

\item{position}{Position adjustment, either as a string, or the result of a
call to a position adjustment function.}

\item{...}{other arguments passed on to \code{\link[ggplot2]{layer}}. This
can include aesthetics whose values you want to set, not map. See
\code{\link[ggplot2]{layer}} for more details.}

\item{table.theme}{NULL, list or function A gridExtra ttheme defintion, or
a constructor for a ttheme or NULL for default.}

\item{table.rownames, table.colnames}{logical flag to enable or disable
printing of row names and column names.}

\item{table.hjust}{numeric Horizontal justification for the core and column
headings of the table.}

\item{parse}{If TRUE, the labels will be parsed into expressions and
displayed as described in \code{?plotmath}.}

\item{na.rm}{If \code{FALSE} (the default), removes missing values with a
warning.  If \code{TRUE} silently removes missing values.}

\item{show.legend}{logical. Should this layer be included in the legends?
\code{NA}, the default, includes if any aesthetics are mapped. \code{FALSE}
never includes, and \code{TRUE} always includes.}

\item{inherit.aes}{If \code{FALSE}, overrides the default aesthetics, rather
than combining with them. This is most useful for helper functions that
define both data and aesthetics and shouldn't inherit behaviour from the
default plot specification, e.g. \code{\link[ggplot2]{borders}}.}
}
\description{
\code{geom_table} adds a textual table directly to the ggplot using syntax
similar to that of \code{\link[ggplot2]{geom_label}} while
\code{geom_table_npc} is similar to \code{geom_label_npc} in that \code{x}
and \code{y} coordinates are given in npc units. In most respects they
behave as any other ggplot geometry: a layer con contain multiple tables
and faceting works as usual.
}
\details{
These geoms work only with tibbles as \code{data}, as they expects a
  list of data frames or tibbles ("tb" objects) to be mapped to the
  \code{label} aesthetic. Aesthetics mappings in the inset plot are
  independent of those in the base plot.

  In the case of \code{geom_table()}, \code{x} and \code{y} aesthetics
  determine the position of the whole inset table, similarly to that of a
  text label, justification is interpreted as indicating the position of the
  table with respect to the $x$ and $y$ coordinates in the data, and
  \code{angle} is used to rotate the table as a whole.

  In the case of \code{geom_table_npc()}, \code{npcx} and \code{npcy}
  aesthetics determine the position of the whole inset table, similarly to
  that of a text label, justification is interpreted as indicating the
  position of the table with respect to the $x$ and $y$ coordinates in "npc"
  units, and \code{angle} is used to rotate the table as a whole.

  The "width" and "height" of an inset as for a text element are
  0, so stacking and dodging inset plots will not work by default, and axis
  limits are not automatically expanded to include all inset plots.
  Obviously, insets do have height and width, but they are physical units,
  not data units. The amount of space they occupy on the main plot is not
  constant in data units of the base plot: when you modify scale limits,
  inset plots stay the same size relative to the physical size of the base
  plot.
}
\note{
As all geoms, \code{geom_table()} and \code{geom_table_npc()} add a
  layer to a plot, and behave as expected in the grammar of graphics: ggplot
  themes do not affect how layers are rendered. The formatting of the inset
  table is done according to the the argument passed to \code{table.theme}.

  As the table is built with function gridExtra::gtable(), for formatting
  details, please, consult \code{\link[gridExtra]{tableGrob}}. If the
  argument passed to \code{table.theme} is a constructor function, the values
  mapped to \code{size}, \code{color}, \code{fill}, \code{alpha}, and
  \code{family} aesthetics will the passed to the theme constructor for each
  table. In the case of \code{colour} and \code{fill}, the default mapping
  is to \code{NA} which triggers the use of the default base_colour of the
  \code{ttheme}.

  The constructor \code{ttheme_gtdefault} is used by default, but others
  are available predefined or can created by the user. If instead of a
  constructor a ready constructed ttheme as a list object is passed as
  argument, it will be used as is. In such a case mapped aesthetics normally
  mapped aesthetics are ignored if present.

  Complex tables with annotations or different coloring of rows or cells can
  be constructed with functions in package 'gridExtra' or in any other way
  as long as they can be saved as grid graphical objects and added to a
  ggplot as a new layer with \code{\link{geom_grob}}.
}
\section{Alignment}{
 You can modify table alignment with the \code{vjust} and
  \code{hjust} aesthetics. These can either be a number between 0
  (right/bottom) and 1 (top/left) or a character ("left", "middle", "right",
  "bottom", "center", "top").
}

\section{Inset size}{
 You can modify inset table size with the \code{size}
  aesthetics, which determines the size of text within the table.
}

\section{Warning!}{

  \strong{\code{annotate()} cannot be used with \code{geom = "table"}}. Use
  \code{\link[ggplot2]{annotation_custom}} directly when adding inset tables
  as annotations.
}

\examples{
library(dplyr)
library(tibble)

mtcars \%>\%
  group_by(cyl) \%>\%
  summarize(wt = mean(wt), mpg = mean(mpg)) \%>\%
  ungroup() \%>\%
  mutate(wt = sprintf("\%.2f", wt),
         mpg = sprintf("\%.1f", mpg)) -> tb

df <- tibble(x = 5.45, y = 34, tb = list(tb))

# using defaults
ggplot(mtcars, aes(wt, mpg, colour = factor(cyl))) +
  geom_point() +
  geom_table(data = df, aes(x = x, y = y, label = tb))

ggplot(mtcars, aes(wt, mpg, colour = factor(cyl))) +
  geom_point() +
  geom_table(data = df, aes(x = x, y = y, label = tb),
             table.rownames = TRUE, table.theme = ttheme_gtstripes)

# settings aesthetics to constants
ggplot(mtcars, aes(wt, mpg, colour = factor(cyl))) +
  geom_point() +
  geom_table(data = df, aes(x = x, y = y, label = tb),
             color = "red", fill = "#FFCCCC", family = "serif", size = 5,
             angle = 90, vjust = 0)

# passing a theme constructor as argument
ggplot(mtcars, aes(wt, mpg, colour = factor(cyl))) +
  geom_point() +
  geom_table(data = df, aes(x = x, y = y, label = tb),
             table.theme = ttheme_gtminimal) +
  theme_classic()

df2 <- tibble(x = 5.45, y = c(34, 29, 24), cyl = c(4, 6, 8),
              tb = list(tb[1, 1:3], tb[2, 1:3], tb[3, 1:3]))

# mapped aesthetics
ggplot(data = mtcars, mapping = aes(wt, mpg, color = factor(cyl))) +
  geom_point() +
  geom_table(data = df2,
             inherit.aes = TRUE,
             mapping = aes(x = x, y = y, label = tb))

# Using native plot coordinates instead of data coordinates
dfnpc <- tibble(x = 0.95, y = 0.95, tb = list(tb))

ggplot(mtcars, aes(wt, mpg, colour = factor(cyl))) +
  geom_point() +
  geom_table_npc(data = dfnpc, aes(npcx = x, npcy = y, label = tb))

}
\references{
This geometry is inspired on answers to two questions in
  Stackoverflow. In contrast to these earlier examples, the current geom
  obeys the grammar of graphics, and attempts to be consistent with the
  behaviour of 'ggplot2' geometries.
  \url{https://stackoverflow.com/questions/12318120/adding-table-within-the-plotting-region-of-a-ggplot-in-r}
  \url{https://stackoverflow.com/questions/25554548/adding-sub-tables-on-each-panel-of-a-facet-ggplot-in-r?}
}
\seealso{
function \code{\link[gridExtra]{tableGrob}} as it is used to
  construct the table.

Other geometries for adding insets to ggplots: 
\code{\link{geom_grob}()},
\code{\link{geom_plot}()},
\code{\link{ttheme_gtdefault}()}
}
\concept{geometries for adding insets to ggplots}
