% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/add_pvalue.R
\name{add_pvalue}
\alias{add_pvalue}
\title{Add p-values to a ggplot}
\usage{
add_pvalue(
  data,
  label = NULL,
  xmin = "group1",
  xmax = "group2",
  x = NULL,
  y.position = "y.position",
  label.size = 3.2,
  colour = NULL,
  color = NULL,
  tip.length = 0.03,
  bracket.size = 0.6,
  bracket.colour = NULL,
  bracket.color = NULL,
  bracket.shorten = 0,
  bracket.nudge.y = 0,
  step.increase = 0,
  step.group.by = NULL,
  remove.bracket = FALSE,
  coord.flip = FALSE,
  position = "identity",
  ...
)
}
\arguments{
\item{data}{A \code{data.frame} with the statistics to plot. Default format
has the following columns: \code{group1 | group2 | p.adj | y.position | etc}.
\code{group1} and \code{group2} are the two groups that were compared.
\code{p.adj} is the adjusted p-value. \code{y.position} is the y coordinate
that specifies where on the plot the p-value should go. The column names
can differ from the default as long as their are specified when calling
the function.}

\item{label}{\code{string}. Name of column in \code{data} that
contains the text to plot (e.g. \code{label = "p.adj"}).
Can also be an expression that can be formatted by
\code{\link[glue]{glue}} (e.g. \code{label = "p = {p.adj}"}).}

\item{xmin}{\code{string}. Name of column in \code{data}  that
contains the position of the left side of the brackets.
Default is \code{"group1"}.}

\item{xmax}{Optional. \code{string}. Name of column in \code{data} that
contains the position of the right side of the brackets.
Default is \code{"group2"}. If \code{NULL}, the p-values are plotted as
text only.}

\item{x}{\code{string} or \code{numeric}. x coordinate of the p-value text.
Only use when plotting p-value as text only (no brackets).}

\item{y.position}{\code{string}. Name of column in
\code{data} containing the y coordinates (\code{numeric}) of each p-value to
be plotted. Can also be a single number to plot all p-values at the same
height or a \code{numeric} vector that will override \code{data}.}

\item{label.size}{\code{numeric}. Size of text.}

\item{colour, color}{\code{string}. Colour of text.}

\item{tip.length}{\code{numeric} vector. Length of bracket tips.
Use \code{0} to remove tips.}

\item{bracket.size}{\code{numeric}. Line width of bracket.}

\item{bracket.colour, bracket.color}{\code{string}. Colour of bracket. Default is
\code{NULL} which causes brackets to inherit the colour of the text.}

\item{bracket.shorten}{\code{numeric}. Shortens the brackets slightly to
allow them to be plotted side-by-side at the same y position.}

\item{bracket.nudge.y}{\code{numeric}. Changes the y position of
p-values. Useful for slightly adjusting p-values if the text is cut off.}

\item{step.increase}{\code{numeric}. Changes the space between brackets.}

\item{step.group.by}{\code{string}. Variable to group brackets by.}

\item{remove.bracket}{\code{logical}. If \code{TRUE} all brackets are
removed and p-value is shown as text only.}

\item{coord.flip}{\code{logical}. If \code{TRUE} p-values are rotated by
90 degrees. Should be used with \code{\link[ggplot2]{coord_flip}}}

\item{position}{\code{string} or call to position function such as
\code{\link[ggplot2]{position_dodge}}. Typically used for adjusting x
position of p-values to be in line with dodged data.}

\item{...}{Additional aesthetics or arguments passed to
\code{\link[ggplot2]{layer}}. See below for allowed values.}
}
\value{
Returns a \emph{layer} ggproto object with either \code{geom = GeomBracket} or
\code{geom = GeomText}.
}
\description{
Add p-values with or without brackets to a ggplot.

See \href{https://csdaw.github.io/ggprism/articles/pvalues.html}{here}
or the examples section below for examples of how to use.

\code{add_pvalue} is a refactored version of
\code{stat_pvalue_manual} from
\href{https://github.com/kassambara/ggpubr}{kassambara/ggpubr}, altered to
have less dependencies, and more flexibility with input format and
aesthetics. Any examples using \code{stat_pvalue_manual} found on
\href{https://www.datanovia.com/en/?s=p-value&search-type=default}{Datanovia}
will also work with \code{add_pvalue}.
}
\section{Allowed ... values}{

\code{add_pvalue} understands the following additional aesthetics or arguments:

\describe{
\item{\code{fontface}}{\code{string}. Fontface of text (e.g. \code{"bold"}).}
\item{\code{fontfamily}}{\code{string}. Fontfamily of text (e.g. \code{"Arial"}).}
\item{\code{hjust}}{\code{numeric}. Horizontal justification of text.}
\item{\code{vjust}}{\code{numeric}. Vertical justification of text.}
\item{\code{alpha}}{\code{numeric}. Transparency of text and/or brackets.}
\item{\code{linetype}}{\code{string} or \code{numeric}. Linetype of brackets
(e.g. \code{"dashed"}).}
\item{\code{lineend}}{\code{string}. Lineend of brackets (e.g. \code{"butt"}).}
\item{\code{na.rm}}{\code{logical}. If \code{FALSE} (default), removes
missing values with a warning. If \code{TRUE} silently removes missing
values.}
\item{\code{show.legend}}{\code{logical}. Should this layer be included in
the legends? If \code{NA} (default), include if any aesthetics are mapped.
If \code{FALSE}, never include or if \code{TRUE}, always include. It can
also be a named \code{logical} vector to finely select the aesthetics to
display.}
\item{\code{inherit.aes}}{\code{logical}. If \code{FALSE}, overrides the
default aesthetics, rather than combining with them.}
}
}

\examples{
library(ggplot2)

## we will use the ToothGrowth dataset for all examples
tg <- ToothGrowth
tg$dose <- as.factor(tg$dose)
tg$group <- factor(rep(c("grp1", "grp2"), 30))

## p-value bracket comparing two means
# p-value table (its best to use these column names)
two.means <- tibble::tribble(
  ~group1, ~group2, ~p,     ~y.position,
  "OJ",    "VC",    0.0606, 36
)

# boxplot (or another geom...)
ggplot(tg, aes(x = supp, y = len)) +
  geom_boxplot() +
  add_pvalue(two.means)

# if your table has special column names you will need to specify them
two.means <- tibble::tribble(
  ~apple, ~banana, ~my.pval, ~some.y.position,
  "OJ",    "VC",    0.0606,  36
)

ggplot(tg, aes(x = supp, y = len)) +
  geom_boxplot() +
  add_pvalue(
    two.means,
    xmin = "apple",
    xmax = "banana",
    label = "my.pval",
    y.position = "some.y.position"
  )

## you can make the label a glue expression
two.means <- tibble::tribble(
  ~group1, ~group2, ~p,     ~y.position,
  "OJ",    "VC",    0.0606, 36
)

ggplot(tg, aes(x = supp, y = len)) +
  geom_boxplot() +
  add_pvalue(two.means, label = "p = {p}")

## you can change aesthetics of the bracket and label
ggplot(tg, aes(x = supp, y = len)) +
  geom_boxplot() +
  add_pvalue(
    two.means,
    label = "p = {p}",
    colour = "red", # label
    label.size = 6, # label
    fontface = "bold", # label
    fontfamily = "serif", # label
    angle = 45, # label
    bracket.colour = "blue", # bracket
    bracket.size = 1, # bracket
    linetype = "dashed", # bracket
    lineend = "round" # bracket
  )

## you can change the tip length of the bracket
# make them longer
ggplot(tg, aes(x = supp, y = len)) +
  geom_boxplot() +
  add_pvalue(two.means, tip.length = 0.1)

# make them disappear
ggplot(tg, aes(x = supp, y = len)) +
  geom_boxplot() +
  add_pvalue(two.means, tip.length = 0)

# make one side longer than the other
ggplot(tg, aes(x = supp, y = len)) +
  geom_boxplot() +
  add_pvalue(two.means, tip.length = c(0.1, 0))

## p-value brackets with comparisons to a reference sample
each.vs.ref <- tibble::tribble(
  ~group1, ~group2, ~p.adj,   ~y.position,
  "0.5",   "1",     8.80e-14, 35,
  "0.5",   "2",     1.27e-7,  38
)

ggplot(tg, aes(x = dose, y = len)) +
  geom_boxplot(aes(fill = dose)) +
  add_pvalue(each.vs.ref)

## p-value brackets with pairwise comparisons
pairwise <- tibble::tribble(
  ~group1, ~group2,  ~p.signif, ~y.position,
  "0.5",   "1",      "****",    38,
  "0.5",   "2",      "****",    36,
  "1",     "2",      "****",    38
)

# you can shorten the length of brackets that are close together
ggplot(tg, aes(x = dose, y = len)) +
  geom_boxplot(aes(fill = dose)) +
  add_pvalue(
    pairwise,
    bracket.shorten = c(0.05, 0, 0.05)
  )

# you can nudge brackets that are not quite in the correct y position
# instead of changing the p-value table
ggplot(tg, aes(x = dose, y = len)) +
  geom_boxplot(aes(fill = dose)) +
  add_pvalue(
    pairwise,
    bracket.shorten = c(0.05, 0, 0.05),
    bracket.nudge.y = c(0.5, 0, 0.5)
  )

## p-value brackets with pairwise comparisons of grouped data
pairwise.grouped <- tibble::tribble(
  ~group1, ~group2, ~p.adj,  ~y.position, ~supp,
  "0.5",   "1",     2.63e-4, 33.5,        "OJ",
  "0.5",   "2",     3.96e-6, 37.6,        "OJ",
  "1",     "2",     1.18e-1, 41.6,        "OJ",
  "0.5",   "1",     2.04e-6, 36.5,        "VC",
  "0.5",   "2",     1.40e-7, 40.6,        "VC",
  "1",     "2",     2.75e-4, 44.6,        "VC"
)

# use step.increase to change the spacing between different brackets in the
# groups specified by step.group.by
ggplot(tg, aes(x = dose, y = len)) +
  geom_boxplot(aes(fill = supp)) +
  add_pvalue(
    pairwise.grouped,
    colour = "supp",
    tip.length = 0,
    step.group.by = "supp",
    step.increase = 0.03
  )

## p-value (brackets) with single facet variable
two.means.grouped1 <- tibble::tribble(
  ~group1, ~group2, ~p.adj,  ~y.position, ~dose,
  "OJ",    "VC",    0.0127,  24,          "0.5",
  "OJ",    "VC",    0.00312, 30,          "1",
  "OJ",    "VC",    0.964,   36.5,        "2"
)

ggplot(tg, aes(x = supp, y = len)) +
  geom_boxplot() +
  facet_wrap(~ dose, scales = "free") +
  add_pvalue(two.means.grouped1) # table must have dose column

## p-value (brackets) with single facet variable and multiple brackets per facet
pairwise.grouped <- tibble::tribble(
  ~group1, ~group2, ~p.adj,  ~y.position, ~supp,
  "0.5",   "1",     2.63e-4, 33.5,        "OJ",
  "0.5",   "2",     3.96e-6, 37.6,        "OJ",
  "1",     "2",     1.18e-1, 41.6,        "OJ",
  "0.5",   "1",     2.04e-6, 36.5,        "VC",
  "0.5",   "2",     1.40e-7, 40.6,        "VC",
  "1",     "2",     2.75e-4, 44.6,        "VC"
)

ggplot(tg, aes(x = dose, y = len)) +
  geom_boxplot(aes(fill = supp)) +
  facet_wrap(~ supp) +
  add_pvalue(pairwise.grouped)

## p-value (brackets) with two facet variables
two.means.grouped2 <- tibble::tribble(
  ~group1, ~group2, ~p.signif, ~y.position, ~group, ~dose,
  "OJ",    "VC",    "*",       21,          "grp1", "0.5",
  "OJ",    "VC",    "**",      30,          "grp2", "1"
)

ggplot(tg, aes(x = supp, y = len)) +
  geom_boxplot() +
  facet_wrap(group ~ dose) +
  add_pvalue(two.means.grouped2) # table must have dose and group column

## p-value (text only) comparing two means
two.means <- tibble::tribble(
  ~group1, ~group2, ~p,     ~y.position,
  "OJ",    "VC",    0.0606, 36
)

ggplot(tg, aes(x = supp, y = len)) +
  geom_boxplot() +
  add_pvalue(two.means, remove.bracket = TRUE, x = 1.5)

## p-value (text only) with coord_flip, override y.position, change angle
ggplot(tg, aes(x = supp, y = len)) +
  geom_boxplot() +
  add_pvalue(
    two.means,
    remove.bracket = TRUE,
    x = 1.5,
    y.position = 32,
    angle = 45
  ) +
  coord_flip()

## p-value (text only) comparing to the null
one.mean <- tibble::tribble(
  ~group1, ~group2,      ~p.signif, ~y.position, ~dose,
  "1",     "null model", "****",    35,          "0.5",
  "1",     "null model", "****",    35,          "1",
  "1",     "null model", "****",    35,          "2"
)

ggplot(tg, aes(x = dose, y = len)) +
  geom_boxplot(aes(fill = dose)) +
  add_pvalue(one.mean, x = "dose")

## p-value (text only) with comparisons to a base mean
each.vs.basemean <- tibble::tribble(
  ~group1, ~group2, ~p.adj, ~y.position, ~x,
  "all",   "0.5",   "****", 35, 1,
  "all",   "0.5",   "ns",   35, 2,
  "all",   "0.5",   "****", 35, 3
)

ggplot(tg, aes(x = dose, y = len)) +
  geom_boxplot(aes(fill = dose)) +
  add_pvalue(each.vs.basemean, x = "x")

## p-value (text only) with comparison to reference sample
each.vs.ref <- tibble::tribble(
  ~group1, ~group2, ~p.adj,   ~y.position,
  "0.5",   "1",     8.80e-14, 35,
  "0.5",   "2",     1.27e-7,  38
)

ggplot(tg, aes(x = dose, y = len)) +
  geom_boxplot(aes(fill = dose)) +
  add_pvalue(each.vs.ref, coord.flip = TRUE, remove.bracket = TRUE)

## p-value (text only) with a grouping variable
two.means.grouped1 <- tibble::tribble(
  ~group1, ~group2, ~p.adj,  ~y.position, ~dose,
  "OJ",    "VC",    0.0127,  24,          "0.5",
  "OJ",    "VC",    0.00312, 30,          "1",
  "OJ",    "VC",    0.964,   36.5,        "2"
)

ggplot(tg, aes(x = dose, y = len)) +
  geom_boxplot(aes(fill = supp)) +
  add_pvalue(two.means.grouped1, x = "dose")
}
