#' @name indSEM
#' @aliases indSEM
#' @title Individual-level structural equation model search.
#' @description This function identifies structural equation models for each individual. It does not utilize
#' any shared information from the sample.
#' @usage
#' indSEM(data   = "",
#'        sep    = "",
#'        header = ,
#'        out    = "",
#'        ar     = FALSE,
#'        plot   = TRUE,
#'        paths  = NULL)
#' @param data Path to the directory where the data files are located. Each file must contain one matrix for
#' each individual containing a T (time) by p (number of variables) matrix where the columns represent
#' variables and the rows represent time.
#' @param sep The spacing of the data files. "" indicates space-delimited, "/t" indicates tab-delimited, ","
#' indicates comma delimited.
#' @param header Logical. Indicate TRUE for data files with a header.
#' @param out The path to the directory where the results will be stored. This directory must be generated by
#' the user prior to running the function.
#' @param ar Logical. If TRUE, begins search for individual models with autoregressive (AR) paths open.
#' Defaults to FALSE.
#' @param plot Logical. If TRUE, graphs depicting relations among variables of interest will automatically be
#' created. Defaults to TRUE. For individual-level plots, red paths represent positive weights and blue paths represent negative weights.
#' @param paths lavaan-style syntax containing paths with which to begin model estimation. That is, Y~X indicates that Y is regressed on X, or X predicts Y. If no header is used,
#' then variables should be referred to with V followed (with no separation) by the column number. If a
#' header is used, variables should be referred to using variable names. To reference lag variables, "lag"
#' should be added to the end of the variable name with no separation. Defaults to NULL.
#' @details
#'  In main output directory:
#'  \itemize{
#'  \item{all.elements.summary} {Contains summary information for paths identified at the individual-level.}
#'  \item{all.elements} {Contains information for all paths, per individual, identified at the individual-level.}
#'  \item{all.fit} {Contains model fit information for individual-level models. If subgroups are requested, this file also contains the subgroup membership for each individual.}
#'  \item{finalpaths_contemp} Contains counts of total number of paths (contemporaneous predicting contemporaneous) estimated for sample.
#'  \item{finalpaths_lagged} Contains counts of total number of paths (lagged predicting contemporaneous) estimated for the sample.
#'  }
#'  In individual output directory:
#'  \itemize{
#'  \item{betas} Directory, contains individual-level estimates of each path for each individual. Names are id_contemp and id_lagged.
#'  \item{SEs} Directory, contains individual-level standard errors for each path for each indiivdual. Names are id_contemp and id_lagged.
#'  \item{plots} Directory, contains individual-level plots. Red paths represent positive weights and blue paths represent negative weights.
#' }
#' @author Stephanie Lane
#' @examples
#'  \dontrun{
#' indSEM(data   = "C:/data100",
#'        sep    = ",",
#'        header = FALSE,
#'        out    = "C:/data100_indSEM_out",
#'        ar     = TRUE,
#'        plot   = TRUE,
#'        paths  = NULL)
#'  }
#'@keywords indSEM
#'@export
indSEM <- function(data,
                   sep,
                   header,
                   out,
                   ar = FALSE,
                   plot = TRUE,
                   paths = NULL){

  setup.out        <- setup(data   = data,
                            sep    = sep,
                            header = header,
                            out    = out,
                            plot   = plot,
                            ar     = ar,
                            paths   = paths,
                            subgroup = FALSE)

  ## this is the individual-level search, adds path for each individual
  ## runs one person at a time  indsem.internal.out <- list
  indsem.internal.out <- indsem.internal(setup.out        = setup.out,
                                         subgroup         = FALSE,
                                         evalbetassub.out = NULL,
                                         evalbetas.out    = NULL)

  ## just grabs information from individual-level search (once complete) and prints
  ## summary output and makes tables
  wrapup.out <- wrapup(indsem.internal.out = indsem.internal.out,
                       setup.out           = setup.out,
                       agg                 = FALSE,
                       subgroup            = FALSE)

  print.gimme.indSEM(x=NULL,
              y=NULL,
              z=setup.out)
}

print.gimme.indSEM <- function(x,y,z){
  writeLines("indSEM finished running normally")
  writeLines(paste("output is stored in", z$out))
}
