% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/zz_help_files.R
\name{LightMapper}
\alias{LightMapper}
\title{LightMapper}
\description{
This class provides a framework for creating scatter plots based on a point light simulation. Points are mapped with a inverse power gradient to a raster with specified dimensions and properties.
}
\section{Usage}{

\preformatted{m <- LightMapper$new(xdim=1000, ydim=800, blend_mode = "screen", contrast_limit = 1e5, nthreads = 1)

m$map(x, y, radius, intensity = 1, radius, falloff_exponent = 1, distance_exponent = 2, xlimits = c(NA_real_, NA_real_), ylimits = c(NA_real_, NA_real_), append = FALSE)

m$output_raw(saturation = NA_real_)

m$output_dataframe(saturation = NA_real_)

m$aspect()

m$xlim()

m$ylim()
}
}

\section{Arguments}{

\describe{
  \item{xdim}{- The first dimension of the output matrix raster.}
  \item{ydim}{- The second dimension of the output matrix raster.}
  \item{blend_mode}{- Either screen or additive blending mode. See details.}
  \item{nthreads}{- Number of threads to use.}
  \item{x}{- X coordinate of points.}
  \item{y}{- Y coordinate of points.}
  \item{radius}{- Relative spread of glow intensity. The radius should be proportional to the x and y-ranges of the plot. Values between 1/10 to 1/100 of the range of the plot generally produce good results.}
  \item{intensity}{- Maximum intensity at the center of a point.}
  \item{falloff_exponent}{- Exponent to determine how fast light intensity decreases from the point origin. A value of 0.5 corresponds to a linear falloff; a value of 2 corresponds to an inverse square. Generally you want this value to be high, otherwise you'll flood your plot with light.}
  \item{distance_exponent}{- Exponent of the distance calculation when calculating intensities. A value of 2 corresponds to euclidean distance; a value of 1 corresponds to manhattan distance.}
  \item{xlimits}{- The x-limits of the output plot. If NA, the limits are +/- 5\% of the maximum and minimum points.}
  \item{ylimits}{- The y-limits of the output plot. If NA, the limits are +/- 5\% of the maximum and minimum points.}
  \item{append}{ - Whether to add to the existing output or overwrite.}
  \item{saturation}{- When retrieving the output with \code{$output_raw} or \code{$output_dataframe}, maximum intensity values are capped at the given value. This is often useful when using additive blend mode to increase contrast.}
}
}

\section{Details}{

\code{$new()} creates a new LightMapper object, which holds parameters, plotting data, and the output (a matrix of glow intensities). Creates a canvas to plot point data. With additive blending, the intensities of each point are added arithmetically, which is how light intensities are added in the physical world. This is equivalent to an fast/approximate un-normalized 2D kernel density estimate. 

#' With "screen" blending, two intensities are added according to the formula: \code{I_out = 1 - (1-I_a)*(1-I_b)}. Both additive blending and screen blending are commutative operations, meaning the order of points in a plot does not affect the output.

Note: Mapping "lights" (inverse power intensity) is much slower than "glow" effects (gaussian intensities) for various reasons. Plotting more than a few hundred points with \code{LightMapper} or \code{LightMapper4} may be computationally prohibitive. 

Screen blending can often improve contrast in a plot and is the default. 

\code{$map()} maps points to the canvas. 

\code{$output_raw()} output raw matrix rasters. Useful for plotting in base R.

\code{$output_dataframe()} output the raster as a dataframe with XY coordinates. This is meant to pipe directly into ggplot.

\code{$aspect()}, \code{$xlim()}, \code{$ylim()} return the aspect ratio, x-limits and y-limits of the raster. These functions are intended to be used with plotting functions (e.g. \code{ggplot2::coord_fixed()}) so that the output raster is not distorted. See example below.
}

\examples{
# Plot Data: x,y,r
x <- numeric(length=50)
y <- numeric(length=50)
r <- numeric(length=50)
for(t in 1:50) {
  xy <- exp(1i * t/2 - t/12)
  x[t] <- Re(xy)
  y[t] <- Im(xy)
  r[t] <- sqrt(x[t]^2 + y[t]^2)
}

# New class object
m <- LightMapper$new(xdim=500, ydim = 400, blend_mode = "screen")

# Map data on to raster
m$map(x=x, y=y, intensity = 1, radius = r/100, falloff_exponent = 0.5, distance_exponent = 2)

# Output raster data as a dataframe
pd <- m$output_dataframe(saturation = 1)

# Plot with ggplot
ggplot(pd, aes(x = x, y = y, fill = value)) + 
  geom_raster(show.legend = FALSE) +
  scale_fill_gradientn(colors=additive_alpha(c("black", "purple", "white"))) +
  coord_fixed(ratio = m$aspect(), xlim = m$xlim(), ylim = m$ylim(), expand = FALSE) + 
  theme_night(bgcolor = "black")
}
