
#' @title Predict method for class 'gmvar' objects
#'
#' @description Forecast GMVAR process defined by object of class \code{'gmvar'}. Forecasts are
#'   computed by performing independent simulations and using the sample means or medians as predictions
#'   and empirical quantiles as confidence intervals. For one-step-ahead predictions using the exact
#'   conditional mean is also supported.
#'
#' @param object object of class \code{'gmvar'}, generated by function \code{fitGMVAR()} or \code{GMVAR()}.
#' @param n_ahead how many steps ahead should be predicted?
#' @param n_simu to how many simulations should the forecast be based on?
#' @param ci a numeric vector specifying the confidence levels of the confidence intervals.
#' @param ci_type should the confidence intervals be "two-sided", "upper" or "lower"?
#' @param pred_type should the prediction be based on sample "mean" or "median"? Or should it
#'   be one-step-ahead forecast based on conditional mean (\code{"cond_mean"})? Confidence intervals
#'   won't be calculated if conditional mean is used.
#' @param plot_res should the results be plotted?
#' @param nt a positive integer specifying the number of observations to be plotted
#'   along with the prediction (ignored if \code{plot_res==FALSE}). Default is \code{round(nrow(data)*0.2)}.
#' @param ... additional arguments passed to \code{grid()} (ignored if \code{plot_res==FALSE}).
#' @return Returns an object of class '\code{gmvarpred}' which containts the predictions and the
#'   confidence intervals. It has it's own print and plot methods \code{print.gmvarped()} and \code{plot.gmvarped()}.
#' @inherit in_paramspace_int references
#' @examples
#' \donttest{
#' ## These are long running examples that use parallel computing!
#'
#' # These examples use the data 'eurusd' which comes with the
#' # package, but in a scaled form.
#' data <- cbind(10*eurusd[,1], 100*eurusd[,2])
#' colnames(data) <- colnames(eurusd)
#'
#' # GMVAR(1,2) model
#' fit12 <- fitGMVAR(data, p=1, M=2)
#' p1 <- predict(fit12, n_ahead=1, pred_type="cond_mean",
#'               plot_res=FALSE)
#' p1
#' p2 <- predict(fit12, n_ahead=10)
#' p2
#' p3 <- predict(fit12, n_ahead=10, ci_type="upper")
#' p3
#'
#'
#' # GMVAR(2,2) model
#' fit22 <- fitGMVAR(data, p=2, M=2)
#' p1 <- predict(fit22, n_ahead=20, pred_type="median")
#' p1
#' p2 <- predict(fit22, n_ahead=10, nt=20, lty=1)
#' p2
#' p3 <- predict(fit22, n_ahead=10, ci=c(0.99, 0.90, 0.80, 0.70),
#'               nt=30, lty=0)
#' p3
#' }
#' @export

predict.gmvar <- function(object, ..., n_ahead, n_simu=2000, ci=c(0.95, 0.80), ci_type=c("two-sided", "upper", "lower", "none"),
                          pred_type=c("mean", "median", "cond_mean"), plot_res=TRUE, nt) {
  gmvar <- object
  check_gmvar(gmvar)
  check_null_data(gmvar)
  if(missing(n_ahead)) {
    warning("Argument n_ahead is missing. Using n_ahead = 1.")
    n_ahead <- 1
  }
  if(!all_pos_ints(c(n_ahead, n_simu))) stop("Arguments n_ahead and n_simu must be positive integers")
  stopifnot(all(ci > 0 & ci < 1))
  ci_type <- match.arg(ci_type)
  pred_type <- match.arg(pred_type)
  stopifnot(ci_type %in% c("two-sided", "upper", "lower", "none"))
  stopifnot(pred_type %in% c("mean", "median", "cond_mean"))
  if(missing(nt)) {
    nt <- round(nrow(data)*0.2)
  } else {
    stopifnot(nt > 0 & nt %% 1 == 0)
    if(nt > nrow(data)) {
      warning("nt > nrow(data), using nt = nrow(data)")
      nt <- nrow(data)
    }
  }

  if(pred_type == "cond_mean") {
    if(n_ahead > 1) warning("Only one-step-ahead forecasts are supported for prediction type 'cond_mean'")
    p <- gmvar$model$p
    M <- gmvar$model$M
    d <- gmvar$model$d
    constraints <- gmvar$model$constraints
    params <- gmvar$params
    data <- gmvar$data
    n_obs <- nrow(data)
    mw <- loglikelihood_int(data, p, M, params=params, conditional=gmvar$model$conditional,
                            constraints=constraints, to_return="mw_tplus1")
    mw <- mw[nrow(mw),]

    # Collect parameter values
    if(gmvar$model$parametrization == "mean") {
      params <- change_parametrization(p=p, M=M, d=d, params=params, constraints=constraints,
                                       change_to="intercept")
    }
    params <- reform_constrained_pars(p=p, M=M, d=d, params=params, constraints=constraints)
    all_phi0 <- pick_phi0(p=p, M=M, d=d, params=params)
    all_A <- pick_allA(p=p, M=M, d=d, params=params)

    # Calculate the conditional mean
    pred <- rowSums(vapply(1:M, function(m) mw[m]*(all_phi0[, m] + rowSums(vapply(1:p, function(i1) all_A[, , i1, m]%*%data[n_obs + 1 - i1,],
                                                                                  numeric(d)))), numeric(d)))
    conf_ints <- NULL
    ci <- NULL
    ci_type <- "none"
    q_tocalc <- numeric(0)
  } else {

    # Simulations
    simulations <- simulateGMVAR(gmvar, nsimu=n_ahead, init_values=gmvar$data, ntimes=n_simu)

    # Calculate quantiles from the third dimension of 3D simulation array
    dim3_quantiles <- function(x, q) {
      apply(x, c(1, 2), quantile, probs=q, names=FALSE)
    }

    # Predictions
    if(pred_type == "mean") {
      pred <- rowMeans(simulations, dims=2)
    } else {
      pred <- dim3_quantiles(simulations, q=0.5)
    }
    colnames(pred) <- colnames(gmvar$data)

    # Confidence intervals
    if(ci_type == "upper") {
      q_tocalc <- ci
    } else if(ci_type == "lower") {
      q_tocalc <- 1-ci
    } else if(ci_type == "two-sided") {
      lower <- (1-ci)/2
      upper <- rev(1 - lower)
      q_tocalc <- c(lower, upper)
    } else {
      q_tocalc <- numeric(0)
      ci <- NULL
    }
    q_tocalc <- sort(q_tocalc, decreasing=FALSE)

    conf_ints <- lapply(q_tocalc, function(q) dim3_quantiles(simulations, q)) # aperm(dim3_quantiles(simulations, q=q_tocalc), perm=c(2, 3, 1))
    names(conf_ints) <- q_tocalc
  }

  ret <- structure(list(gmvar=gmvar,
                        pred=pred,
                        conf_ints=conf_ints,
                        n_ahead=n_ahead,
                        n_simu=n_simu,
                        ci=ci,
                        ci_type=ci_type,
                        pred_type=pred_type,
                        q=q_tocalc),
                   class="gmvarpred")
  if(plot_res == TRUE) plot.gmvarpred(x=ret, nt=nt, ...)
  ret
}

