% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simulate.R
\name{simulate.gsmvar}
\alias{simulate.gsmvar}
\title{Simulate method for class 'gsmvar' objects}
\usage{
\method{simulate}{gsmvar}(
  object,
  nsim = 1,
  seed = NULL,
  ...,
  init_values = NULL,
  init_regimes = 1:sum(gsmvar$model$M),
  ntimes = 1,
  drop = TRUE,
  girf_pars = NULL
)
}
\arguments{
\item{object}{an object of class \code{'gsmvar'}, typically created with \code{fitGSMVAR} or \code{GSMVAR}.}

\item{nsim}{number of observations to be simulated.}

\item{seed}{set seed for the random number generator?}

\item{...}{currently not in use.}

\item{init_values}{a size \eqn{(pxd)} matrix specifying the initial values, where d is the number
of time series in the system. The \strong{last} row will be used as initial values for the first lag,
the second last row for second lag etc. If not specified, initial values will be drawn according to
mixture distribution specifed by the argument \code{init_regimes}.}

\item{init_regimes}{a numeric vector of length at most \eqn{M} and elements
in \eqn{1,...,M} specifying the regimes from which the initial values
should be generated from. The initial values will be generated from a
mixture distribution with the mixture components being the stationary
distributions of the specific regimes and the (proportional) mixing weights
given by the mixing weight parameters of those regimes. Note that if
\code{init_regimes=1:M}, the initial values are generated from the
stationary distribution of the process and if \code{init_regimes=m}, the
initial value are generated from the stationary distribution of the
\eqn{m}th regime. Ignored if the argument \code{init_values} is specified.}

\item{ntimes}{how many sets of simulations should be performed?}

\item{drop}{if \code{TRUE} (default) then the components of the returned list are coerced to lower dimension if \code{ntimes==1}, i.e.,
\code{$sample} and \code{$mixing_weights} will be matrices, and \code{$component} will be vector.}

\item{girf_pars}{This argument is used internally in the estimation of generalized impulse response functions (see \code{?GIRF}).
You should ignore it (specifying something else than null to it will change how the function behaves).}
}
\value{
If \code{drop==TRUE} and \code{ntimes==1} (default): \code{$sample}, \code{$component}, and \code{$mixing_weights} are matrices.
  Otherwise, returns a list with...
  \describe{
    \item{\code{$sample}}{a size (\code{nsim}\eqn{ x d x }\code{ntimes}) array containing the samples: the dimension \code{[t, , ]} is
     the time index, the dimension \code{[, d, ]} indicates the marginal time series, and the dimension \code{[, , i]} indicates
     the i:th set of simulations.}
    \item{\code{$component}}{a size (\code{nsim}\eqn{ x }\code{ntimes}) matrix containing the information from which mixture component
     each value was generated from.}
    \item{\code{$mixing_weights}}{a size (\code{nsim}\eqn{ x M x }\code{ntimes}) array containing the mixing weights corresponding to
     the sample: the dimension \code{[t, , ]} is the time index, the dimension \code{[, m, ]} indicates the regime, and the dimension
     \code{[, , i]} indicates the i:th set of simulations.}
  }
}
\description{
\code{simulate.gsmvar} is a simulate method for class 'gsmvar' objects.
  It allows to simulate observations from a GMVAR, StMVAR, or G-StMVAR process.
}
\details{
The argument \code{ntimes} is intended for forecasting: a GMVAR, StMVAR, or G-StMVAR process can be forecasted by simulating
 its possible future values. One can easily perform a large number simulations and calculate the sample quantiles from the simulated
 values to obtain prediction intervals (see the forecasting example).
}
\examples{
 # GMVAR(1,2), d=2 process, initial values from the stationary
 # distribution
 params12 <- c(0.55, 0.112, 0.344, 0.055, -0.009, 0.718, 0.319, 0.005,
  0.03, 0.619, 0.173, 0.255, 0.017, -0.136, 0.858, 1.185, -0.012, 0.136,
  0.674)
 mod12 <- GSMVAR(p=1, M=2, d=2, params=params12)
 set.seed(1)
 sim12 <- simulate(mod12, nsim=500)
 plot.ts(sim12$sample)
 ts.plot(sim12$mixing_weights, col=c("blue", "red"), lty=2)
 plot(sim12$component, type="l")

 # StMVAR(2, 2), d=2 model
 params22t <- c(0.554, 0.033, 0.184, 0.005, -0.186, 0.683, 0.256, 0.031,
  0.026, 0.204, 0.583, -0.002, 0.048, 0.697, 0.154, 0.049, 0.374, 0.476,
   0.318, -0.645, -0.302, -0.222, 0.193, 0.042, -0.013, 0.048, 0.818,
    4.334, 20)
 mod22t <- GSMVAR(gdpdef, p=2, M=2, params=params22t, model="StMVAR")
 sim22t <- simulate(mod22t, nsim=100)
 plot.ts(sim22t$mixing_weights)

 ## FORECASTING EXAMPLE ##
 # Forecast 5-steps-ahead, 500 sets of simulations with initial
 # values from the data:
 # GMVAR(2,2), d=2 model
 params22 <- c(0.36, 0.121, 0.223, 0.059, -0.151, 0.395, 0.406, -0.005,
  0.083, 0.299, 0.215, 0.002, 0.03, 0.484, 0.072, 0.218, 0.02, -0.119,
   0.722, 0.093, 0.032, 0.044, 0.191, 1.101, -0.004, 0.105, 0.58)
 mod22 <- GSMVAR(gdpdef, p=2, M=2, params=params22)
 sim22 <- simulate(mod22, nsim=5, ntimes=500)

 # Point forecast + 95\% prediction intervals:
 apply(sim22$sample, MARGIN=1:2, FUN=quantile, probs=c(0.025, 0.5, 0.972))

 # Similar forecast for the mixing weights:
 apply(sim22$mixing_weights, MARGIN=1:2, FUN=quantile,
       probs=c(0.025, 0.5, 0.972))
}
\references{
\itemize{
   \item Kalliovirta L., Meitz M. and Saikkonen P. 2016. Gaussian mixture vector autoregression.
         \emph{Journal of Econometrics}, \strong{192}, 485-498.
   \item Lütkepohl H. 2005. New Introduction to Multiple Time Series Analysis,
         \emph{Springer}.
   \item McElroy T. 2017. Computation of vector ARMA autocovariances.
         \emph{Statistics and Probability Letters}, \strong{124}, 92-96.
   \item Virolainen S. (forthcoming). A statistically identified structural vector autoregression with endogenously
          switching volatility regime. \emph{Journal of Business & Economic Statistics}.
   \item Virolainen S. 2022. Gaussian and Student's t mixture vector autoregressive model with application to the
     asymmetric effects of monetary policy shocks in the Euro area. Unpublished working
     paper, available as arXiv:2109.13648.
 }
}
\seealso{
\code{\link{fitGSMVAR}}, \code{\link{GSMVAR}}, \code{\link{diagnostic_plot}}, \code{\link{predict.gsmvar}},
 \code{\link{profile_logliks}}, \code{\link{quantile_residual_tests}}, \code{\link{GIRF}}, \code{\link{GFEVD}}
}
