% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/linearIRF.R, R/plotMethods.R, R/printMethods.R
\name{linear_IRF}
\alias{linear_IRF}
\alias{plot.irf}
\alias{print.irf}
\title{Estimate linear impulse response function based on a single regime of a structural GMVAR,
  StMVAR, or G-StMVAR model.}
\usage{
linear_IRF(
  gsmvar,
  N = 30,
  regime = 1,
  which_cumulative = numeric(0),
  scale = NULL,
  ci = NULL,
  bootstrap_reps = 100,
  ncores = 2,
  ncalls = 1,
  seeds = NULL,
  ...
)

\method{plot}{irf}(x, shocks_to_plot, ...)

\method{print}{irf}(x, ..., digits = 2, N_to_print, shocks_to_print)
}
\arguments{
\item{gsmvar}{an object of class \code{'gsmvar'} defining a structural or reduced form
GSMVAR model. For a reduced form model, the shocks are automatically identified by
the lower triangular Cholesky decomposition.}

\item{N}{a positive integer specifying the horizon how far ahead should the
linear impulse responses be calculated.}

\item{regime}{Based on which regime the linear IRF should be calculated?
An integer in \eqn{1,...,M}.}

\item{which_cumulative}{a numeric vector with values in \eqn{1,...,d}
(\code{d=ncol(data)}) specifying which the variables for which the linear impulse
responses should be cumulative. Default is none.}

\item{scale}{should the linear IRFs to some of the shocks be scaled so that they
correspond to a specific instantaneous response of some specific
variable? Provide a length three vector where the shock of interest
is given in the first element (an integer in \eqn{1,...,d}), the variable of
interest is given in the second element (an integer in \eqn{1,...,d}), and
its instantaneous response in the third element (a non-zero real number).
If the linear IRFs of multiple shocks should be scaled, provide a matrix which has one
column for each of the shocks with the columns being the length three vectors described above.}

\item{ci}{a real number in \eqn{(0, 1)} specifying the confidence level of the
confidence intervals calculated via a fixed-design wild residual bootstrap method.
Available only for models that impose linear autoregressive dynamics
(excluding changes in the volatility regime).}

\item{bootstrap_reps}{the number of bootstrap repetitions for estimating confidence bounds.}

\item{ncores}{the number of CPU cores to be used in parallel computing when bootstrapping confidence bounds.}

\item{ncalls}{on how many estimation rounds should each bootstrap estimation be based on?
Does not have to be very large since initial estimates used are based on the initially fitted model.
Larger number of rounds gives more reliable results but is computationally more demanding.}

\item{seeds}{a numeric vector of length \code{bootstrap_reps} initializing the seed for the random
generator for each bootstrap replication.}

\item{...}{currently not used.}

\item{x}{object of class \code{'irf'} generated by the function \code{linear_IRF}.}

\item{shocks_to_plot}{IRFs of which shocks should be plotted? A numeric vector
with elements in \code{1,...,d}.}

\item{digits}{the number of decimals to print}

\item{N_to_print}{an integer specifying the horizon how far to print the estimates and
confidence intervals. The default is that all the values are printed.}

\item{shocks_to_print}{the responses to which should should be printed?
A numeric vector with elements in \code{1,...,d}. The default is that responses to
all the shocks are printed.}
}
\value{
Returns a class \code{'irf'} list with the following elements:
  \describe{
    \item{\code{$point_est}:}{a 3D array \code{[variables, shock, horizon]} containing the point estimates of the IRFs.
       Note that the first slice is for the impact responses and the slice i+1 for the period i. The response of the
       variable 'i1' to the shock 'i2' is subsetted as \code{$point_est[i1, i2, ]}.}
    \item{\code{$conf_ints}:}{bootstrapped confidence intervals for the IRFs in a \code{[variables, shock, horizon, bound]}
       4D array. The lower bound is obtained as \code{$conf_ints[, , , 1]}, and similarly the upper bound as
        \code{$conf_ints[, , , 2]}. The subsetted 3D array is then the bound in a form similar to \code{$point_est}.}
    \item{\code{$all_bootstrap_reps}:}{IRFs from all of the bootstrap replications in a \code{[variables, shock, horizon, rep]}.
       4D array. The IRF from replication i1 is obtained as \code{$all_bootstrap_reps[, , , i1]}, and the subsetted 3D array
       is then the in a form similar to \code{$point_est}.}
    \item{Other elements:}{contains some of the arguments the \code{linear_IRF} was called with.}
  }
}
\description{
\code{linear_IRF} estimates linear impulse response function based on a single regime
  of a structural GMVAR, StMVAR, or G-StMVAR model.
}
\details{
The model DOES NOT need to be structural in order for this function to be
  applicable. When an identified structural GMVAR, StMVAR, or G-StMVAR model is
  provided in the argument \code{gsmvar}, the identification imposed by the model
  is used. When a reduced form model is provided in the argument \code{gsmvar},
  lower triangular Cholesky identification is used to identify the shocks.

  If the autoregressive dynamics of the model are linear (i.e., either M == 1 or mean and AR parameters
  are constrained identical across the regimes), confidence bounds can be calculated based on a type of fixed-design
  wild residual bootstrap method. See Virolainen (forthcoming) for a related discussion. We employ the method described
  in Herwartz and Lütkepohl (2014); see also the relevant chapters in Kilian and Lütkepohl (2017).
}
\section{Functions}{
\itemize{
\item \code{plot(irf)}: plot method

\item \code{print(irf)}: print method

}}
\examples{
  \donttest{
 # These are long running examples that take a few minutes to run

 ## GMVAR, p=5, M=2, d=2 model with linear AR dynamics.
 # recursive identification, IRF based on the first regime:
 params52cm <- c(0.788, 0.559, 0.277, 0.038, -0.061, 0.463, 0.286, 0,
               0.035, 0.161, -0.112, 0.031, -0.313, 0.183, 0.103, 0.014,
               0.002, 0.143, -0.089, -0.013, 0.182, -0.04, 1.3, 0.008,
               0.139, 0.277, -0.005, 0.032, 0.118)
 mod52cm <- GSMVAR(data=gdpdef, p=5, M=2, params=params52cm,
                   constraints=rbind(diag(5*2^2), diag(5*2^2)),
                   same_means=list(1:2), parametrization="mean")
 irf1 <- linear_IRF(mod52cm, regime=1, N=20, scale=cbind(c(1, 1, 1), c(2, 2, 1)))
 print(irf1, digits=3)
 plot(irf1)

 # Identification by heteroskedasticity, bootstrapped confidence intervals and
 # and scaled instantaneous effects of the shocks. Note that in actual
 # empirical application, a larger number of bootstrap reps should be used.
 mod52cms <- gsmvar_to_sgsmvar(mod52cm)
 irf2 <- linear_IRF(mod52cms, regime=1, N=20, ci=0.68, bootstrap_reps=10,
                    ncalls=1, seeds=1:10, ncores=1)
 plot(irf2)
 }
}
\references{
\itemize{
   \item Herwartz H. and Lütkepohl H. 2014. Structural vector autoregressions with Markov switching:
     Combining conventional with statistical identification of shocks. \emph{Journal of Econometrics},
     183, pp. 104-116.
   \item Kilian L. and Lütkepohl H. 2017. Structural Vectors Autoregressive Analysis.
         \emph{Cambridge University Press}, Cambridge.
   \item Virolainen S. 2025. A statistically identified structural vector autoregression with endogenously
         switching volatility regime. \emph{Journal of Business & Economic Statistics}, \strong{43}, 1, 44-54.
 }
}
\seealso{
\code{\link{GIRF}}, \code{\link{GFEVD}}, \code{\link{fitGSMVAR}}, \code{\link{GSMVAR}},
  \code{\link{gsmvar_to_sgsmvar}}, \code{\link{reorder_W_columns}}, \code{\link{swap_W_signs}}
}
