% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/diagnose.R
\name{qq_plot}
\alias{qq_plot}
\alias{qq_plot.default}
\alias{qq_plot.gam}
\alias{qq_plot.glm}
\alias{qq_plot.lm}
\title{Quantile-quantile plot of model residuals}
\usage{
qq_plot(model, ...)

\method{qq_plot}{default}(model, ...)

\method{qq_plot}{gam}(
  model,
  method = c("uniform", "simulate", "normal", "direct"),
  type = c("deviance", "response", "pearson"),
  n_uniform = 10,
  n_simulate = 50,
  seed = NULL,
  level = 0.9,
  ylab = NULL,
  xlab = NULL,
  title = NULL,
  subtitle = NULL,
  caption = NULL,
  ci_col = "black",
  ci_alpha = 0.2,
  point_col = "black",
  point_alpha = 1,
  line_col = "red",
  ...
)

\method{qq_plot}{glm}(model, ...)

\method{qq_plot}{lm}(model, ...)
}
\arguments{
\item{model}{a fitted model. Currently models inheriting from class \code{"gam"},
as well as classes \code{"glm"} and \code{"lm"} from calls to \link[stats:glm]{stats::glm} or
\link[stats:lm]{stats::lm} are supported.}

\item{...}{arguments passed ot other methods.}

\item{method}{character; method used to generate theoretical quantiles.
The default is \code{"uniform"}, which generates reference quantiles using
random draws from a uniform distribution and the inverse cummulative
distribution function (CDF) of the fitted values. The reference quantiles
are averaged over \code{n_uniform} draws. \code{"simulate"} generates reference
quantiles by simulating new response data from the model at the observed
values of the covariates, which are then residualised to generate reference
quantiles, using \code{n_simulate} simulated data sets. \code{"normal"} generates
reference quantiles using the standard normal distribution. \code{"uniform"} is
more computationally efficient, but \code{"simulate"} allows reference bands to
be drawn on the QQ-plot. \code{"normal"} should be avoided but is used as a fall
back if a random number generator (\code{"simulate"}) or the inverse of the CDF
are not available from the \code{family} used during model fitting
(`"uniform"``).

Note that \code{method = "direct"} is deprecated in favour of
\code{method = "uniform"}.}

\item{type}{character; type of residuals to use. Only \code{"deviance"},
\code{"response"}, and \code{"pearson"} residuals are allowed.}

\item{n_uniform}{numeric; number of times to randomize uniform quantiles
in the direct computation method (\code{method = "uniform"}).}

\item{n_simulate}{numeric; number of data sets to simulate from the estimated
model when using the simulation method (\code{method = "simulate"}).}

\item{seed}{numeric; the random number seed to use for \code{method = "simulate"}
and \code{method = "uniform"}.}

\item{level}{numeric; the coverage level for reference intervals. Must be
strictly \verb{0 < level < 1}. Only used with \code{method = "simulate"}.}

\item{ylab}{character or expression; the label for the y axis. If not
supplied, a suitable label will be generated.}

\item{xlab}{character or expression; the label for the y axis. If not
supplied, a suitable label will be generated.}

\item{title}{character or expression; the title for the plot. See
\code{\link[ggplot2:labs]{ggplot2::labs()}}. May be a vector, one per penalty.}

\item{subtitle}{character or expression; the subtitle for the plot. See
\code{\link[ggplot2:labs]{ggplot2::labs()}}. May be a vector, one per penalty.}

\item{caption}{character or expression; the plot caption. See
\code{\link[ggplot2:labs]{ggplot2::labs()}}. May be a vector, one per penalty.}

\item{ci_col}{fill colour for the reference interval when
\code{method = "simulate"}.}

\item{ci_alpha}{alpha transparency for the reference
interval when \code{method = "simulate"}.}

\item{point_col}{colour of points on the QQ plot.}

\item{point_alpha}{alpha transparency of points on the QQ plot.}

\item{line_col}{colour used to draw the reference line.}
}
\description{
Quantile-quantile plots (QQ-plots) for GAMs using the reference quantiles of
Augustin \emph{et al} (2012).
}
\note{
The wording used in \code{\link[mgcv:qq.gam]{mgcv::qq.gam()}} uses \emph{direct} in reference to the
simulated residuals method (\code{method = "simulated"}). To avoid confusion,
\code{method = "direct"} is deprecated in favour of \code{method = "uniform"}.
}
\examples{
load_mgcv()
## simulate binomial data...
dat <- data_sim("eg1", n = 200, dist = "binary", scale = .33, seed = 0)
p <- binomial()$linkinv(dat$f) # binomial p
n <- sample(c(1, 3), 200, replace = TRUE) # binomial n
dat <- transform(dat, y = rbinom(n, n, p), n = n)
m <- gam(y / n ~ s(x0) + s(x1) + s(x2) + s(x3),
  family = binomial, data = dat, weights = n,
  method = "REML"
)

## Q-Q plot; default using direct randomization of uniform quantiles
qq_plot(m)

## Alternatively use simulate new data from the model, which
## allows construction of reference intervals for the Q-Q plot
qq_plot(m,
  method = "simulate",
  seed = 42,
  point_col = "steelblue",
  point_alpha = 0.4
)

## ... or use the usual normality assumption
qq_plot(m, method = "normal")
}
\references{
The underlying methodology used when \code{method} is \code{"simulate"} or \code{"uniform"}
is described in Augustin \emph{et al} (2012):

Augustin, N.H., Sauleau, E.-A., Wood, S.N., (2012) On quantile quantile plots
for generalized linear models. \emph{Computational Statatistics and Data Analysis}
\strong{56}, 2404-2409 \doi{doi:10.1016/j.csda.2012.01.026}.
}
\seealso{
\link[mgcv:qq.gam]{mgcv::qq.gam} for more details on the methods used.
}
