% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gs_spending_bound.R
\name{gs_spending_bound}
\alias{gs_spending_bound}
\title{Derive spending bound for group sequential boundary}
\usage{
gs_spending_bound(
  k = 1,
  par = list(sf = gsDesign::sfLDOF, total_spend = 0.025, param = NULL, timing = NULL,
    max_info = NULL),
  hgm1 = NULL,
  theta = 0.1,
  info = 1:3,
  efficacy = TRUE,
  test_bound = TRUE,
  r = 18,
  tol = 1e-06
)
}
\arguments{
\item{k}{Analysis for which bound is to be computed.}

\item{par}{A list with the following items:
\itemize{
\item \code{sf} (class spending function).
\item \code{total_spend} (total spend).
\item \code{param} (any parameters needed by the spending function \code{sf()}).
\item \code{timing} (a vector containing values at which spending function
is to be evaluated or \code{NULL} if information-based spending is used).
\item \code{max_info} (when \code{timing} is \code{NULL}, this can be input as positive number
to be used with \code{info} for information fraction at each analysis).
}}

\item{hgm1}{Subdensity grid from \code{h1()} (k=2) or \code{hupdate()} (k>2)
for analysis k-1; if k=1, this is not used and may be \code{NULL}.}

\item{theta}{Natural parameter used for lower bound only spending;
represents average drift at each time of analysis at least up to analysis k;
upper bound spending is always set under null hypothesis (theta = 0).}

\item{info}{Statistical information at all analyses, at least up to analysis k.}

\item{efficacy}{\code{TRUE} (default) for efficacy bound, \code{FALSE} otherwise.}

\item{test_bound}{A logical vector of the same length as \code{info}
should indicate which analyses will have a bound.}

\item{r}{Integer value controlling grid for numerical integration
as in Jennison and Turnbull (2000); default is 18, range is 1 to 80.
Larger values provide larger number of grid points and greater accuracy.
Normally \code{r} will not be changed by the user.}

\item{tol}{Tolerance parameter for convergence (on Z-scale).}
}
\value{
Returns a numeric bound (possibly infinite) or, upon failure,
generates an error message.
}
\description{
Computes one bound at a time based on spending under given distributional
assumptions. While user specifies \code{gs_spending_bound()} for use with other
functions, it is not intended for use on its own.
Most important user specifications are made through a list provided to
functions using \code{gs_spending_bound()}.
Function uses numerical integration and Newton-Raphson iteration to derive
an individual bound for a group sequential design that satisfies a
targeted boundary crossing probability. Algorithm is a simple extension of
that in Chapter 19 of Jennison and Turnbull (2000).
}
\section{Specification}{

\if{latex}{
 \itemize{
   \item Set the spending time at analysis.
   \item Compute the cumulative spending at analysis.
   \item Compute the incremental spend at each analysis.
   \item Set test_bound a vector of length k > 1 if input as a single value.
   \item Compute spending for current bound.
   \item Iterate to convergence as in gsbound.c from gsDesign.
   \item Compute subdensity for final analysis in rejection region.
   \item Validate the output and return an error message in case of failure.
   \item Return a numeric bound (possibly infinite).
  }
}
\if{html}{The contents of this section are shown in PDF user manual only.}
}

\examples{
info <- (1:3) * 10
info_frac <- info / max(info)
k <- length(info_frac)

# 1st analysis
a1 <- gs_spending_bound(
  k = 1, efficacy = FALSE, theta = 0,
  par = list(sf = gsDesign::sfLDOF, total_spend = 0.025, timing = info_frac, param = NULL),
  hgm1 = NULL
)

b1 <- gs_spending_bound(
  k = 1, efficacy = TRUE, theta = 0,
  par = list(sf = gsDesign::sfLDOF, total_spend = 0.025, timing = info_frac, param = NULL),
  hgm1 = NULL
)
cat("The (lower, upper) boundary at the 1st analysis is (", a1, ", ", b1, ").\n")

# 2nd analysis
a2 <- gs_spending_bound(
  k = 2, efficacy = FALSE, theta = 0,
  par = list(sf = gsDesign::sfLDOF, total_spend = 0.025, timing = info_frac, param = NULL),
  hgm1 = h1(r = 18, theta = 0, info = info[1], a = a1, b = b1)
)

b2 <- gs_spending_bound(
  k = 2, efficacy = TRUE, theta = 0,
  par = list(sf = gsDesign::sfLDOF, total_spend = 0.025, timing = info_frac, param = NULL),
  hgm1 = h1(r = 18, theta = 0, info = info[1], a = a1, b = b1)
)
cat("The upper boundary at the 2nd analysis is (", a2, ", ", b2, ").\n")
}
\references{
Jennison C and Turnbull BW (2000),
\emph{Group Sequential Methods with Applications to Clinical Trials}.
Boca Raton: Chapman and Hall.
}
\author{
Keaven Anderson \email{keaven_anderson@merck.com}
}
