% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/format_data.R
\name{fmt_icon}
\alias{fmt_icon}
\title{Use icons within a table's body cells}
\usage{
fmt_icon(
  data,
  columns = everything(),
  rows = everything(),
  height = "1em",
  sep = " ",
  stroke_color = NULL,
  stroke_width = NULL,
  stroke_alpha = NULL,
  fill_color = NULL,
  fill_alpha = NULL,
  vertical_adj = NULL,
  margin_left = NULL,
  margin_right = NULL,
  a11y = c("semantic", "decorative", "none")
)
}
\arguments{
\item{data}{\emph{The gt table data object}

\verb{obj:<gt_tbl>} // \strong{required}

This is the \strong{gt} table object that is commonly created through use of the
\code{\link[=gt]{gt()}} function.}

\item{columns}{\emph{Columns to target}

\code{\link[=rows-columns]{<column-targeting expression>}} // \emph{default:} \code{everything()}

Can either be a series of column names provided in \code{c()}, a vector of
column indices, or a select helper function (e.g. \code{\link[=starts_with]{starts_with()}},
\code{\link[=ends_with]{ends_with()}}, \code{\link[=contains]{contains()}}, \code{\link[=matches]{matches()}}, \code{\link[=num_range]{num_range()}} and \code{\link[=everything]{everything()}}).}

\item{rows}{\emph{Rows to target}

\code{\link[=rows-columns]{<row-targeting expression>}} // \emph{default:} \code{everything()}

In conjunction with \code{columns}, we can specify which of their rows should
undergo formatting. The default \code{\link[=everything]{everything()}} results in all rows in
\code{columns} being formatted. Alternatively, we can supply a vector of row
captions within \code{c()}, a vector of row indices, or a select helper
function (e.g. \code{\link[=starts_with]{starts_with()}}, \code{\link[=ends_with]{ends_with()}}, \code{\link[=contains]{contains()}}, \code{\link[=matches]{matches()}},
\code{\link[=num_range]{num_range()}}, and \code{\link[=everything]{everything()}}). We can also use expressions to filter
down to the rows we need (e.g., \verb{[colname_1] > 100 & [colname_2] < 50}).}

\item{height}{\emph{Height of icon}

\verb{scalar<character>} // \emph{default:} \code{"1em"}

The absolute height of the icon in the table cell. By default, this is set
to \code{"1em"}.}

\item{sep}{\emph{Separator between icons}

\verb{scalar<character>} // \emph{default:} \code{" "}

In the output of icons within a body cell, \code{sep} provides the separator
between each icon. By default, this is a single space character (\code{" "}).}

\item{stroke_color}{\emph{Color of the icon stroke/outline}

\verb{scalar<character>} // \emph{default:} \code{NULL} (\code{optional})

The icon stroke is essentially the outline of the icon. The color of the
stroke can be modified by applying a single color here. If not provided
then the default value of \code{"currentColor"} is applied so that the stroke
color matches that of the parent HTML element's color attribute.}

\item{stroke_width}{\emph{Width of the icon stroke/outline}

\verb{scalar<character|numeric|integer>} // \emph{default:} \code{NULL} (\code{optional})

The \code{stroke_width} option allows for setting the color of the icon outline
stroke. By default, the stroke width is very small at \code{"1px"} so a size
adjustment here can sometimes be useful.}

\item{stroke_alpha}{\emph{Transparency value for icon stroke/outline}

\verb{scalar<numeric>} // \emph{default:} \code{NULL} (\code{optional})

The level of transparency for the icon stroke can be controlled with a
decimal value between \code{0} and \code{1}.}

\item{fill_color}{\emph{Color of the icon fill}

\verb{scalar<character>|vector<character>} // \emph{default:} \code{NULL} (\code{optional})

The fill color of the icon can be set with \code{fill_color}; providing a single
color here will change the color of the fill but not of the icon's 'stroke'
or outline (use \code{stroke_color} to modify that). A named vector or named
list comprising the icon names with corresponding fill colors can
alternatively be used here (e.g.,
\verb{list("circle-check" = "green", "circle-xmark" = "red"}). If nothing is
provided then the default value of \code{"currentColor"} is applied so that the
fill matches the color of the parent HTML element's color attribute.}

\item{fill_alpha}{\emph{Transparency value for icon fill}

\verb{scalar<numeric|integer>(0>=val>=1)} // \emph{default:} \code{NULL} (\code{optional})

The level of transparency for the icon fill can be controlled with a
decimal value between \code{0} and \code{1}.}

\item{vertical_adj}{\emph{Vertical adjustment of icon from baseline}

\verb{scalar<character|numeric|integer>} // \emph{default:} \code{NULL} (\code{optional})

The vertical alignment of the icon. By default, a length of \code{"-0.125em"}
is used.}

\item{margin_left}{\emph{Margin width left of icon}

\verb{scalar<character|numeric|integer>} // \emph{default:} \code{NULL} (\code{optional})

The length value for the margin that's to the left of the icon can be set
with \code{margin_left}. By default, \code{"auto"} is used for this but if space is
needed on the left-hand side then a length of \code{"0.2em"} is recommended as a
starting point.}

\item{margin_right}{\emph{Margin width right of icon}

\verb{scalar<character|numeric|integer>} // \emph{default:} \code{NULL} (\code{optional})

The length value for the margin that's to the right of the icon can be set
with \code{margin_right}. By default, \code{"auto"} is used for this but if space is
needed on the right-hand side then a length of \code{"0.2em"} is recommended as
a starting point.}

\item{a11y}{\emph{Accessibility mode for icon}

\verb{singl-kw:[semantic|decorative|none]} // \emph{default:} \code{"semantic"}

The accessibility mode for the icon display can be set with the \code{a11y}
argument. Icons can either be \code{"semantic"} or \code{"decorative"}. Using
\code{"none"} will result in no accessibility features for the icons.}
}
\value{
An object of class \code{gt_tbl}.
}
\description{
We can draw from a library of thousands of icons and selectively insert them
into a \strong{gt} table. The \code{fmt_icon()} function makes this possible and it
operates a lot like \code{\link[=fmt_flag]{fmt_flag()}} in that input cells need to contain some
reference to an icon name. We are exclusively using \emph{Font Awesome} icons here
(and we do need to have the \strong{fontawesome} package installed) so the
reference is the short icon name. Multiple icons can be included per cell by
separating icon names with commas (e.g., \code{"hard-drive,clock"}). The \code{sep}
argument allows for a common separator to be applied between flag icons.
}
\section{Compatibility of formatting function with data values}{


\code{fmt_icon()} is compatible with body cells that are of the
\code{"character"} or \code{"factor"} types. Any other types of body cells are ignored
during formatting. This is to say that cells of incompatible data types may
be targeted, but there will be no attempt to format them.
}

\section{Compatibility of arguments with the \code{from_column()} helper function}{


\code{\link[=from_column]{from_column()}} can be used with certain arguments of \code{fmt_icon()} to obtain
varying parameter values from a specified column within the table. This means
that each row could be formatted a little bit differently. These arguments
provide support for \code{\link[=from_column]{from_column()}}:
\itemize{
\item \code{height}
\item \code{sep}
\item \code{stroke_color}
\item \code{stroke_width}
\item \code{stroke_alpha}
\item \code{fill_color}
\item \code{fill_alpha}
\item \code{vertical_adj}
\item \code{margin_left}
\item \code{margin_right}
\item \code{a11y}
}

Please note that for each of the aforementioned arguments, a \code{\link[=from_column]{from_column()}}
call needs to reference a column that has data of the correct type (this is
different for each argument). Additional columns for parameter values can be
generated with \code{\link[=cols_add]{cols_add()}} (if not already present). Columns that contain
parameter data can also be hidden from final display with \code{\link[=cols_hide]{cols_hide()}}.
Finally, there is no limitation to how many arguments the \code{\link[=from_column]{from_column()}}
helper is applied so long as the arguments belong to this closed set.
}

\section{Icons that can be used}{


\code{fmt_icon()} relies on an installation of the \strong{fontawesome} package to
operate and every icon within that package can be accessed here with either
an icon name or a full name. For example, the \emph{Arrow Down} icon has an icon
name of \code{"arrow-down"} and its corresponding full name is
\code{"fas fa-arrow-down"}. In most cases you'll want to use the shorter name, but
some icons have both a \emph{Solid} (\code{"fas"}) and a \emph{Regular} (\code{"far"}) variant so
only the full name can disambiguate the pairing. In the latest release of
\strong{fontawesome} (\code{v0.5.2}), there are 2,025 icons and you can view the entire
icon listing by calling \code{\link[=info_icons]{info_icons()}}. What you'll get from that is an
information table showing every icon and associated set of identifiers.
}

\section{Examples}{


For this first example of generating icons with \code{fmt_icon()}, let's make a
simple tibble that has two columns of \emph{Font Awesome} icon names. We separate
multiple icons per cell with commas. By default, the icons are 1 em in
height; we're going to make the icons slightly larger here (so we can see the
fine details of them) by setting \code{height = "4em"}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{dplyr::tibble(
  animals = c(
    "hippo", "fish,spider", "mosquito,locust,frog",
    "dog,cat", "kiwi-bird"
  ),
  foods = c(
    "bowl-rice", "egg,pizza-slice", "burger,lemon,cheese",
    "carrot,hotdog", "bacon"
  )
) |>
  gt() |>
  fmt_icon(height = "4em") |>
  cols_align(align = "center", columns = everything())
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_fmt_icon_1.png" alt="This image of a table was generated from the first code example in the `fmt_icon()` help file." style="width:100\%;">
}}

Let's take a few rows from the \code{\link{towny}} dataset and make it so the
\code{csd_type} column contains \emph{Font Awesome} icon names (we want only the
\code{"city"} and \code{"house-chimney"} icons here). After using \code{fmt_icon()} to
format the \code{csd_type} column, we get icons that are representative of the two
categories of municipality for this subset of data.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{towny |>
  dplyr::select(name, csd_type, population_2021) |>
  dplyr::filter(csd_type \%in\% c("city", "town")) |>
  dplyr::group_by(csd_type) |>
  dplyr::slice_max(population_2021, n = 5) |>
  dplyr::ungroup() |>
  dplyr::mutate(
    csd_type = ifelse(csd_type == "town", "house-chimney", "city")
  ) |>
  gt() |>
  fmt_integer() |>
  fmt_icon(columns = csd_type) |>
  cols_move_to_start(columns = csd_type) |>
  cols_label(
    csd_type = "",
    name = "City/Town",
    population_2021 = "Population"
  )
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_fmt_icon_2.png" alt="This image of a table was generated from the second code example in the `fmt_icon()` help file." style="width:100\%;">
}}

Let's use a portion of the \code{\link{metro}} dataset to create a \strong{gt} table.
Depending on which train services are offered at the subset of stations,
\emph{Font Awesome} icon names will be applied to cells where the different
services exist (the specific names are \code{"train-subway"}, \code{"train"}, and
\code{"train-tram"}). With \code{tidyr::unite()}, those icon names can be converged
into a single column (\code{services}) with the \code{NA} values removed. Since the
names correspond to icons and they are in the correct format (separated by
commas), they can be formatted as \emph{Font Awesome} icons with \code{fmt_icon()}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{metro |>
  dplyr::select(name, lines, connect_rer, connect_tramway, location) |>
  dplyr::slice_tail(n = 10) |>
  dplyr::mutate(lines = "train-subway") |>
  dplyr::mutate(connect_rer = ifelse(!is.na(connect_rer), "train", NA)) |>
  dplyr::mutate(
    connect_tramway = ifelse(!is.na(connect_tramway), "train-tram", NA)
  ) |>
  tidyr::unite(
    col = services,
    lines:connect_tramway,
    sep = ",",
    na.rm = TRUE
  ) |>
  gt() |>
  fmt_icon(
    columns = services,
    a11y = "decorative"
  ) |>
  cols_merge(
    columns = c(name, services),
    pattern = "\{1\} (\{2\})"
  ) |>
  cols_label(
    name = "Station",
    location = "Location"
  )
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_fmt_icon_3.png" alt="This image of a table was generated from the third code example in the `fmt_icon()` help file." style="width:100\%;">
}}

Taking a handful of starred reviews from a popular film review website, we
will attempt to format a numerical score (0 to 4) to use the \code{"star"} and
\code{"star-half"} icons. In this case, it is useful to generate the repeating
sequence of icon names (separated by commas) in the \code{rating} column before
introducing the table to \code{\link[=gt]{gt()}}. We can make use of the numerical rating
values in \code{stars} within \code{fmt_icon()} with a little help from
\code{\link[=from_column]{from_column()}}. Using that, we can dynamically adjust the icon's
\code{fill_alpha} (i.e., opacity) value and accentuate the films with higher
scores.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{dplyr::tibble(
  film = c(
    "The Passengers of the Night", "Serena", "The Father",
    "Roma", "The Handmaiden", "Violet", "Vice"
  ),
  stars = c(3, 1, 3.5, 4, 4, 2.5, 1.5)
) |>
  dplyr::mutate(rating = dplyr::case_when(
    stars \%\% 1 == 0 ~ strrep("star,", stars),
    stars \%\% 1 != 0 ~ paste0(strrep("star,", floor(stars)), "star-half")
  )) |>
  gt() |>
  fmt_icon(
    columns = rating,
    fill_color = "red",
    fill_alpha = from_column("stars", fn = function(x) x / 4)
  ) |>
  cols_hide(columns = stars) |>
  tab_source_note(
    source_note = md(
      "Data obtained from <https://www.rogerebert.com/reviews>."
    )
  )
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_fmt_icon_4.png" alt="This image of a table was generated from the fourth code example in the `fmt_icon()` help file." style="width:100\%;">
}}

A fairly common thing to do with icons in tables is to indicate whether
a quantity is either higher or lower than another. Up and down arrow symbols
can serve as good visual indicators for this purpose. We can make use of the
\code{"up-arrow"} and \code{"down-arrow"} icons here. The \code{fmt_icon()} function has to
find those text values in cells to generate the icons, so, lets generate the
text within a new column with \code{\link[=cols_add]{cols_add()}} (an expression is used therein to
generate the correct text given the \code{close} and \code{open} values). Following
that, \code{fmt_icon()} is used and its \code{fill_color} argument is provided with a
named vector that indicates which color should be used for each icon.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{sp500 |>
  dplyr::slice_head(n = 10) |>
  dplyr::select(date, open, close) |>
  dplyr::arrange(-dplyr::row_number()) |>
  gt(rowname_col = "date") |>
  cols_add(week = date, .after = date) |>
  cols_add(dir = ifelse(close > open, "arrow-up", "arrow-down")) |>
  cols_merge(columns = c(date, week), pattern = "\{1\} (\{2\})") |>
  fmt_date(columns = date, date_style = "m_day_year") |>
  fmt_datetime(columns = week, format = "w", pattern = "W\{x\}") |>
  fmt_currency() |>
  fmt_icon(
    columns = dir,
    fill_color = c("arrow-up" = "green", "arrow-down" = "red")
  ) |>
  cols_label(
    open = "Opening Value",
    close = "Closing Value",
    dir = ""
  ) |>
  opt_stylize(style = 1, color = "gray")
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_fmt_icon_5.png" alt="This image of a table was generated from the fifth code example in the `fmt_icon()` help file." style="width:100\%;">
}}
}

\section{Function ID}{

3-26
}

\section{Function Introduced}{

\code{v0.10.0} (October 7, 2023)
}

\seealso{
Other data formatting functions: 
\code{\link{data_color}()},
\code{\link{fmt}()},
\code{\link{fmt_auto}()},
\code{\link{fmt_bins}()},
\code{\link{fmt_bytes}()},
\code{\link{fmt_chem}()},
\code{\link{fmt_country}()},
\code{\link{fmt_currency}()},
\code{\link{fmt_date}()},
\code{\link{fmt_datetime}()},
\code{\link{fmt_duration}()},
\code{\link{fmt_email}()},
\code{\link{fmt_engineering}()},
\code{\link{fmt_flag}()},
\code{\link{fmt_fraction}()},
\code{\link{fmt_image}()},
\code{\link{fmt_index}()},
\code{\link{fmt_integer}()},
\code{\link{fmt_markdown}()},
\code{\link{fmt_number}()},
\code{\link{fmt_partsper}()},
\code{\link{fmt_passthrough}()},
\code{\link{fmt_percent}()},
\code{\link{fmt_roman}()},
\code{\link{fmt_scientific}()},
\code{\link{fmt_spelled_num}()},
\code{\link{fmt_tf}()},
\code{\link{fmt_time}()},
\code{\link{fmt_units}()},
\code{\link{fmt_url}()},
\code{\link{sub_large_vals}()},
\code{\link{sub_missing}()},
\code{\link{sub_small_vals}()},
\code{\link{sub_values}()},
\code{\link{sub_zero}()}
}
\concept{data formatting functions}
