% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/format_vec.R
\name{vec_fmt_datetime}
\alias{vec_fmt_datetime}
\title{Format a vector as date-time values}
\usage{
vec_fmt_datetime(
  x,
  date_style = 2,
  time_style = 2,
  sep = " ",
  format = NULL,
  tz = NULL,
  pattern = "{x}",
  output = c("auto", "plain", "html", "latex", "rtf", "word")
)
}
\arguments{
\item{x}{A numeric vector.}

\item{date_style}{The date style to use. Supply a number (from \code{1} to \code{14})
that corresponds to the preferred date style, or, provide a named date
style (\code{"wday_month_day_year"}, \code{"m_day_year"}, \code{"year.mn.day"}, etc.). Use
\code{\link[=info_date_style]{info_date_style()}} to see the different numbered and named date presets.}

\item{time_style}{The time style to use. Supply a number (from \code{1} to \code{5})
that corresponds to the preferred time style, or, provide a named time
style (\code{"hms"}, \code{"hms_p"}, \code{"h_p"}, etc.). Use \code{\link[=info_time_style]{info_time_style()}} to see
the different numbered and named time presets.}

\item{sep}{The separator string to use between the date and time components.
By default, this is a single space character (\code{" "}). Only used when not
specifying a \code{format} code.}

\item{format}{An optional format code used for generating custom dates/times.
If used then the arguments governing preset styles (\code{date_style} and
\code{time_style}) will be ignored in favor of formatting via the \code{format}
string.}

\item{tz}{The time zone for printing dates/times (i.e., the output). The
default of \code{NULL} will preserve the time zone of the input data in the
output. If providing a time zone, it must be one that is recognized by the
user's operating system (a vector of all valid \code{tz} values can be produced
with \code{\link[=OlsonNames]{OlsonNames()}}).}

\item{pattern}{A formatting pattern that allows for decoration of the
formatted value. The value itself is represented by \code{{x}} and all other
characters are taken to be string literals.}

\item{output}{The output style of the resulting character vector. This can
either be \code{"auto"} (the default), \code{"plain"}, \code{"html"}, \code{"latex"}, \code{"rtf"},
or \code{"word"}. In \strong{knitr} rendering (i.e., Quarto or R Markdown), the
\code{"auto"} option will choose the correct \code{output} value}
}
\value{
A character vector.
}
\description{
Format vector values to date-time values using one of fourteen presets for
the date component and one of five presets for the time component. Input can
be in the form of \code{POSIXct} (i.e., date-times), the \code{Date} type, or
\code{character} (must be in the ISO 8601 form of \verb{YYYY-MM-DD HH:MM:SS} or
\code{YYYY-MM-DD}).

Once the appropriate data cells are targeted with \code{columns} (and, optionally,
\code{rows}), we can simply apply preset date and time styles to format the
date-time values. The following date styles are available for formatting of
the date portion (all using the input date of \code{2000-02-29} in the example
output dates):
\enumerate{
\item \code{"iso"}: \code{2000-02-29}
\item \code{"wday_month_day_year"}: \verb{Tuesday, February 29, 2000}
\item \code{"wd_m_day_year"}: \verb{Tue, Feb 29, 2000}
\item \code{"wday_day_month_year"}: \verb{Tuesday 29 February 2000}
\item \code{"month_day_year"}: \verb{February 29, 2000}
\item \code{"m_day_year"}: \verb{Feb 29, 2000}
\item \code{"day_m_year"}: \verb{29 Feb 2000}
\item \code{"day_month_year"}: \verb{29 February 2000}
\item \code{"day_month"}: \verb{29 February}
\item \code{"year"}: \code{2000}
\item \code{"month"}: \code{February}
\item \code{"day"}: \code{29}
\item \code{"year.mn.day"}: \code{2000/02/29}
\item \code{"y.mn.day"}: \code{00/02/29}
}

The following time styles are available for formatting of the time portion
(all using the input time of \code{14:35:00} in the example output times):
\enumerate{
\item \code{"hms"}: \code{14:35:00}
\item \code{"hm"}: \code{14:35}
\item \code{"hms_p"}: \verb{2:35:00 PM}
\item \code{"hm_p"}: \verb{2:35 PM}
\item \code{"h_p"}: \verb{2 PM}
}

We can use the \code{\link[=info_date_style]{info_date_style()}} and \code{\link[=info_time_style]{info_time_style()}} functions as
useful references for all of the possible inputs to \code{date_style} and
\code{time_style}.
}
\section{Date and Time Formats}{

Using \code{format} to create custom time formats isn't so hard once we know about
all of the different format codes. The formats are all indicated with a
leading \verb{\%} and literal characters are any of those without the leading \verb{\%}.
We'll use the date and time \code{"2015-06-08 23:05:37.48"} for all of the
examples here.

First off, let's look at a few format code combinations that work well
together as format codes. This will give us an intuition on how these
generally work.
\itemize{
\item \code{"\%m/\%d/\%Y"} -> \code{"06/08/2015"}
\item \code{"\%A, \%B \%e, \%Y"} -> \code{"Monday, June 8, 2015"}
\item \code{"\%b \%e \%a"} -> \code{"Jun 8 Mon"}
\item \code{"\%H:\%M"} -> \code{"23:05"}
\item \code{"\%I:\%M \%p"} -> \code{"11:05 pm"}
\item \code{"\%A, \%B \%e, \%Y at \%I:\%M \%p"} -> \code{"Monday, June 8, 2015 at 11:05 pm"}
}

Here are the individual format codes for date components:
\itemize{
\item \code{"\%a"} -> \code{"Mon"} (abbreviated day of week name)
\item \code{"\%A"} -> \code{"Monday"} (full day of week name)
\item \code{"\%w"} -> \code{"1"} (day of week number in \verb{0..6}; Sunday is \code{0})
\item \code{"\%u"} -> \code{"1"} (day of week number in \verb{1..7}; Monday is \code{1}, Sunday \code{7})
\item \code{"\%y"} -> \code{"15"} (abbreviated year, using the final two digits)
\item \code{"\%Y"} -> \code{"2015"} (full year)
\item \code{"\%b"} -> \code{"Jun"} (abbreviated month name)
\item \code{"\%B"} -> \code{"June"} (full month name)
\item \code{"\%m"} -> \code{"06"} (month number)
\item \code{"\%d"} -> \code{"08"} (day number, zero-padded)
\item \code{"\%e"} -> \code{"8"} (day number without zero padding)
}

Here are the individual format codes for time components:
\itemize{
\item \code{"\%H"} -> \code{"23"} (24h hour)
\item \code{"\%I"} -> \code{"11"} (12h hour)
\item \code{"\%M"} -> \code{"05"} (minute)
\item \code{"\%S"} -> \code{"37"} (second)
\item \code{"\%OS3"} -> \code{"37.480"} (seconds with decimals; \code{3} decimal places here)
\item \verb{\%p} -> \code{"pm"} (AM or PM indicator, may not appear in certain locales)
}

Here are some extra formats that you may find useful:
\itemize{
\item \code{"\%j"} -> \code{"159"} (day of the year, always zero-padded)
\item \code{"\%W"} -> \code{"23"} (week number for the year, always zero-padded)
\item \code{"\%V"} -> \code{"24"} (week number for the year, following ISO 8601 standard)
\item \code{"\%C"} -> \code{"20"} (the century number)
\item \code{"\%z"} -> \code{"+0000"} (signed time zone offset, here using UTC)
\item \code{"\%F"} -> \code{"2015-06-08"} (the date in the ISO 8601 date format)
\item \code{"\%\%"} -> \code{"\%"} (the literal "\verb{\%}" character, in case you need it)
}
}

\section{Examples}{


Let's create a character vector of datetime values in the ISO-8601 format
for the next few examples:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{str_vals <- c("2022-06-13 18:36", "2019-01-25 01:08", NA)
}\if{html}{\out{</div>}}

Using \code{vec_fmt_datetime()} with the default options will create a character
vector of formatted datetime values. Any \code{NA} values remain as \code{NA} values.
The rendering context will be autodetected unless specified in the \code{output}
argument (here, it is of the \code{"plain"} output type).

\if{html}{\out{<div class="sourceCode r">}}\preformatted{vec_fmt_datetime(str_vals)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "Monday, June 13, 2022 18:36"
#> [2] "Friday, January 25, 2019 01:08"
#> [3] NA
}\if{html}{\out{</div>}}

We can change the formatting style of the date and time portions separately
with the \code{date_style} (values \code{1}-\code{14}) and \code{time_style} (values \code{1}-\code{5})
arguments. The \code{sep} option allows for a customized separator string between
the date and time.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{vec_fmt_datetime(
  str_vals,
  date_style = 2,
  time_style = 4,
  sep = " at "
)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "Monday, June 13, 2022 at 6:36 PM"
#> [2] "Friday, January 25, 2019 at 1:08 AM"
#> [3] NA
}\if{html}{\out{</div>}}

We can always use \code{\link[=info_date_style]{info_date_style()}} or \code{\link[=info_time_style]{info_time_style()}} to call up info
tables that serve as handy references to all of the date and time styles.

It's possible to supply our own time formatting pattern and have greater
control over the final formatting (even including string literals as please):

\if{html}{\out{<div class="sourceCode r">}}\preformatted{vec_fmt_datetime(str_vals, format = "\%A, \%B \%e, \%Y at \%I:\%M \%p")
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "Monday, June 13, 2022 at 06:36 PM"
#> [2] "Friday, January 25, 2019 at 01:08 AM"
#> [3] NA
}\if{html}{\out{</div>}}

As a last example, one can wrap the datetime values in a pattern with the
\code{pattern} argument. Note here that \code{NA} values won't have the pattern
applied.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{vec_fmt_datetime(
  str_vals,
  date_style = 6,
  time_style = 4,
  sep = " at ",
  pattern = "Date and Time: \{x\}"
)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "Date and Time: Jun 13, 2022 at 6:36 PM"
#> [2] "Date and Time: Jan 25, 2019 at 1:08 AM"
#> [3] NA
}\if{html}{\out{</div>}}
}

\section{Function ID}{

14-12
}

\seealso{
Other vector formatting functions: 
\code{\link{vec_fmt_bytes}()},
\code{\link{vec_fmt_currency}()},
\code{\link{vec_fmt_date}()},
\code{\link{vec_fmt_duration}()},
\code{\link{vec_fmt_engineering}()},
\code{\link{vec_fmt_fraction}()},
\code{\link{vec_fmt_integer}()},
\code{\link{vec_fmt_markdown}()},
\code{\link{vec_fmt_number}()},
\code{\link{vec_fmt_partsper}()},
\code{\link{vec_fmt_percent}()},
\code{\link{vec_fmt_scientific}()},
\code{\link{vec_fmt_time}()}
}
\concept{vector formatting functions}
