% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/parameterize_schmitt.R
\name{parameterize_schmitt}
\alias{parameterize_schmitt}
\title{Parameters for Schmitt's (2008) Tissue Partition Coefficient Method}
\usage{
parameterize_schmitt(
  chem.cas = NULL,
  chem.name = NULL,
  dtxsid = NULL,
  parameters = NULL,
  species = "Human",
  default.to.human = FALSE,
  force.human.fup = FALSE,
  adjusted.Funbound.plasma = TRUE,
  suppress.messages = FALSE,
  class.exclude = TRUE,
  minimum.Funbound.plasma = 1e-04
)
}
\arguments{
\item{chem.cas}{Chemical Abstract Services Registry Number (CAS-RN) -- if
parameters is not specified then the chemical must be identified by either
CAS, name, or DTXISD}

\item{chem.name}{Chemical name (spaces and capitalization ignored) --  if
parameters is not specified then the chemical must be identified by either
CAS, name, or DTXISD}

\item{dtxsid}{EPA's DSSTox Structure ID (\url{https://comptox.epa.gov/dashboard})
 -- if parameters is not specified then the chemical must be identified by 
either CAS, name, or DTXSIDs}

\item{parameters}{Chemcial and physiological description parameters needed
to run the Schmitt et al. (2008) model}

\item{species}{Species desired (either "Rat", "Rabbit", "Dog", "Mouse", or
default "Human").}

\item{default.to.human}{Substitutes missing fraction of unbound plasma with
human values if true.}

\item{force.human.fup}{Returns human fraction of unbound plasma in
calculation for rats if true.
When species is specified as rabbit, dog, or mouse, the human unbound
fraction is substituted.}

\item{adjusted.Funbound.plasma}{Uses Pearce et al. (2017) lipid binding adjustment
for Funbound.plasma (which impacts partition coefficients) when set to TRUE (Default).}

\item{suppress.messages}{Whether or not the output message is suppressed.}

\item{class.exclude}{Exclude chemical classes identified as outside of 
domain of applicability by relevant modelinfo_[MODEL] file (default TRUE).}

\item{minimum.Funbound.plasma}{Monte Carlo draws less than this value are set 
equal to this value (default is 0.0001 -- half the lowest measured Fup in our
dataset).}
}
\value{
\item{Funbound.plasma}{Unbound fraction in plasma, adjusted for lipid binding according to Pearce et al. (2017)}
\item{unadjusted.Funbound.plasma}{measured unbound fraction in plasma (0.005
if below limit of detection)} \item{Pow}{octanol:water partition coefficient
(not log transformed)} \item{pKa_Donor}{compound H dissociation equilibrium
constant(s)} \item{pKa_Accept}{compound H association equilibrium
constant(s)} \item{MA}{phospholipid:water distribution coefficient, membrane
affinity} \item{Fprotein.plasma}{protein fraction in plasma}
\item{plasma.pH}{pH of the plasma}
}
\description{
This function provides the necessary parameters to run
\code{\link{predict_partitioning_schmitt}}, excluding the data in table
\code{\link{tissue.data}}. The model is based on the  
Schmitt (2008) (\doi{10.1016/j.tiv.2007.09.010})
method for predicting tissue:plasma partition coefficients as modified by 
Pearce et al. (2017) (\doi{10.1007/s10928-017-9548-7}). 
The modifications include approaches adapted from 
Peyret et al. (2010) (\doi{10.1016/j.taap.2010.09.010}).
}
\examples{

library(httk)

# Create a list of parameters (that you can potentially change):
p <- parameterize_schmitt(chem.name="bisphenola")

# Predict the partition coefficients using a list of parameters:
PCs <- predict_partitioning_schmitt(parameters = p)

# Lump the tissues into the compartments for model "pbtk":
lump_tissues(PCs)

# Lump the tissues into a single volume of distribution
calc_vdist(parameters=c(p, PCs))

}
\references{
\insertRef{pearce2017httk}{httk}

\insertRef{schmitt2008general}{httk}

\insertRef{schmitt2008corrigendum}{httk}

\insertRef{pearce2017evaluation}{httk}

\insertRef{peyret2010unified}{httk}
}
\seealso{
\code{\link{predict_partitioning_schmitt}}

\code{\link{tissue.data}}

\code{\link{calc_ma}}

\code{\link{apply_fup_adjustment}}
}
\author{
Robert Pearce and John Wambaugh
}
\keyword{Parameter}
\keyword{schmitt}
