% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tck.R, R/trk.R
\name{io-trk}
\alias{io-trk}
\alias{io_read_tck}
\alias{io_write_tck}
\alias{io_read_trk}
\alias{io_write_trk}
\title{Read or write \code{'TCK'} streamlines}
\usage{
io_read_tck(file)

io_write_tck(
  x,
  con,
  datatype = c("Float32LE", "Float32BE", "Float64LE", "Float64BE")
)

io_read_trk(file, half_voxel_offset = TRUE)

io_write_trk(x, con, half_voxel_offset = NA)
}
\arguments{
\item{file, con}{file path to the streamline file}

\item{x}{\code{\link{imaging-streamlines}} instance}

\item{datatype}{data storage type to write, default is \code{'Float32LE'},
4-byte little 'endian' float; other choices are \code{'Float32BE'},
\code{'Float64LE'}, and \code{'Float64BE'}}

\item{half_voxel_offset}{whether to add 0.5 millimeter shift on each side,
default is \code{TRUE}. See 'Details' for explanation.}
}
\value{
\code{io_read_tck} returns a \code{ieegio} streamline object,
\code{io_write_tck} returns the connection or file path.

\code{io_read_trk} returns an \code{\link{imaging-streamlines}}
instance.
}
\description{
Low-level functions; for high-level functions,
please use \code{\link{read_streamlines}} or
\code{\link{as_ieegio_streamlines}} instead.

Low-level functions, supports compressed files; for high-level functions,
please use \code{\link{read_streamlines}} or
\code{\link{as_ieegio_streamlines}} instead.
}
\details{
\code{'TRK'} gains popularity due to its ability to store streamline attributes.
However, this file format suffer from ambiguous definition in the initial
\code{'TrackVis'} implementation. Typically in a medical image file, there
might exists a 4-by-4 matrix that maps the volume indices to the
corresponding anatomical right-anterior-superior \code{'RAS'} locations.
However, the original definition of \code{'TRK'} does not have this. Since
version 2, \code{'TRK'} introduced such matrix, but it was interpreted
differently. Instead of the volume index space, the source space is conformed
1 millimeter space, with the origin at the first 'voxel' corner instead of
the center. Therefore there is a 0.5 mm shift at each direction, and
\code{half_voxel_offset} is designed to offset this shift.

What has made this issue complicated was that some software, such as
\code{'DSI-studio'}, seemed to ignore that offset when converting from their
own format to the \code{'TRK'} format. If the file is generated in such
way, please set \code{half_voxel_offset=FALSE} to turn off the offset
correction. We always recommend that user store data in \code{'TCK'} format.
}
\examples{


# run `ieegio_sample_data("streamlines/CNVII_R.tck")` to
# download sample data

if( ieegio_sample_data("streamlines/CNVII_R.tck", test = TRUE) ) {

  path <- ieegio_sample_data("streamlines/CNVII_R.tck")

  # Read
  streamlines <- io_read_tck(path)

  plot(streamlines)

  # write
  tfile <- tempfile(fileext = ".tck")
  io_write_tck(streamlines, tfile, datatype = streamlines$header$datatype)

  # verify two files are identical
  digest::digest(file = tfile) == digest::digest(file = path)


  unlink(tfile)

}





# This example uses sample data, run
# `ieegio_sample_data("streamlines/CNVII_R.trk")` to download

if( ieegio_sample_data("streamlines/CNVII_R.trk", test = TRUE) ) {

  path <- ieegio_sample_data("streamlines/CNVII_R.trk")
  tfile <- tempfile(fileext = ".trk")

  # read
  x <- io_read_trk(path)

  # write
  io_write_trk(x, tfile)

  # compare two files
  file.size(path) == file.size(tfile)

  src_raw <- readBin(path, "raw", n = file.size(path))
  dst_raw <- readBin(tfile, "raw", n = file.size(tfile))

  equal_raw <- src_raw == dst_raw

  # Some reserved information are removed
  all(equal_raw[-c(945:947)])

  unlink(tfile)

}



}
