% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get_variances.R
\name{get_variance}
\alias{get_variance}
\alias{get_variance.merMod}
\alias{get_variance.glmmTMB}
\alias{get_variance_residual}
\alias{get_variance_fixed}
\alias{get_variance_random}
\alias{get_variance_distribution}
\alias{get_variance_dispersion}
\alias{get_variance_intercept}
\alias{get_variance_slope}
\alias{get_correlation_slope_intercept}
\alias{get_correlation_slopes}
\title{Get variance components from random effects models}
\usage{
get_variance(x, ...)

\method{get_variance}{merMod}(
  x,
  component = c("all", "fixed", "random", "residual", "distribution", "dispersion",
    "intercept", "slope", "rho01", "rho00"),
  tolerance = 1e-08,
  null_model = NULL,
  approximation = "lognormal",
  verbose = TRUE,
  ...
)

\method{get_variance}{glmmTMB}(
  x,
  component = c("all", "fixed", "random", "residual", "distribution", "dispersion",
    "intercept", "slope", "rho01", "rho00"),
  model_component = NULL,
  tolerance = 1e-08,
  null_model = NULL,
  approximation = "lognormal",
  verbose = TRUE,
  ...
)

get_variance_residual(x, verbose = TRUE, ...)

get_variance_fixed(x, verbose = TRUE, ...)

get_variance_random(x, verbose = TRUE, tolerance = 1e-08, ...)

get_variance_distribution(x, verbose = TRUE, ...)

get_variance_dispersion(x, verbose = TRUE, ...)

get_variance_intercept(x, verbose = TRUE, ...)

get_variance_slope(x, verbose = TRUE, ...)

get_correlation_slope_intercept(x, verbose = TRUE, ...)

get_correlation_slopes(x, verbose = TRUE, ...)
}
\arguments{
\item{x}{A mixed effects model.}

\item{...}{Currently not used.}

\item{component}{Character value, indicating the variance component that
should be returned. By default, all variance components are returned. The
distribution-specific (\code{"distribution"}) and residual (\code{"residual"}) variance
are the most computational intensive components, and hence may take a few
seconds to calculate.}

\item{tolerance}{Tolerance for singularity check of random effects, to decide
whether to compute random effect variances or not. Indicates up to which
value the convergence result is accepted. The larger tolerance is, the
stricter the test will be. See \code{\link[performance:check_singularity]{performance::check_singularity()}}.}

\item{null_model}{Optional, a null-model to be used for the calculation of
random effect variances. If \code{NULL}, the null-model is computed internally.}

\item{approximation}{Character string, indicating the approximation method
for the distribution-specific (observation level, or residual) variance. Only
applies to non-Gaussian models. Can be \code{"lognormal"} (default), \code{"delta"} or
\code{"trigamma"}. For binomial models, the default is the \emph{theoretical}
distribution specific variance, however, it can also be
\code{"observation_level"}. See \emph{Nakagawa et al. 2017}, in particular supplement
2, for details.}

\item{verbose}{Toggle off warnings.}

\item{model_component}{For models that can have a zero-inflation component,
specify for which component variances should be returned. If \code{NULL} or
\code{"full"} (the default), both the conditional and the zero-inflation component
are taken into account. If \code{"conditional"}, only the conditional component is
considered.}
}
\value{
A list with following elements:
\itemize{
\item \code{var.fixed}, variance attributable to the fixed effects
\item \code{var.random}, (mean) variance of random effects
\item \code{var.residual}, residual variance (sum of dispersion and distribution-specific/observation level variance)
\item \code{var.distribution}, distribution-specific (or observation level) variance
\item \code{var.dispersion}, variance due to additive dispersion
\item \code{var.intercept}, the random-intercept-variance, or between-subject-variance (\ifelse{html}{\out{&tau;<sub>00</sub>}}{\eqn{\tau_{00}}})
\item \code{var.slope}, the random-slope-variance (\ifelse{html}{\out{&tau;<sub>11</sub>}}{\eqn{\tau_{11}}})
\item \code{cor.slope_intercept}, the random-slope-intercept-correlation (\ifelse{html}{\out{&rho;<sub>01</sub>}}{\eqn{\rho_{01}}})
\item \code{cor.slopes}, the correlation between random slopes (\ifelse{html}{\out{&rho;<sub>00</sub>}}{\eqn{\rho_{00}}})
}
}
\description{
This function extracts the different variance components of a mixed model and
returns the result as list. Functions like \code{get_variance_residual(x)} or
\code{get_variance_fixed(x)} are shortcuts for \code{get_variance(x, component = "residual")}
etc.
}
\details{
This function returns different variance components from mixed models,
which are needed, for instance, to calculate r-squared measures or the
intraclass-correlation coefficient (ICC).
}
\section{Fixed effects variance}{

The fixed effects variance, \ifelse{html}{\out{&sigma;<sup>2</sup><sub>f</sub>}}{\eqn{\sigma^2_f}},
is the variance of the matrix-multiplication \ifelse{html}{\out{&beta;&lowast;X}}{\eqn{\beta*X}}
(parameter vector by model matrix).
}

\section{Random effects variance}{

The random effect variance, \ifelse{html}{\out{&sigma;<sup>2</sup><sub>i</sub>}}{\eqn{\sigma^2_i}},
represents the \emph{mean} random effect variance of the model. Since this
variance reflects the "average" random effects variance for mixed models, it
is also appropriate for models with more complex random effects structures,
like random slopes or nested random effects. Details can be found in
\emph{Johnson 2014}, in particular equation 10. For simple random-intercept models,
the random effects variance equals the random-intercept variance.
}

\section{Distribution-specific (observation level) variance}{

The distribution-specific variance,
\ifelse{html}{\out{&sigma;<sup>2</sup><sub>d</sub>}}{\eqn{\sigma^2_d}},
is the conditional variance of the response given the predictors , \code{Var[y|x]},
which depends on the model family.
\itemize{
\item \strong{Gaussian:} For Gaussian models, it is
\ifelse{html}{\out{&sigma;<sup>2</sup>}}{\eqn{\sigma^2}} (i.e. \code{sigma(model)^2}).
\item \strong{Bernoulli:} For models with binary outcome, it is \eqn{\pi^2 / 3} for
logit-link, \code{1} for probit-link, and \eqn{\pi^2 / 6} for cloglog-links.
\item \strong{Binomial:} For other binomial models, the distribution-specific variance
for Bernoulli models is used, divided by a weighting factor based on the
number of trials and successes.
\item \strong{Gamma:} Models from Gamma-families use \eqn{\mu^2} (as obtained from
\code{family$variance()}).
\item For all other models, the distribution-specific variance is by default
based on lognormal approximation, \eqn{log(1 + var(x) / \mu^2)} (see
\emph{Nakagawa et al. 2017}). Other approximation methods can be specified with
the \code{approximation} argument.
\item \strong{Zero-inflation models:} The expected variance of a zero-inflated model
is computed according to \emph{Zuur et al. 2012, p277}.
}
}

\section{Variance for the additive overdispersion term}{

The variance for the additive overdispersion term,
\ifelse{html}{\out{&sigma;<sup>2</sup><sub><em>e</em></sub>}}{\eqn{\sigma^2_e}},
represents "the excess variation relative to what is expected from a certain
distribution" (\emph{Nakagawa et al. 2017}). In (most? many?) cases, this will be
\code{0}.
}

\section{Residual variance}{

The residual variance, \ifelse{html}{\out{&sigma;<sup>2</sup><sub>&epsilon;</sub>}}{\eqn{\sigma^2_\epsilon}},
is simply \ifelse{html}{\out{&sigma;<sup>2</sup><sub>d</sub> + &sigma;<sup>2</sup><sub><em>e</em></sub>}}{\eqn{\sigma^2_d + \sigma^2_e}}.
}

\section{Random intercept variance}{

The random intercept variance, or \emph{between-subject} variance
(\ifelse{html}{\out{&tau;<sub>00</sub>}}{\eqn{\tau_{00}}}), is obtained from
\code{VarCorr()}. It indicates how much groups or subjects differ from each other,
while the residual variance \ifelse{html}{\out{&sigma;<sup>2</sup><sub>&epsilon;</sub>}}{\eqn{\sigma^2_\epsilon}}
indicates the \emph{within-subject variance}.
}

\section{Random slope variance}{

The random slope variance (\ifelse{html}{\out{&tau;<sub>11</sub>}}{\eqn{\tau_{11}}})
is obtained from \code{VarCorr()}. This measure is only available for mixed models
with random slopes.
}

\section{Random slope-intercept correlation}{

The random slope-intercept correlation
(\ifelse{html}{\out{&rho;<sub>01</sub>}}{\eqn{\rho_{01}}}) is obtained from
\code{VarCorr()}. This measure is only available for mixed models with random
intercepts and slopes.
}

\section{Supported models and model families}{

This function supports models of class \code{merMod} (including models from
\strong{blme}), \code{clmm}, \code{cpglmm}, \code{glmmadmb}, \code{glmmTMB}, \code{MixMod}, \code{lme}, \code{mixed},
\code{rlmerMod}, \code{stanreg}, \code{brmsfit} or \code{wbm}. Support for objects of class
\code{MixMod} (\strong{GLMMadaptive}), \code{lme} (\strong{nlme}) or \code{brmsfit} (\strong{brms}) is
not fully implemented or tested, and therefore may not work for all models
of the aforementioned classes.

The results are validated against the solutions provided by \emph{Nakagawa et al. (2017)},
in particular examples shown in the Supplement 2 of the paper. Other model
families are validated against results from the \strong{MuMIn} package. This means
that the returned variance components should be accurate and reliable for
following mixed models or model families:
\itemize{
\item Bernoulli (logistic) regression
\item Binomial regression (with other than binary outcomes)
\item Poisson and Quasi-Poisson regression
\item Negative binomial regression (including nbinom1 and nbinom2 families)
\item Gaussian regression (linear models)
\item Gamma regression
\item Tweedie regression
\item Beta regression
\item Ordered beta regression
}

Following model families are not yet validated, but should work:
\itemize{
\item Zero-inflated and hurdle models
\item Beta-binomial regression
\item Compound Poisson regression
\item Generalized Poisson regression
\item Log-normal regression
}

Extracting variance components for models with zero-inflation part is not
straightforward, because it is not definitely clear how the distribution-specific
variance should be calculated. Therefore, it is
recommended to carefully inspect the results, and probably validate against
other models, e.g. Bayesian models (although results may be only roughly
comparable).

Log-normal regressions (e.g. \code{lognormal()} family in \strong{glmmTMB} or
\code{gaussian("log")}) often have a very low fixed effects variance (if they were
calculated as suggested by \emph{Nakagawa et al. 2017}). This results in very low
ICC or r-squared values, which may not be meaningful (see
\code{\link[performance:icc]{performance::icc()}} or \code{\link[performance:r2_nakagawa]{performance::r2_nakagawa()}}).
}

\examples{
\dontshow{if (require("lme4", quietly = TRUE)) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
\donttest{
library(lme4)
data(sleepstudy)
m <- lmer(Reaction ~ Days + (1 + Days | Subject), data = sleepstudy)

get_variance(m)
get_variance_fixed(m)
get_variance_residual(m)
}
\dontshow{\}) # examplesIf}
}
\references{
\itemize{
\item Johnson, P. C. D. (2014). Extension of Nakagawa & Schielzeth’s R2 GLMM to
random slopes models. Methods in Ecology and Evolution, 5(9), 944–946.
\doi{10.1111/2041-210X.12225}
\item Nakagawa, S., Johnson, P. C. D., & Schielzeth, H. (2017). The coefficient
of determination R2 and intra-class correlation coefficient from generalized
linear mixed-effects models revisited and expanded. Journal of The Royal
Society Interface, 14(134), 20170213. \doi{10.1098/rsif.2017.0213}
\item Zuur, A. F., Savel'ev, A. A., & Ieno, E. N. (2012). Zero inflated models
and generalized linear mixed models with R. Newburgh, United Kingdom:
Highland Statistics.
}
}
