\name{optimizeNetwork}
\alias{optimizeNetwork}
\title{Optimization of networks}
\description{
This function allows to choose several options for optimizing a network of 
observation locations
}
\usage{
optimizeNetwork(observations, predGrid, candidates, method, action,
nDiff, model, criterion = "MUKV", plotOptim = TRUE, nGridCells, nTry, nr_iterations, ...)}

\arguments{
\item{observations}{ object of class \code{\link[base]{data.frame}} with x,y coordinates and possible covariates }
\item{predGrid}{ object of class \code{\link[base]{data.frame}} with x,y coordinates of prediction locations when \code{method} is \code{"ssa"} }
\item{candidates}{ when method is \code{"manual"}, candidates is the study area of class 
\code{\link[sp:SpatialPolygons]{SpatialPolygonsDataFrame}}; for other methods, 
when action is \code{"add"}, candidates is a \code{\link[base]{data.frame}} with x,y coordinates or 
a \code{\link[sp:SpatialPolygons]{SpatialPolygonsDataFrame}}}
\item{method}{ character string: \code{"spcov"} for spatial coverage, \code{"ssa"} for spatial simulated annealing or \code{"manual"} for manual processing of the network}
\item{action}{ character string indicating which type of action to simulate: 
\code{"add"} to add new measurement stations to the existing network or \code{"del"} to turn off existing stations }
\item{nDiff}{ number of stations to add or delete }
\item{model}{ variogram model to consider when \code{criterion = "mukv"}; 
         object of class \code{variogramModel},
         as generated by \code{\link[gstat:vgm]{vgm}}}
\item{criterion}{ Only in use for method \code{"ssa"}:  character string like \code{"mukv"} }
\item{plotOptim}{ plot or not the result; TRUE by default }
\item{nGridCells}{ when method is \code{"spcov"} and action is \code{"add"}:  string forcing the number gridcells to explore for candidate locations of new measurements}
\item{nTry}{ the method will try \code{nTry} initial configurations and will keep the best solution in order to reduce the risk of ending up with an unfavorable solution  }
\item{nr_iterations}{ number of iterations before stoping }
\item{...}{ other arguments to be passed on at lower level functions }
}

\details{
This function allows to build optimization scenarios with the choice of several options. 
The user should be aware of the following limitations:\\
Method \code{ssa} is only implemented for the criterion \code{mukv}.
Input \code{candidates} should preferably be a continuous domain such as \code{SpatialPolygonsDataFrame}
When \code{action = "add"}, candidates are taken from existing network \code{observations}
}

\value{
Data frame with x,y coordinates of optimized locations
}

\references{

[1] \url{http://www.intamap.org/}

[2] D. J. Brus, G. B. M. Heuvelink (2007). Optimization of sample patterns for universal 
kriging of environmental variables, Geoderma, 138: 86-95 (2007).

[3] D. J. Brus, J. de Gruijter, J. van Groenigen (2006). Designing spatial coverage samples using
the k-means clustering algorithm. In A. McBratney M. Voltz and P. Lagacherie,
editor, Digital Soil Mapping: An Introductory Perspective, Developments in Soil
Science, vol. 3., Elsevier, Amsterdam.

}

\author{ Olivier Baume }

\examples{
\dontrun{
library(intamapInteractive)
library(maptools)
# use SIC2004 dataset
data(sic2004)
# load a shape file of the study area
rpath = system.file("inst/external", package="intamapInteractive")
setwd(rpath)
candidates = readShapePoly("sic2004")
# load the input for the observation and prediction locations
# including the covariates for UK, and changing to more intuitive 
# variable names
coordinates(sic.val) = ~x+y
observations = sic.val["dayx"] 
predGrid= sic.grid
# limit the number of prediction locations to have faster UK 
# computations
nGrid = length(predGrid$x)
predGrid = predGrid[sample(seq(1,nGrid),1000),]
coordinates(predGrid) = ~x+y
# fit the variogram model (using function fit.variogram from package
# gstat)
model = fit.variogram(variogram(dayx~x+y, observations), vgm(50, "Sph", 250000, 250))
plot(variogram(dayx~x+y, observations), model=model)
# compute the Mukv of the current network
initMukv <- calculateMukv(observations, predGrid, model)
print(initMukv)
# delete optimally 20 stations from current network with method "ssa"
# (spatial simulated annealing) and criterion "mukv"


windows()
optim = optimizeNetwork(observations, predGrid, candidates, method = "ssa", 
            action = "del", nDiff = 20, model = model, criterion = "MUKV", plot = TRUE )
}
}

\keyword{spatial}
