% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/nhgis_read.R
\name{read_nhgis}
\alias{read_nhgis}
\title{Read tabular data from an NHGIS extract}
\usage{
read_nhgis(
  data_file,
  file_select = NULL,
  vars = NULL,
  col_types = NULL,
  n_max = Inf,
  guess_max = min(n_max, 1000),
  do_file = NULL,
  var_attrs = c("val_labels", "var_label", "var_desc"),
  remove_extra_header = TRUE,
  verbose = TRUE,
  data_layer = deprecated()
)
}
\arguments{
\item{data_file}{Path to a .zip archive containing an NHGIS extract or
a single file from an NHGIS extract.}

\item{file_select}{If \code{data_file} is a .zip archive that
contains multiple files, an expression identifying the file to load.
Accepts a character vector specifying the
file name, a \link[=selection_language]{tidyselect selection}, or an index
position. This must uniquely identify a file.}

\item{vars}{Names of variables to include in the output. Accepts a
vector of names or a \link[=selection_language]{tidyselect selection}.
If \code{NULL}, includes all variables in the file.}

\item{col_types}{One of \code{NULL}, a \code{\link[readr:cols]{cols()}}
specification or a string. If \code{NULL}, all column types will be inferred
from the values in the first \code{guess_max} rows of each column.
Alternatively, you can use a compact string representation to specify
column types:
\itemize{
\item c = character
\item i = integer
\item n = number
\item d = double
\item l = logical
\item f = factor
\item D = date
\item T = date time
\item t = time
\item ? = guess
\item _ or - = skip
}

See \code{\link[readr:read_delim]{read_delim()}} for more details.}

\item{n_max}{Maximum number of lines to read.}

\item{guess_max}{For .csv files, maximum number of lines to use for guessing
column types. Will never use more than the number of lines read.}

\item{do_file}{For fixed-width files, path to the .do file associated with
the provided \code{data_file}. The .do file contains the parsing instructions
for the data file.

By default, looks in the same path as \code{data_file} for
a .do file with the same name. See Details section below.}

\item{var_attrs}{Variable attributes to add from the codebook (.txt) file
included in the extract. Defaults to all available attributes.

See \code{\link[=set_ipums_var_attributes]{set_ipums_var_attributes()}} for more details.}

\item{remove_extra_header}{If \code{TRUE}, remove the additional descriptive
header row included in some NHGIS .csv files.

This header row is not
usually needed as it contains similar information to that
included in the \code{"label"} attribute of each data column (if \code{var_attrs}
includes \code{"var_label"}).}

\item{verbose}{Logical controlling whether to display output when loading
data. If \code{TRUE}, displays IPUMS conditions, a progress bar, and
column types. Otherwise, all are suppressed.

Will be overridden by \code{readr.show_progress} and \code{readr.show_col_types}
options, if they are set.}

\item{data_layer}{\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#deprecated}{\figure{lifecycle-deprecated.svg}{options: alt='[Deprecated]'}}}{\strong{[Deprecated]}} Please
use \code{file_select} instead.}
}
\value{
A \code{\link[tibble:tbl_df-class]{tibble}} containing the data found in
\code{data_file}
}
\description{
Read a csv or fixed-width (.dat) file downloaded from the NHGIS extract
system.

To read spatial data from an NHGIS extract, use \code{\link[=read_ipums_sf]{read_ipums_sf()}}.
}
\details{
The .do file that is included when downloading an NHGIS fixed-width
extract contains the necessary metadata (e.g. column positions and implicit
decimals) to correctly parse the data file. \code{read_nhgis()} uses this
information to parse and recode the fixed-width data appropriately.

If you no longer have access to the .do file, consider resubmitting the
extract that produced the data. You can also change the desired data
format to produce a .csv file, which does not require additional metadata
files to be loaded.

For more about resubmitting an existing extract via the IPUMS API, see
\code{vignette("ipums-api", package = "ipumsr")}.
}
\examples{
# Example files
csv_file <- ipums_example("nhgis0972_csv.zip")
fw_file <- ipums_example("nhgis0730_fixed.zip")

# Provide the .zip archive directly to load the data inside:
read_nhgis(csv_file)

# For extracts that contain multiple files, use `file_select` to specify
# a single file to load. This accepts a tidyselect expression:
read_nhgis(fw_file, file_select = matches("ds239"), verbose = FALSE)

# Or an index position:
read_nhgis(fw_file, file_select = 2, verbose = FALSE)

# For CSV files, column types are inferred from the data. You can
# manually specify column types with `col_types`. This may be useful for
# geographic codes, which should typically be interpreted as character values
read_nhgis(csv_file, col_types = list(MSA_CMSAA = "c"), verbose = FALSE)

# Fixed-width files are parsed with the correct column positions
# and column types automatically:
read_nhgis(fw_file, file_select = contains("ts"), verbose = FALSE)

# You can also read in a subset of the data file:
read_nhgis(
  csv_file,
  n_max = 15,
  vars = c(GISJOIN, YEAR, D6Z002),
  verbose = FALSE
)
}
\seealso{
\code{\link[=read_ipums_sf]{read_ipums_sf()}} to read spatial data from an IPUMS extract.

\code{\link[=read_nhgis_codebook]{read_nhgis_codebook()}} to read metadata about an IPUMS NHGIS extract.

\code{\link[=ipums_list_files]{ipums_list_files()}} to list files in an IPUMS extract.
}
