#' U.S. Cigarette Demand Data
#'
#' Determinants of cigarette demand for the 48 continental US States in 1995 and
#' compared between 1995 and 1985.
#'
#' The data are taken from the online complements to Stock and Watson (2007) and
#' had been prepared as panel data (in long form) in \code{\link[AER]{CigarettesSW}}
#' from the AER package (Kleiber and Zeileis 2008). Here, the data are provided by
#' state (in wide form), readily preprocessed to contain all variables needed for
#' illustrations of OLS and IV regressions. More related examples from Stock and
#' Watson (2007) are provided in the AER package in \code{\link[AER]{StockWatson2007}}.
#' A detailed discussion of the various cigarette demand examples with R code
#' is provided by Hanck et al. (2020, Chapter 12).
#'
#' @usage data("CigaretteDemand", package = "ivreg")
#'
#' @format A data frame with 48 rows and 10 columns.
#' \describe{
#'   \item{packs}{Number of cigarette packs per capita sold in 1995.}
#'   \item{rprice}{Real price in 1995 (including sales tax).}
#'   \item{rincome}{Real per capita income in 1995.}
#'   \item{salestax}{Sales tax in 1995.}
#'   \item{cigtax}{Cigarette-specific taxes (federal and average local excise taxes) in 1995.}
#'   \item{packsdiff}{Difference in \code{log(packs)} (between 1995 and 1985).}
#'   \item{pricediff}{Difference in \code{log(rprice)} (between 1995 and 1985).}
#'   \item{incomediff}{Difference in \code{log(rincome)} (between 1995 and 1985).}
#'   \item{salestaxdiff}{Difference in \code{salestax} (between 1995 and 1985).}
#'   \item{cigtaxdiff}{Difference in \code{cigtax} (between 1995 and 1985).}
#'   }
#'
#' @source Online complements to Stock and Watson (2007).
#' @seealso \code{\link[AER]{CigarettesSW}}.
#' @references Hanck, C., Arnold, M., Gerber, A., and Schmelzer, M. (2020).
#' \emph{Introduction to Econometrics with R}. \url{https://www.econometrics-with-r.org/}
#' 
#' Kleiber, C. and Zeileis, A. (2008). \emph{Applied Econometrics with R}. Springer-Verlag
#' 
#' Stock, J.H. and Watson, M.W. (2007). \emph{Introduction to Econometrics}, 2nd ed., Addison Wesley.
#'
#' @examples
#' ## load data
#' data("CigaretteDemand", package = "ivreg")
#' 
#' ## basic price elasticity: OLS vs. IV
#' cig_ols <- lm(log(packs) ~ log(rprice), data = CigaretteDemand)
#' cig_iv <- ivreg(log(packs) ~ log(rprice) | salestax, data = CigaretteDemand)
#' cbind(OLS = coef(cig_ols), IV = coef(cig_iv))
#' 
#' ## adjusting for income differences (exogenous)
#' cig_iv2 <- ivreg(log(packs) ~ log(rprice) + log(rincome) | salestax + log(rincome),
#'   data = CigaretteDemand)
#' ## adding a second instrument for log(rprice)
#' cig_iv3 <- update(cig_iv2, . ~ . | . + cigtax)
#' 
#' ## comparison using heteroscedasticity-consistent standard errors
#' library("lmtest")
#' library("sandwich")
#' coeftest(cig_iv2, vcov = vcovHC, type = "HC1")
#' coeftest(cig_iv3, vcov = vcovHC, type = "HC1")
#' 
#' ## long-run price elasticity using differences between 1995 and 1985
#' cig_ivdiff1 <- ivreg(packsdiff ~ pricediff + incomediff | incomediff + salestaxdiff,
#'   data = CigaretteDemand)
#' cig_ivdiff2 <- update(cig_ivdiff1, . ~ . | . - salestaxdiff + cigtaxdiff)
#' cig_ivdiff3 <- update(cig_ivdiff1, . ~ . | . + cigtaxdiff)
#' coeftest(cig_ivdiff1, vcov = vcovHC, type = "HC1")
#' coeftest(cig_ivdiff2, vcov = vcovHC, type = "HC1")
#' coeftest(cig_ivdiff3, vcov = vcovHC, type = "HC1")
"CigaretteDemand"
