% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/jamba-rowgroupmeans-madoutliers.R
\name{rowRmMadOutliers}
\alias{rowRmMadOutliers}
\title{Remove outlier points per row by MAD factor threshold}
\usage{
rowRmMadOutliers(
  x,
  madFactor = 5,
  na.rm = TRUE,
  minDiff = 0,
  minReps = 3,
  includeAttributes = FALSE,
  rowMadValues = NULL,
  verbose = FALSE,
  ...
)
}
\arguments{
\item{x}{numeric matrix}

\item{madFactor}{\code{numeric} value to multiply by each row MAD
to define the threshold for outlier detection.}

\item{na.rm}{\code{logical} indicating whether to ignore NA values
when calculating the MAD value. It should probably always be
\code{TRUE}, however setting to \code{FALSE} will prevent any
calculations in rows that contain \code{NA} values, which could
be useful.}

\item{minDiff}{\code{numeric} value indicating the minimum difference
from median to qualify as an outlier. This value protects
against removing outliers which are already extremely
similar. Consider this example:
\itemize{
\item Three numeric values: \code{c(10.0001, 10.0002, 10.001)}.
\item The third value differs from median by only 0.0008.
\item The third value \code{10.001} is 5x MAD factor away from median.
\item \code{minDiff = 0.01} would require the minimum difference from
median to be at least 0.01 to be eligible to be an outlier point.
}}

\item{minReps}{\code{numeric} minimum number of non-NA values per row
for outliers to be filtered on the row. This argument is typically
only relevant for rows with \code{n=2} non-NA values, and when
\code{rowMadValues} is supplied and may define a threshold less than
half the difference in the two points on the given row.
Otherwise, n=2 defines each point at exactly 1x MAD from median,
and would therefore never be considered an outlier.}

\item{includeAttributes}{\code{logical} indicating whether to return
attributes that describe the threshold and type of threshold
used per row, in addition to the madFactor and minDiff values
defined.}

\item{rowMadValues}{\code{numeric} optional set of row MAD values to use,
which is mostly helpful when combining MAD values across multiple
samples groups on each row of data, where the combined MAD values
may be more reliable than individual group MAD values.}

\item{verbose}{\code{logical} indicating whether to print verbose output.}

\item{...}{additional parameters are ignored.}
}
\value{
\code{numeric} matrix with the same dimensions
as the input \code{x} matrix. Outliers are replaced with \code{NA}.

If \code{includeAttributes=TRUE} then attributes will be
included:
\itemize{
\item \code{outlierDF} which is a \code{data.frame} with colnames
\itemize{
\item rowMedians: \code{numeric} median on each row
\item rowMadValues: \code{numeric} MAD for each row
\item rowThresholds: \code{numeric} threshold after applying \code{madFactor} and
\code{minDiff}
\item rowReps: \code{integer} number of non-NA values in the input data
\item rowTypes: \code{factor} indicating the type of threshold: \code{"madFactor"}
means the row applied the normal \code{MAD * madFactor} threshold;
\code{"minDiff"} means the row applied the \code{minDiff} threshold which
was the larger threshold.
}
\item \code{minDiff} with the \code{numeric} value supplied
\item \code{madFactor} with the \code{numeric} MAD factor threshold supplied
\item \code{outliersRemoved} with the \code{integer} total number of new NA values
produced by the outlier removal process.
}
}
\description{
Remove outlier points per row by MAD factor threshold
}
\details{
This function applies outlier detection and removal per
row of the input numeric matrix.
\itemize{
\item It first calculates MAD per row.
\item The MAD threshold cutoff is a multiple of the MAD value,
defined by \code{madFactor}, multiplying the per-row MAD by the
\code{madFactor}.
\item The absolute difference from median is calculated for each
point.
\item Outlier points are defined:
\enumerate{
\item Points with MAD above the MAD threshold, and
\item Points with difference from median at or above \code{minDiff}
}
}

The \code{minDiff} parameter affects cases such as 3 replicates,
where all replicates are well within a known threshold
indicating low variance, but where two replicates might
be nearly identical. Consider:
\itemize{
\item Three numeric values: \code{c(10.0001, 10.0002, 10.001)}.
\item The third value differs from median by only 0.0008.
\item The third value \code{10.001} is 5x MAD factor away from median.
\item \code{minDiff = 0.01} would require the minimum difference from
median to be at least 0.01 to be eligible to be an outlier point.
}

One option to define \code{minDiff} from the data is to use:
\code{minDiff <- stats::median(rowMads(x))}

In this case, the threshold is defined by the median difference
from median across all rows.
This type of threshold will only be reasonable if the variance
across all rows is expected to be fairly similar.

This function is substantially faster when the
\code{matrixStats} package is installed, but will use the
\code{apply(x, 1, mad)} format as a last option.
\subsection{Assumptions}{
\enumerate{
\item This function assumes the input data is appropriate for
the use of MAD as a summary statistic.
\item Specifically, numeric values per row are expected to be roughly
normally distributed.
\item Outlier points are assumed to be present in less than half overall
non-NA data.
\item Outlier points are assumed to be technical outliers, and therefore
not the direct result of the experimental measurements being studied.
Technical outliers are often caused by some instrument measurement,
methodological failure, or other upstream protocol failure.
}

The default threshold of 5x MAD factor is a fairly lenient
criteria, above which the data may even be assumed not to conform
to most downstream statistical techniques.

For measurements considered to be more robust, or required to be more
robust, the threshold 2x MAD is applied. This criteria is usually a
reasonable expectation of housekeeper gene expression across replicates
within each sample group.
}
}
\examples{
set.seed(123);
x <- matrix(ncol=5, stats::rnorm(25))*5 + 10;
## Define some outlier points
x[1:2,3] <- x[1:2,3]*5 + 50;
x[2:3,2] <- x[2:3,2]*5 - 100;
rownames(x) <- head(letters, nrow(x));

rowRmMadOutliers(x, madFactor=5);

x2 <- rowRmMadOutliers(x, madFactor=2,
   includeAttributes=TRUE);
x2

x3 <- rowRmMadOutliers(x2,
   madFactor=2,
   rowMadValues=attr(x2, "outlierDF")$rowMadValues,
   includeAttributes=TRUE);
x3

}
\seealso{
Other jam numeric functions: 
\code{\link{deg2rad}()},
\code{\link{noiseFloor}()},
\code{\link{normScale}()},
\code{\link{rad2deg}()},
\code{\link{rowGroupMeans}()},
\code{\link{warpAroundZero}()}
}
\concept{jam numeric functions}
