% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/kdecop.R
\name{kdecop}
\alias{kdecop}
\title{Bivariate kernel copula density estimation}
\usage{
kdecop(udata, bw = NA, mult = 1, method = "TLL2", knots = 50,
  renorm.iter = 3L, info = TRUE)
}
\arguments{
\item{udata}{\code{nx2} matrix of copula data.}

\item{bw}{bandwidth specification; if \code{NA}, \code{bw} is selected
automatically; Otherwise, please provide (for the respective method) \cr
\code{"MR", "beta"}: a positive real number, \cr
\code{"T"}: a \eqn{2x2} matrix for method, \cr
\code{"TLL1", "TLL2"}: a list with (named) entries \code{B} and \code{alpha}
containing the \eqn{2x2} rotation matrix \code{B} and the nearest-neighbor 
fraction \code{alpha}, \cr
\code{"TTCV", "TTPI"}: a numeric vector of length four containing \eqn{(h, 
\rho, \theta_1, \theta_2)}, c.f. Wen and Wu (2015).}

\item{mult}{bandwidth multiplier, has to be positive; useful for making 
estimates more/less smooth manually.}

\item{method}{\code{"MR"}: mirror-reflection estimator, \cr 
\code{"beta"}: beta kernel estimator, \cr 
\code{"T"}: transformation estimator with standard bivariate kernel 
estimation, \cr 
\code{"TLL1"}: transformation estimator with log-linear local likelihood
estimation and nearest-neighbor bandwidths (Geenenens et al., 2014), \cr 
\code{"TLL2"}: transformation estimator with log-quadradtic local likelihood 
estimation and nearest-neighbor bandwidths (Geenenens et al., 2014), \cr
\code{"TTPI"}: tapered transformation estimator with plug-in bandwidths, \cr
\code{"TTCV"}: tapered transformation estimator with profile cross-validation
bandwidths.}

\item{knots}{integer; number of knots in each dimension for the spline
approximation.}

\item{renorm.iter}{integer; number of iterations for the renormalization
procedure (see \emph{Details}).}

\item{info}{logical; if \code{TRUE}, additional information about the
estimate will be gathered (see \emph{Value}).}
}
\value{
The function \code{\link[kdecopula:kdecop]{kdecop}} returns an
object of class \code{kdecopula} that contains all information necessary for
evaluation of the estimator. If no bandwidth was provided in the function
call, the automatically selected value can be found in the variable
\code{object$bw}. If \code{info=TRUE}, also the following will be available
under \code{object$info}: 
\item{likvalues}{Estimator evaluated in sample points} 
\item{loglik}{Log likelihood} 
\item{effp}{Effective number of parameters} 
\item{AIC}{Akaike information criterion}
\item{cAIC}{Bias-corrected version of Akaike information criterion}
\item{BIC}{Bayesian information criterion.} \cr 
The density estimate can be evaluated on arbitrary points with 
\code{\link[kdecopula:dkdecop]{dkdecop}}; the cdf with 
\code{\link[kdecopula:pkdecop]{pkdecop}}. Furthermore, synthetic data can be
simulated with \code{\link[kdecopula:rkdecop]{rkdecop}}, and several plotting
options are available with \code{\link[kdecopula:plot.kdecopula]{plot}}
and \code{\link[kdecopula:contour.kdecopula]{contour}}.
}
\description{
Based on samples from a bivariate copula, the copula density is estimated.
The user can choose between different methods. If no bandwidth is provided
by the user, it will be set by a method-specific automatic selection
procedure. The related (d/p/r)kdecop functions evaluate the density and cdf 
or simulate synthetic data, respectively.
}
\details{
Details on the estimation methods and bandwidth selection can be
found in Geenens et al. (2014) for methods \code{TLL1/2} and Nagler (2014) 
for other methods. We use a Gaussian product kernel function for all methods 
except the beta kernel estimator.\cr 

Kernel estimates are usually no proper copula densities. In particular, the
estimated marginal densities are not uniform. We mitigate this issue bei
implementing a renormalization procedure. The number of iterations of the
renormalization algorithm can be specified with the \code{renorm.iter}
argument. Typically, a very small number of iterations is sufficient. \cr

The implementation of the tapered transformation estimator ("TTPI"/"TTCV") 
was kindly provided by Kuangyu Wen.
}
\examples{

## load data and transform with empirical cdf
data(wdbc)
udat <- apply(wdbc[, -1], 2, function(x) rank(x)/(length(x)+1))

## estimation of copula density of variables 5 and 6
dens.est <- kdecop(udat[, 5:6])
summary(dens.est)
plot(dens.est) 

## evaluate density estimate at (u1,u2)=(0.123,0.321)
dkdecop(c(0.123, 0.321), dens.est) 

## evaluate cdf estimate at (u1,u2)=(0.123,0.321)
pkdecop(c(0.123, 0.321), dens.est) 

## simulate 500 samples from density estimate
plot(rkdecop(500, dens.est))  # pseudo-random
plot(rkdecop(500, dens.est), quasi = TRUE)  # quasi-random

}
\author{
Thomas Nagler
}
\references{
Geenens, G., Charpentier, A., and Paindaveine, D. (2014).
Probit transformation for nonparametric kernel estimation of the copula
density.
arXiv:1404.4414 [stat.ME]. 
\cr \cr 
Nagler, T. (2014). 
Kernel Methods for Vine Copula Estimation.
Master's Thesis, Technische Universitaet Muenchen,
\url{https://mediatum.ub.tum.de/node?id=1231221} 
\cr \cr
Wen, K. and Wu, X. (2015).
Transformation-Kernel Estimation of the Copula Density,
Working paper,
\url{http://agecon2.tamu.edu/people/faculty/wu-ximing/agecon2/public/copula.pdf}
}
\seealso{
\code{\link[kdecopula:kdecopula]{kdecopula}},
\code{\link[kdecopula:plot.kdecopula]{plot.kdecopula}},
\code{\link[kdecopula:dkdecop]{dkdecop}},
\code{\link[kdecopula:pkdecop]{pkdecop}},
\code{\link[kdecopula:rkdecop]{rkdecop}}
}

