#' Visualize time series data for National Forest Inventory Data
#' 
#' @description
#' tsvis_nfi() function analyzes and visualizes data in a time series format. It can visualize 'biomass', 'cwd', and 'iv' data as 'table', 'line', 'bar', or 'map'.
#' Users need to select the specific biomass variable, such as volume or carbon to visualize biomass. 
#' For map visualization of biomass, users must choose the administrative unit level.
#' It uses the \code{\link{iv_nfi}} and \code{\link{biomass_nfi}} and \code{\link{cwd_biomass_nfi}} function for core calculations.
#' 
#' @param data : A `list` generated by \code{\link{read_nfi}} that contains 'plot' and 'tree' data frames.
#' @param y : A character vector; the variable to visualize. Must be one of 'biomass', 'cwd', 'iv'.
#' @param bm_type : A character vector; the specific variable to visualize for 'biomass' or 'cwd'. Must be one of 'volume', 'biomass', 'AG_biomass', 'carbon', 'co2'.
#' @param output : A character vector; the desired type of visualization. Must be one of 'table', 'line', 'bar', 'map'.
#' @param plotgrp : A character vector; specifies variables from the 'plot' table to use for grouping. Use \code{c()} to combine multiple variables, but if the output is a map, line or bar plot, only one variable can be used.
#' @param isannual : A logical flag (default TRUE); if TRUE, the result is provided annually, if FALSE, it is provided in 5-year intervals.
#' @param admin : A character vector; the administrative unit for visualizing 'biomass' or 'cwd' as a map. Must be one of 'sido', 'sgg', 'emg'.
#' @param strat : A character vector; the variable used for post-stratification. In the National Forest Inventory of Korea, it is typically used by forest type.
#' @param clusterplot : A logical flag (default FALSE); if TRUE, treats each cluster plot as a single unit. If FALSE, calculates for each subplot separately.
#' @param largetreearea : A logical flag (default FALSE); if TRUE, includes large tree survey plots in the analysis. If FALSE, only uses standard tree plots.
#' @param stockedland : A logical flag (default TRUE); if TRUE, includes only stocked land. If FALSE, includes all land types.
#' @param talltree : A logical flag (default TRUE); if TRUE, includes only tall trees. If FALSE, includes both trees and shrubs.
#' @param sp : A character vector; the column name of species information (e.g., "SP" for species, "GENUS" for genus-level analysis).
#' @param frequency : A logical flag (default TRUE); if TRUE, includes frequency in importance value calculations.
#' 
#' @return If the output is a map, line, or bar plot: an object of class \pkg{ggplot}. 
#' If the output is a table: a `data.frame`.
#' 
#' @examples
#' 
#' data("nfi_donghae")
#' 
#' # Visualize importance values as a table
#' tsvis_iv <- tsvis_nfi(nfi_donghae, y = "iv", output = "table")
#' 
#' # Create a bar plot of importance values at 5-year intervals
#' tsvis_iv_bar <- tsvis_nfi(nfi_donghae, y = "iv", output = "bar", isannual = FALSE)
#' 
#' # Generate a line plot of carbon biomass over time, grouped by age class
#' tsvis_bm_line <- tsvis_nfi(nfi_donghae, y = "biomass", 
#'                             bm_type = "carbon", output = "line", plotgrp = "AGE_CLS")
#'                             
#' \donttest{
#' # Create a map of volume at the sido level
#' tsvis_bm_map <- tsvis_nfi(nfi_donghae, admin = "sido", 
#'                            y = "biomass", bm_type = "volume", output = "map")
#' }
#' 
#' 
#' @note 
#' To visualize data as a map, you need to agree to install the `kadmin` package during the function execution or install it in advance. 
#' The `kadmin` package loads shapefiles for Korea's Si, Do or Si, Gun, Gu or Eup, Myeon, Dong.
#' Use `drat::addRepo("SYOUNG9836") install.packages("kadmin")` or `remotes::install_github("SYOUNG9836/kadmin")`
#' 
#' @seealso
#' \code{\link[knfi]{iv_nfi}} for calculating the importance values.
#' \code{\link[knfi]{biomass_nfi}} for calculating the biomass.
#' \code{\link[knfi]{cwd_biomass_nfi}} for calculating the biomass of Coarse Woody Debris.
#' 
#' @export

tsvis_nfi <- function(data, y = "biomass", bm_type = NULL, output ="line", plotgrp = NULL, isannual=TRUE, admin=NULL, strat="FORTYP_SUB", 
                         clusterplot=FALSE, largetreearea=TRUE, stockedland=TRUE, talltree=TRUE, sp="SP", frequency= TRUE){
  
  
  ## error message-------------------------------------------------------------- 
  if(y != "iv" & output != "table"){
    if(is.null(y)){
      stop("param 'y' is a required parameter")
    }else{
      if(!is.character(bm_type)) {
        stop("param 'bm_type' must be 'character'")
      }
      if(y=="biomass"){
        if(!bm_type %in%  c("volume", "biomass", "AG_biomass", "carbon", "co2")){
          stop("param 'bm_type' must be one of 'volume', 'biomass', 'AG_biomass', 'carbon', 'co2'")
        }
      }else{
        if(!bm_type %in%  c("volume", "carbon", "co2")){
          stop("param 'bm_type' must be one of 'volume', 'carbon', 'co2'")
        }
      }
    }
  }
  
  if(output != "table"){
    
    if(length(plotgrp)>1){
      stop("param 'plotgrp' must be a single vector for map, line and bar plots.")
    }
  }
  
  if(y != "iv" & output == "map"){
    
    if(!admin %in%  c('sido', 'sgg', 'emg')){
      stop("param 'admin' must be one of 'sido', 'sgg', 'emg'")
    }
    
    admin <- switch(admin,
                    "sido" = "SIDO_CD",
                    "sgg" = "SGG_CD",
                    "EMD_CD")
    plotgrp2 <- plotgrp
    plotgrp <- c(admin, plotgrp)
    
  }
  
  if(y == "iv" & output == "map"){
    stop("Importance value cannot be visualized by map")
  }
  
  if(!y %in%  c("biomass", "cwd", "iv")){
    stop("param 'y' must be one of 'biomass', 'cwd', 'iv'")
  }
  
  if(!output %in%  c("table", "line", 'bar',"map")){
    stop("param 'output' must be one of 'table', 'line', 'bar', 'map'")
  }
  
  if(output %in%  c("line", 'bar',"map")){
    ggplot2::theme_set(ggplot2::theme_classic())
    ggplot2::theme_update(text = ggplot2::element_text(size=13))
  }
  
  if(y %in%  c("biomass", "cwd")){
    if (clusterplot){
      if(strat=="FORTYP_SUB"){
        warning("When the param 'clusterplot' is set to TRUE, param 'strat' uses FORTYP_CLST (the forest type for the cluster plot) instead of FORTYP_SUB (the forest type for each subplot).")
        strat <- c("FORTYP_CLST")
      }
    }
  }
  
  # if(output == "map"){
  #    if (!requireNamespace("kadmin", quietly = TRUE)) {
  #      consent <- utils::menu(choices = c("Yes", "No"), title = "The functionality you are trying to use requires 'kadmin'. Do you want to install it? (Yes/No)")
  #      
  #       #Install the package if agreed upon.
  #      if (consent == 1) {
  #        drat::addRepo("SYOUNG9836")
  #        install.packages("kadmin")
  #      } else {
  #        stop("Package 'kadmin' is not installed. This functionality cannot be used without installing the required package.")
  #      }
  #    }
  #  }
  
  if(output == "map"){
     if (!requireNamespace("kadmin", quietly = TRUE)) {
       stop("Package 'kadmin' is not installed. This functionality cannot be used without installing the required package.")
     }
   }
  
  
  
  ## Preprocessing--------------------------------------------------------------
  
  invyr <- sort(unique(data$plot$INVYR))
  all_years <- seq(min(invyr), max(invyr))
  missing_years <- setdiff(all_years, invyr)
  
  if (length(missing_years) == 0) {
    
    tsvis_list <- vector("list", length = (length(unique(data$plot$INVYR)-4)))
    
    for(i in 1:(length(unique(data$plot$INVYR))-4)){
      
      s_year <- min(data$plot$INVYR)+i-1
      e_year <- min(data$plot$INVYR)+i-1+4
      
      data_temp <- lapply(data, function(x) x %>% filter(x$INVYR  >= s_year & x$INVYR <= e_year))
      
      if(y == "biomass"){
        tsvis_list[[i]] <- biomass_tsvis(data_temp, plotgrp = plotgrp, strat = strat, clusterplot = clusterplot,
                                         largetreearea = largetreearea, stockedland = stockedland, talltree = talltree)
        tsvis_list[[i]]$year <- e_year
      }else if(y == "cwd"){
        tsvis_list[[i]] <- cwd_biomass_tsvis(data_temp, plotgrp = plotgrp, strat = strat, stockedland = stockedland)
        tsvis_list[[i]]$year <- e_year
      }else if(y == "iv"){
        tsvis_list[[i]] <- iv_tsvis(data_temp, sp = sp, plotgrp = plotgrp, frequency = frequency, clusterplot = clusterplot,
                                    largetreearea = largetreearea, stockedland = stockedland, talltree = talltree)
        tsvis_list[[i]]$year <- e_year
      }else{
        stop(paste( y, ' does not exist.'))
      }
    }
    
  }else{
    
    warning(paste0("Missing years:", paste(missing_years, collapse = ", ")))
    
    years_5_multiple <- seq(from = min(unique(data$plot$INVYR)), to = max(unique(data$plot$INVYR)), by = 5)
    tsvis_list <- vector("list", length = length(years_5_multiple))
    i <- 1
    
    for(s_year in years_5_multiple){
      
      s_year <- s_year
      e_year <- s_year+4
      
      data_temp <- lapply(data, function(x) x %>% filter(x$INVYR  >= s_year & x$INVYR <= e_year))
      
      if(y == "biomass"){
        tsvis_list[[i]] <- biomass_tsvis(data_temp, plotgrp = plotgrp, strat = strat, clusterplot = clusterplot,
                                         largetreearea = largetreearea, stockedland = stockedland, talltree = talltree)
        tsvis_list[[i]]$year <- e_year
      }else if(y == "cwd"){
        tsvis_list[[i]] <- cwd_biomass_tsvis(data_temp, plotgrp = plotgrp, strat = strat, stockedland = stockedland)
        tsvis_list[[i]]$year <- e_year
      }else if(y == "iv"){
        tsvis_list[[i]] <- iv_tsvis(data_temp, sp = sp, plotgrp = plotgrp, frequency = frequency, clusterplot = clusterplot,
                                    largetreearea = largetreearea, stockedland = stockedland, talltree = talltree)
        tsvis_list[[i]]$year <- e_year
      }else{
        stop(paste( y, ' does not exist.'))
      }
      
      i <- i + 1 
    }
    
    
    
  }
  
  tsvis_df <- data.table::rbindlist(tsvis_list, fill=TRUE, use.names=TRUE)
  tsvis_df <- as.data.frame(tsvis_df)
  tsvis_df$year <- as.character(tsvis_df$year)
  
  if(!isannual){
    years <- seq(from = min(unique(tsvis_df$year)), by = 5, length.out = 10)
    years <- as.character(years)
    tsvis_df <- tsvis_df %>% filter(year %in% years)
  }
  
  # Calculate biomass or CWD--------------------------------------------------------------
  if(y =="biomass" | y =="cwd"){ 
    
    ylab <- switch(
      bm_type,
      volume = bquote("Volume (" ~ m^3 ~ ")"),
      biomass = bquote("Biomass (ton/ha)"),
      AG_biomass = bquote("AG biomass (ton/ha)"),
      carbon = bquote("Carbon stock (tC/ha)"),
      co2 = bquote(CO[2] ~ " stock (t" ~ CO[2] ~ "/ha)"),
      0
    )
    
    if(output =="table"){
      
      # tsvis <- ggpubr::ggtexttable(tsvis_df,rows=NULL,
      #                      theme=ttheme(
      #                        colnames.style = colnames_style(color = "#333333", size=13 ,fill = "#EAEDF0"), 
      #                        tbody.style= tbody_style( face="plain", color = "#333333", fill=c("white","#F4F6F7")),
      #                        padding = unit(c(7,7),"mm")))
      
      tsvis <- tsvis_df
      
      
    }else if(output =="map"){

      # district_code[,1] <- (gsub("-", "", district_code[,1]))
      
      if(length(plotgrp2)==0){
        tsvis_df$plotgrp <- c("Study area")
      }else{
        tsvis_df$plotgrp <- tsvis_df[,plotgrp2]
      }
      
      
      if(nchar(tsvis_df[, admin][1]) == 10){
        emd <- kadmin::emd
        bm_map <- right_join(emd, tsvis_df, by=c("EMD_CD" = quo_name(admin)))
        #bm_map <- sf::st_as_sf( bm_map )
        
      }else if(nchar(tsvis_df[,admin][1]) == 5){
        sgg <- kadmin::sgg
        bm_map <- right_join(sgg, tsvis_df, by=c("SIG_CD" = quo_name(admin)))
        #bm_map <- sf::st_as_sf( bm_map )
        
      }else{
        do <- kadmin::do
        bm_map <- right_join(do, tsvis_df, by=c("CTPRVN_CD" = quo_name(admin)))
        #bm_map <- sf::st_as_sf( bm_map )
      }
      
      # if(nchar(bm_map[, admin][1])==8){
      #   bm_map[, admin] <- paste0(bm_map[, admin],"00")
      # }
      
      # bm_map$name <- unlist(lapply(bm_map[, admin], 
      #                               FUN=function(x){district_code$district_name[which(x==district_code$district_CD)]}))
      # 
      
      value <- colnames(bm_map)[grep(paste0("^", bm_type, "|^cwd_", bm_type), colnames(bm_map))]
      se <- colnames(bm_map)[grep(paste0("^rse_", bm_type, "|^rse_cwd_", bm_type), colnames(bm_map))]
      
      value  <- rlang::sym(value)
      se  <- rlang::sym(se)
    
      if(length(plotgrp2)==0){
        
        tsvis_value <- ggplot2::ggplot(bm_map) + 
          ggplot2::geom_sf(ggplot2::aes(fill = !!value , geometry = geometry))+
          ggplot2::coord_sf(expand = FALSE, lims_method = "geometry_bbox", default_crs = NULL)+
          #scale_x_discrete(guide = guide_axis(check.overlap = TRUE))+
          ggplot2::facet_wrap(~year)+
          ggplot2::theme(axis.text.x = ggplot2::element_text(angle =90, vjust = 1))+
          #ggspatial::annotation_scale(location = "bl", width_hint = 0.1) +
          #ggspatial::annotation_north_arrow(location = "bl", which_north = "true", 
          #                       pad_x = unit(0.0, "in"), pad_y = unit(0.1, "in"),
          #                       style = north_arrow_fancy_orienteering)+
          ggplot2::scale_fill_viridis_c(direction = -1,  alpha = .7)+
          ggplot2::labs(fill=ylab)
        
        
        tsvis_se <- ggplot2::ggplot(bm_map)+ 
          ggplot2::geom_sf(ggplot2::aes(fill = !!se , geometry = geometry))+
          ggplot2::coord_sf(expand = FALSE, lims_method = "geometry_bbox")+
          ggplot2::facet_wrap(~year)+
          ggplot2::theme(axis.text.x = ggplot2::element_text(angle =90, vjust = 1))+
          #ggspatial::annotation_scale(location = "bl", width_hint = 0.1) +
          #ggspatial::annotation_north_arrow(location = "bl", which_north = "true", 
          #                       pad_x = unit(0.0, "in"), pad_y = unit(0.1, "in"),
          #                       style = north_arrow_fancy_orienteering)+
          ggplot2::scale_fill_viridis_c(direction = -1,  alpha = .7, option="magma")+
          ggplot2::labs(fill=paste0("RSE (%)", paste(rep(" ", nchar(value)-1), collapse = "")))
        
        tsvis <- cowplot::plot_grid(tsvis_value, tsvis_se, ncol = 2)
        
      }else{
        
        create_plot <- function(group_data) {
          
          tsvis_value <- group_data %>%
            ggplot2::ggplot() + 
            ggplot2::geom_sf(ggplot2::aes(fill = !!value , geometry = geometry))+
            ggplot2::coord_sf(expand = FALSE, lims_method = "geometry_bbox", default_crs = NULL)+
            #scale_x_discrete(guide = guide_axis(check.overlap = TRUE))+
            ggplot2::facet_wrap(~year)+
            ggplot2::theme(axis.text.x = ggplot2::element_text(angle =90, vjust = 1))+
            #ggspatial::annotation_scale(location = "bl", width_hint = 0.1) +
            #ggspatial::annotation_north_arrow(location = "bl", which_north = "true", 
            #                       pad_x = unit(0.0, "in"), pad_y = unit(0.1, "in"),
            #                       style = north_arrow_fancy_orienteering)+
            ggplot2::scale_fill_viridis_c(direction = -1,  alpha = .7)+
            ggplot2::labs(fill=ylab)+
            ggplot2::ggtitle(paste(unique(group_data$plotgrp)))
          
          
          tsvis_se <- group_data %>%
            ggplot2::ggplot()+ 
            ggplot2::geom_sf(ggplot2::aes(fill = !!se , geometry = geometry))+
            ggplot2::coord_sf(expand = FALSE, lims_method = "geometry_bbox")+
            ggplot2::facet_wrap(~year)+
            ggplot2::theme(axis.text.x = ggplot2::element_text(angle =90, vjust = 1))+
            #ggspatial::annotation_scale(location = "bl", width_hint = 0.1) +
            #ggspatial::annotation_north_arrow(location = "bl", which_north = "true", 
            #                       pad_x = unit(0.0, "in"), pad_y = unit(0.1, "in"),
            #                       style = north_arrow_fancy_orienteering)+
            ggplot2::scale_fill_viridis_c(direction = -1,  alpha = .7, option="magma")+
            ggplot2::labs(fill=paste0("RSE (%)", paste(rep(" ", nchar(value)-1), collapse = "")))
          
          combined_plot <- cowplot::plot_grid(tsvis_value, tsvis_se, ncol = 2)
          
          return(combined_plot)

        }
        
        plots <- bm_map %>%
          split(.$plotgrp) %>%
          lapply(create_plot)
        
        tsvis <- cowplot::plot_grid(plotlist = plots, ncol = 2)
        
        
      }
      
      
    }else if(output =="line"){

      if(length(plotgrp)==0){
        tsvis_df$name <- c("Study area")
      }else{
        tsvis_df$name <- tsvis_df[,plotgrp]
      }
      
      value <- colnames(tsvis_df)[grep(paste0("^", bm_type, "|^cwd_", bm_type), colnames(tsvis_df))]
      se <- colnames(tsvis_df)[grep(paste0("^rse_", bm_type, "|^rse_cwd_", bm_type), colnames(tsvis_df))]
      
      y_min <- min(tsvis_df[value], na.rm = TRUE)
      y_max <- max(tsvis_df[value], na.rm = TRUE)
      
      value  <- rlang::sym(value)
      se  <- rlang::sym(se)
      
      tsvis <- ggplot2::ggplot(tsvis_df, ggplot2::aes(x=year)) + 
        ggplot2::geom_line(ggplot2::aes(y=!!value, group = name, color = reorder(name, -!!value)), linewidth = 1.1)+ 
        ggplot2::geom_point(ggplot2::aes(y=!!value, group = name, color = reorder(name, -!!value)), size = 3)+ 
        ggplot2::geom_errorbar(ggplot2::aes(ymin=pmax(!!value - !!se, 0), ymax=!!value+!!se, color = reorder(name, -!!value)), width=0.2, size=0.8)+ 
        ggplot2::theme(axis.title.x = ggplot2::element_text(vjust=-1.5),
                       axis.title.y = ggplot2::element_text(vjust=4),
                       plot.margin = ggplot2::unit(c(0.3,0.1,0.5,0.6), "cm"), 
                       legend.title = ggplot2::element_blank())+
        ggplot2::scale_color_viridis_d()+ 
        ggplot2::labs(x="Year", y=ylab)
      
    }else if(output =="bar"){
      
      
      if(length(plotgrp)==0){
        tsvis_df$name <- c("Study area")
      }else{
        tsvis_df$name <- tsvis_df[,plotgrp]
      }
      
      
      value <- colnames(tsvis_df)[grep(paste0("^", bm_type, "|^cwd_", bm_type), colnames(tsvis_df))]
      se <- colnames(tsvis_df)[grep(paste0("^rse_", bm_type, "|^rse_cwd_", bm_type), colnames(tsvis_df))]
      
      y_min <- min(tsvis_df[value], na.rm = TRUE)
      y_max <- max(tsvis_df[value], na.rm = TRUE)
      
      value  <- rlang::sym(value)
      se  <- rlang::sym(se)
      
      if(length(unique(tsvis_df$name))==1){
        
        tsvis <- ggplot2::ggplot(tsvis_df, ggplot2::aes(x=year, group = name))+
          ggplot2::geom_bar(ggplot2::aes(y=!!value, fill = reorder(name, -!!value)), size = 1, stat='identity', position = 'dodge')+ 
          ggplot2::geom_errorbar(ggplot2::aes(ymin=pmax(!!value - !!se, 0), ymax=!!value+!!se), position=ggplot2::position_dodge(0.9),width=0.2, size=0.8)+
          ggplot2::theme(axis.title.x = ggplot2::element_text(vjust=-1.5),
                         axis.title.y = ggplot2::element_text(vjust=4),
                         plot.margin = ggplot2::unit(c(0.3,0.1,0.5,0.6), "cm"), 
                         legend.title = ggplot2::element_blank())+
          ggplot2::scale_fill_viridis_d()+ 
          ggplot2::labs(x="Year", y=ylab)
        
        
      }else{
        
        create_plot <- function(group_data) {
          group_data %>%
            ggplot2::ggplot(ggplot2::aes(x=year))+
            ggplot2::geom_bar(ggplot2::aes(y=!!value, fill = reorder(name, -!!value)), size = 1, stat='identity', position = 'dodge')+ 
            ggplot2::geom_errorbar(ggplot2::aes(ymin=pmax(!!value - !!se, 0), ymax=!!value+!!se), position=ggplot2::position_dodge(0.9),width=0.2, size=0.8)+
            ggplot2::theme(axis.title.x = ggplot2::element_text(vjust=-1.5),
                           axis.title.y = ggplot2::element_text(vjust=4),
                           plot.margin = ggplot2::unit(c(0.3,0.1,0.5,0.6), "cm"), 
                           legend.position = "none")+
            ggplot2::scale_y_continuous(limits = c(0, y_max+20)) +
            ggplot2::scale_fill_viridis_d()+ 
            ggplot2::labs(x="Year", y=ylab)+
            ggplot2::ggtitle(paste(unique(group_data$name)))
        }
        
        plots <- tsvis_df %>%
          split(.$name) %>%
          lapply(create_plot)
        
        tsvis <- cowplot::plot_grid(plotlist = plots, ncol = 2)
        
      }
      
      
      
      
      
    }else{
      stop(paste( output, ' does not exist.'))
      }
    
    # Calculate Importance value
  }else if(y=="iv"){
    
    y_min <- min(tsvis_df$importance.value, na.rm = TRUE)
    y_max <- max(tsvis_df$importance.value, na.rm = TRUE)
    
    
    if(output =="table"){
      
      # tsvis <- ggpubr::ggtexttable(tsvis_df,rows=NULL,
      #                              theme=ttheme(
      #                                colnames.style = colnames_style(color = "#333333", size=13 ,fill = "#EAEDF0"), 
      #                                tbody.style= tbody_style( face="plain", color = "#333333", fill=c("white","#F4F6F7")),
      #                                padding = unit(c(7,7),"mm")))
      
      tsvis <- tsvis_df
                           
      
    }else if(output =="line"){
      
      if(length(plotgrp)==0){
        tsvis_df$name <- c("Study area")
      }else{
        tsvis_df$name <- tsvis_df[,plotgrp]
      }
      
      if(length(unique(tsvis_df$name))==1){
        
        tsvis <- tsvis_df %>% 
          filter(species %in% reorder(species, importance.value)[1:20]) %>%
          ggplot2::ggplot() + 
          ggplot2::geom_line(ggplot2::aes(x=year, y=importance.value, group = species, color = reorder(species, -importance.value)), linewidth = 1.1)+ 
          ggplot2::theme(axis.title.x = ggplot2::element_text(vjust=-1.5),
                         axis.title.y = ggplot2::element_text(vjust=4),
                         plot.margin = ggplot2::unit(c(0.3,0.1,0.5,0.6), "cm"), 
                         legend.title = ggplot2::element_blank()) + 
          ggplot2::labs(x= "Year", y="Importance value (%)")
        
      }else{
        
        create_plot <- function(group_data) {
          group_data %>%
            filter(species %in% reorder(species, importance.value)[1:20]) %>%
            ggplot2::ggplot() + 
            ggplot2::geom_line(ggplot2::aes(x=year, y=importance.value, group = species, color = reorder(species, -importance.value)), linewidth = 1.1)+ 
            ggplot2::theme(axis.title.x = ggplot2::element_text(vjust=-1.5),
                           axis.title.y = ggplot2::element_text(vjust=4),
                           plot.margin = ggplot2::unit(c(0.3,0.1,0.5,0.6), "cm"), 
                           legend.title = ggplot2::element_blank()) + 
            ggplot2::labs(x= "Year", y="Importance value (%)")+
            ggplot2::scale_y_continuous(limits = c(y_min, y_max+5)) +
            ggplot2::ggtitle(paste(unique(group_data$name)))
        }
        
        plots <- tsvis_df %>%
          split(.$name) %>%
          lapply(create_plot)
        
        tsvis <- cowplot::plot_grid(plotlist = plots, ncol = 2) 
        
        
        
      }
        
      
      
      
    }else if(output =="bar"){
      
      if(length(plotgrp)==0){
        tsvis_df$name <- c("Study area")
      }else{
        tsvis_df$name <- tsvis_df[,plotgrp]
      }
      
      if(length(unique(tsvis_df$name))==1){
        
        tsvis <- tsvis_df %>% 
          filter(species %in% reorder(species, importance.value)[1:20]) %>%
          ggplot2::ggplot(ggplot2::aes(y = importance.value, x = reorder(species, importance.value))) +
          ggplot2::geom_bar(ggplot2::aes(fill = reorder(species, importance.value)), width=0.6, stat="identity") +
          #geom_text(aes(label= paste0(round(importance.value, 1),"%")), size=3.5)+
          #scale_y_continuous(limits = c(min(Gangwon_iv$importance.value) - 1, max(Gangwon_iv$importance.value) + 6)) +
          ggplot2::coord_flip() +
          ggplot2::facet_wrap(~year)+
          ggplot2::theme(axis.title.x = ggplot2::element_text(vjust=-1.5),
                         axis.title.y = ggplot2::element_blank(),
                         plot.margin = ggplot2::unit(c(0.3,0.2,0.5,0.1), "cm"),
                         legend.position = "none")+ 
          ggplot2::labs(y="Importance value (%)")
        
      }else{
        
        create_plot <- function(group_data) {
          group_data %>%
            filter(species %in% reorder(species, importance.value)[1:20]) %>%
            ggplot2::ggplot(ggplot2::aes(y = importance.value, x = reorder(species, importance.value))) +
            ggplot2::geom_bar(ggplot2::aes(fill = reorder(species, importance.value)), width=0.6, stat="identity") +
            #geom_text(aes(label= paste0(round(importance.value, 1),"%")), size=3.5)+
            #scale_y_continuous(limits = c(min(Gangwon_iv$importance.value) - 1, max(Gangwon_iv$importance.value) + 6)) +
            ggplot2::coord_flip() +
            ggplot2::facet_wrap(~year)+
            ggplot2::theme(axis.title.x = ggplot2::element_text(vjust=-1.5),
                           axis.title.y = ggplot2::element_blank(),
                           plot.margin = ggplot2::unit(c(0.3,0.2,0.5,0.1), "cm"),
                           legend.position = "none")+ 
            ggplot2::labs(y="Importance value (%)")+
            ggplot2::scale_y_continuous(limits = c(0, y_max+5)) +
            ggplot2::ggtitle(paste(unique(group_data$name)))
        }
        
        plots <- tsvis_df %>%
          split(.$name) %>%
          lapply(create_plot)
        
        tsvis <- cowplot::plot_grid(plotlist = plots, ncol = 2)  
      
      }
    }else{
      stop(paste( output, ' does not exist.'))
    }
  }
  return(tsvis)
}







