% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/modelsearch2-calcDistMax.R
\name{calcDistMax}
\alias{calcDistMax}
\alias{calcDistMaxIntegral}
\alias{calcDistMaxBootstrap}
\title{Adjust the p.values Using the Quantiles of the Max Statistic}
\usage{
calcDistMaxIntegral(
  statistic,
  iid,
  df,
  iid.previous = NULL,
  quantile.previous = NULL,
  quantile.compute = lava.options()$search.calc.quantile.int,
  alpha,
  cpus = 1,
  cl = NULL,
  trace
)

calcDistMaxBootstrap(
  statistic,
  iid,
  iid.previous = NULL,
  quantile.previous = NULL,
  method,
  alpha,
  cpus = 1,
  cl = NULL,
  n.sim,
  trace,
  n.repmax = 100
)
}
\arguments{
\item{statistic}{[numeric vector] the observed Wald statistic.
Each statistic correspond to a null hypothesis (i.e. a coefficient) that one wish to test.}

\item{iid}{[matrix] zero-mean iid decomposition of the coefficient used to compute the statistic.}

\item{df}{[numeric] the degree of freedom defining the multivariate Student's t distribution.
If \code{NULL} the multivariate Gaussian distribution will be used instead.}

\item{iid.previous}{[matrix, EXPERIMENTAL] zero-mean iid decomposition of previously tested coefficient.}

\item{quantile.previous}{[numeric, EXPERIMENTAL] rejection quantiles of the previously tested hypotheses. If not \code{NULL} the values should correspond the variable in to the first column(s) of the argument \code{iid.previous}.}

\item{quantile.compute}{[logical] should the rejection quantile be computed?}

\item{alpha}{[numeric 0-1] the significance cutoff for the p-values.
When the p-value is below, the corresponding link will be retained.}

\item{cpus}{[integer >0] the number of processors to use.
If greater than 1, the computation of the p-value relative to each test is performed in parallel.}

\item{cl}{[cluster] a parallel socket cluster generated by \code{parallel::makeCluster}
that has been registered using \code{registerDoParallel}.}

\item{trace}{[logical] should the execution of the function be traced?}

\item{method}{[character] the method used to compute the p-values.}

\item{n.sim}{[integer >0] the number of bootstrap simulations used to compute each p-values.
Disregarded when the p-values are computed using numerical integration.}

\item{n.repmax}{[integer >0] the maximum number of rejection for each bootstrap sample before switching to a new bootstrap sample.
Only relevant when conditioning on a previous test.
Disregarded when the p-values are computed using numerical integration.}
}
\value{
A list containing
\itemize{
\item p.adjust: the adjusted p-values.
\item z: the rejection threshold.
\item Sigma: the correlation matrix between the test statistic.
\item correctedLevel: the alpha level corrected for conditioning on previous tests.
}
}
\description{
Adjust the p.values using the quantiles of the max statistic.
}
\examples{
library(mvtnorm)

set.seed(10)
n <- 100
p <- 4
link <- letters[1:p]
n.sim <- 1e3 # number of bootstrap simulations 

#### test - not conditional ####
X.iid <- rmvnorm(n, mean = rep(0,p), sigma = diag(1,p))
colnames(X.iid) <- link
statistic <- setNames(1:p,link)


r1 <- calcDistMaxIntegral(statistic = statistic, iid = X.iid, 
            trace = FALSE, alpha = 0.05, df = 1e6) 

r3 <- calcDistMaxBootstrap(statistic = statistic, iid = X.iid,
            method = "residual",
            trace = FALSE, alpha = 0.05, n.sim = n.sim)

r4 <- calcDistMaxBootstrap(statistic = statistic, iid = X.iid,
            method = "wild",
            trace = FALSE, alpha = 0.05, n.sim = n.sim)

rbind(integration = c(r1$p.adjust, quantile = r1$z),
      bootResidual = c(r3$p.adjust, quantile = r3$z),
      bootWild    = c(r4$p.adjust, quantile = r4$z))

#### test - conditional ####
\dontrun{
Z.iid <- rmvnorm(n, mean = rep(0,p+1), sigma = diag(1,p+1))
seqQuantile <- qmvnorm(p = 0.95, delta = rep(0,p+1), sigma = diag(1,p+1), 
                    tail = "both.tails")$quantile

r1c <- calcDistMaxIntegral(statistic = statistic, iid = X.iid,
            iid.previous = Z.iid, quantile.previous =  seqQuantile, 
            trace = FALSE, alpha = 0.05, df = NULL)

r3c <- calcDistMaxBootstrap(statistic = statistic, iid = X.iid,
            iid.previous = Z.iid, quantile.previous =  seqQuantile, method = "residual",
            trace = FALSE, alpha = 0.05, n.sim = n.sim)

r4c <- calcDistMaxBootstrap(statistic = statistic, iid = X.iid,
            iid.previous = Z.iid, quantile.previous =  seqQuantile, method = "wild",
            trace = FALSE, alpha = 0.05, n.sim = n.sim)

rbind(integration = c(r1c$p.adjust, quantile = r1c$z),
      bootResidual = c(r3c$p.adjust, quantile = r3c$z),
      bootWild    = c(r4c$p.adjust, quantile = r4c$z))
}
}
\concept{modelsearch}
\concept{post-selection inference}
