% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mom.R
\name{ldest_comp}
\alias{ldest_comp}
\title{Estimates of composite pairwise LD based either on genotype estimates or
genotype likelihoods.}
\usage{
ldest_comp(
  ga,
  gb,
  K,
  pen = 1,
  useboot = TRUE,
  nboot = 50,
  se = TRUE,
  model = c("norm", "flex")
)
}
\arguments{
\item{ga}{One of two possible inputs:
\enumerate{
\item{A vector of counts, containing the genotypes for each
individual at the first locus. When \code{type = "comp"},
the vector of genotypes may be continuous (e.g. the
posterior mean genotype).}
\item{A matrix of genotype log-likelihoods at the first locus.
The rows index the individuals and the columns index
the genotypes. That is \code{ga[i, j]} is the genotype
likelihood of individual \code{i} for genotype \code{j-1}.}
}}

\item{gb}{One of two possible inputs:
\enumerate{
\item{A vector of counts, containing the genotypes for each
individual at the second locus. When \code{type = "comp"},
the vector of genotypes may be continuous (e.g. the
posterior mean genotype).}
\item{A matrix of genotype log-likelihoods at the second locus.
The rows index the individuals and the columns index
the genotypes. That is \code{gb[i, j]} is the genotype
likelihood of individual \code{i} for genotype \code{j-1}.}
}}

\item{K}{The ploidy of the species. Assumed to be the same for all
individuals.}

\item{pen}{The penalty to be applied to the likelihood. You can think about
this as the prior sample size. Should be greater than 1. Does not
apply if \code{model = "norm"}, \code{type = "comp"}, and using
genotype likelihoods. Also does not apply when \code{type = "comp"}
and using genotypes.}

\item{useboot}{Should we use bootstrap standard errors \code{TRUE} or not
\code{FALSE}? Only applicable if using genotype likelihoods and
\code{model = "flex"}}

\item{nboot}{The number of bootstrap iterations to use is
\code{boot = TRUE}. Only applicable if using genotype likelihoods and
\code{model = "flex"}.}

\item{se}{A logical. Should we calculate standard errors (\code{TRUE}) or
not (\code{FALSE}). Calculating standard errors can be really slow
when \code{type = "comp"}, \code{model = "flex"}, and when using
genotype likelihoods. Otherwise, standard error calculations
should be pretty fast.}

\item{model}{Should we assume the class of joint genotype distributions
is from the proportional bivariate normal (\code{model = "norm"})
or from the general categorical distribution (\code{model = "flex"}).
Only applicable if using genotype likelihoods.}
}
\value{
A vector with some or all of the following elements:
\describe{
\item{\code{D}}{The estimate of the LD coefficient.}
\item{\code{D_se}}{The standard error of the estimate of
the LD coefficient.}
\item{\code{r2}}{The estimate of the squared Pearson correlation.}
\item{\code{r2_se}}{The standard error of the estimate of the
squared Pearson correlation.}
\item{\code{r}}{The estimate of the Pearson correlation.}
\item{\code{r_se}}{The standard error of the estimate of the
Pearson correlation.}
\item{\code{Dprime}}{The estimate of the standardized LD
coefficient. When \code{type} = "comp", this corresponds
to the standardization where we fix allele frequencies.}
\item{\code{Dprime_se}}{The standard error of \code{Dprime}.}
\item{\code{Dprimeg}}{The estimate of the standardized LD
coefficient. This corresponds to the standardization where
we fix genotype frequencies.}
\item{\code{Dprimeg_se}}{The standard error of \code{Dprimeg}.}
\item{\code{z}}{The Fisher-z transformation of \code{r}.}
\item{\code{z_se}}{The standard error of the Fisher-z
transformation of \code{r}.}
\item{\code{p_ab}}{The estimated haplotype frequency of ab.
Only returned if estimating the haplotypic LD.}
\item{\code{p_Ab}}{The estimated haplotype frequency of Ab.
Only returned if estimating the haplotypic LD.}
\item{\code{p_aB}}{The estimated haplotype frequency of aB.
Only returned if estimating the haplotypic LD.}
\item{\code{p_AB}}{The estimated haplotype frequency of AB.
Only returned if estimating the haplotypic LD.}
\item{\code{q_ij}}{The estimated frequency of genotype i at locus 1
and genotype j at locus 2. Only returned if estimating the
composite LD.}
\item{\code{n}}{The number of individuals used to estimate pairwise LD.}
}
}
\description{
This function will estimate the composite LD between two loci, either
using genotype estimates or using genotype likelihoods. The resulting
measures of LD are generalizations of Burrow's "composite" LD measure.
}
\examples{
set.seed(1)
n <- 100 # sample size
K <- 6 # ploidy

## generate some fake genotypes when LD = 0.
ga <- stats::rbinom(n = n, size = K, prob = 0.5)
gb <- stats::rbinom(n = n, size = K, prob = 0.5)
head(ga)
head(gb)

## generate some fake genotype likelihoods when LD = 0.
gamat <- t(sapply(ga, stats::dnorm, x = 0:K, sd = 1, log = TRUE))
gbmat <- t(sapply(gb, stats::dnorm, x = 0:K, sd = 1, log = TRUE))
head(gamat)
head(gbmat)

## Composite LD with genotypes
ldout1 <- ldest_comp(ga = ga,
                     gb = gb,
                     K = K)
head(ldout1)

## Composite LD with genotype likelihoods
ldout2 <- ldest_comp(ga = gamat,
                     gb = gbmat,
                     K = K,
                     se = FALSE,
                     model = "flex")
head(ldout2)

## Composite LD with genotype likelihoods and proportional bivariate normal
ldout3 <- ldest_comp(ga = gamat,
                     gb = gbmat,
                     K = K,
                     model = "norm")
head(ldout3)

}
\author{
David Gerard
}
