% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pbld.R
\name{pbld}
\alias{pbld}
\title{Perform a Perception-based Logical Deduction (PbLD) with given rule-base on
given dataset}
\usage{
pbld(
  x,
  rules,
  partition,
  values,
  type = c("global", "local"),
  parallel = FALSE
)
}
\arguments{
\item{x}{Input to the inference. It should be an object of class
\code{\link[=fsets]{fsets()}} (e.g. created by using the \code{\link[=fcut]{fcut()}} or \code{\link[=lcut]{lcut()}} functions).
It is basically a matrix with columns representing fuzzy sets.

Each row represents a single case of inference. Columns should be named
after predicates in rules' antecedents.}

\item{rules}{A rule-base (a.k.a. linguistic description) either in the form
of the \code{\link[=farules]{farules()}} object or as a list of character vectors where
each element is a fuzzy set name (a predicate) and thus each such vector
forms a rule.}

\item{partition}{A \code{\link[=fsets]{fsets()}} object with columns that are
consequents in \code{rules}. These membership degrees must correspond to
\code{values}.}

\item{values}{Crisp values that correspond to rows of membership degrees in
the \code{partition} matrix.  Function assumes that the values are sorted in
the ascending order.}

\item{type}{The type of inference to use. It can be either \code{"local"} or
\code{"global"} (default).}

\item{parallel}{Whether the processing should be run in parallel or not.
Parallelization is implemented using the \code{\link[foreach:foreach]{foreach::foreach()}}
package. The parallel environment must be set properly in advance, e.g. with
the \code{\link[doMC:registerDoMC]{doMC::registerDoMC()}} function.}
}
\value{
A vector of inferred defuzzified values. The number of resulting
values corresponds to the number of rows of the \code{x} argument.
}
\description{
Take a set of rules (a rule-base) and perform a Perception-based Logical
Deduction (PbLD) on each row of a given \code{\link[=fsets]{fsets()}} object.
}
\details{
Perform a Perception-based Logical Deduction (PbLD) with given rule-base
\code{rules} on each row of input \code{x}. Columns of \code{x} are truth
values of predicates that appear in the antecedent part of \code{rules},
\code{partition} together with \code{values} determine the shape of
predicates in consequents: each element in \code{values} corresponds to a
row of membership degrees in \code{partition}.
}
\examples{

# --- TRAINING PART ---
# custom context of the RHS variable
uptakeContext <- ctx3(7, 28.3, 46)

# convert data into fuzzy sets
d <- lcut(CO2, context=list(uptake=uptakeContext))

# split data into the training and testing set
testingIndices <- 1:5
trainingIndices <- setdiff(seq_len(nrow(CO2)), testingIndices)
training <- d[trainingIndices, ]
testing <- d[testingIndices, ]

# search for rules
r <- searchrules(training, lhs=1:38, rhs=39:58, minConfidence=0.5)

# --- TESTING PART ---
# prepare values and partition
v <- seq(uptakeContext[1], uptakeContext[3], length.out=1000)
p <- lcut(v, name='uptake', context=uptakeContext)

# do the inference
pbld(testing, r, p, v)

}
\references{
A. Dvořák, M. Štěpnička, On perception-based logical deduction
and its variants, in: Proc. 16th World Congress of the International Fuzzy
Systems Association and 9th Conference of the European Society for Fuzzy
Logic and Technology (IFSA-EUSFLAT 2015), Advances in Intelligent Systems
Research, Atlantic Press, Gijon, 2015.
}
\seealso{
\code{\link[=lcut]{lcut()}}, \code{\link[=searchrules]{searchrules()}}, \code{\link[=fire]{fire()}}, \code{\link[=aggregateConsequents]{aggregateConsequents()}}, \code{\link[=defuzz]{defuzz()}}
}
\author{
Michal Burda
}
\keyword{models}
\keyword{robust}
