# ===============================================================================
#
# PROGRAMMERS:
#
# jean-romain.roussel.1@ulaval.ca  -  https://github.com/Jean-Romain/lidR
#
# COPYRIGHT:
#
# Copyright 2016 Jean-Romain Roussel
#
# This file is part of lidR R package.
#
# lidR is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>
#
# ===============================================================================



#' Plot an object of class lasmetrics in 2D
#'
#' This functions implements a plot method for a \code{lasmetrics} data.frame
#'
#' The \dots param provides additional arguments to \link[raster:plot]{plot}.
#'
#' @param x A data.frame or data.table of class \code{lasmetrics}.
#' @param z character. The field to plot. If NULL, autodetect.
#' @param colorPalette characters. a list of colors such as those generated by heat.colors,
#' topo.colors, terrain.colors or similar functions. Default is \code{height.colors(50)}
#' provided by the package \code{lidR}
#' @param trim numeric. Enables trimming of values when outliers break the color palette range.
#' Default is +Inf (no trimming), meaning that the whole range of values is used for the color
#' palette. 10 means that the values greater than 10 are all assigned to the highest color.
#' @param \dots Supplementary parameters for \link[raster:plot]{plot}
#' @examples
#' LASfile <- system.file("extdata", "Megaplot.laz", package="lidR")
#' las = readLAS(LASfile)
#'
#' # Canopy surface model with 4 m^2 cells
#' plot(grid_canopy(las))
#'
#' # Mean height with 400 m^2 cells
#' plot(grid_metrics(las, mean(Z)))
#'
#' # With multiple metrics
#' metrics = grid_metrics(las, .stdmetrics_z)
#'
#' plot(metrics)
#' plot(metrics, "zmean")
#' plot(metrics, "zmax")
#' @seealso
#' \link[lidR:grid_metrics]{grid_metrics}
#' \link[lidR:grid_canopy]{grid_canopy}
#' \link[lidR:height.colors]{height.colors}
#' \link[lidR:forest.colors]{forest.colors}
#' \link[grDevices:heat.colors]{heat.colors}
#' \link[grDevices:colorRamp]{colorRampPalette}
#' @export
#' @method plot lasmetrics
plot.lasmetrics = function(x, z = NULL, colorPalette = height.colors(50), trim = Inf, ...)
{
  inargs = list(...)

  r = as.raster(x, z)

  if (trim < Inf)
    r[r > trim] <- trim

  if (is.null(inargs$col))
    inargs$col = colorPalette

  if (is.null(inargs$xlab))
    inargs$xlab = "X"

  if (is.null(inargs$ylab))
    inargs$ylab = "Y"

  do.call(raster::plot, c(list(x = r), inargs))
}
