% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/algorithm-itd.R
\name{lmf}
\alias{lmf}
\title{Individual Tree Detection Algorithm}
\usage{
lmf(ws, hmin = 2, shape = c("circular", "square"))
}
\arguments{
\item{ws}{numeric or function. Length or diameter of the moving window used to detect the local
maxima in the units of the input data (usually meters). If it is numeric a fixed window size is used.
If it is a function, the function determines the size of the window at any given location on the canopy.
The function should take the height of a given pixel or point as its only argument and return the
desired size of the search window when centered on that pixel/point.}

\item{hmin}{numeric. Minimum height of a tree. Threshold below which a pixel or a point
cannot be a local maxima. Default is 2.}

\item{shape}{character. Shape of the moving window used to find the local maxima. Can be "square"
or "circular".}
}
\description{
This function is made to be used in \link{find_trees}. It implements an algorithm for tree
detection based on a local maximum filter. The windows size can be fixed or variable and its
shape can be square or circular. The internal algorithm works either with a raster or a point cloud.
It is deeply inspired by Popescu & Wynne (2004) (see references).
}
\examples{
LASfile <- system.file("extdata", "MixedConifer.laz", package="lidR")
las <- readLAS(LASfile, select = "xyz", filter = "-drop_z_below 0")

# point-cloud-based
# =================

# 5x5 m fixed window size
ttops <- find_trees(las, lmf(5))

x <- plot(las)
add_treetops3d(x, ttops)

# variable windows size
f <- function(x) { x * 0.07 + 3}
ttops <- find_trees(las, lmf(f))

x <- plot(las)
add_treetops3d(x, ttops)

# raster-based
# ============

# 5x5 m fixed window size
chm <- grid_canopy(las, res = 1, p2r(0.15))
kernel <- matrix(1,3,3)
chm <- raster::focal(chm, w = kernel, fun = median, na.rm = TRUE)

ttops <- find_trees(chm, lmf(5))

plot(chm, col = height.colors(30))
plot(ttops, add = TRUE)

# variable window size
f <- function(x) { x * 0.07 + 3 }
ttops <- find_trees(chm, lmf(f))

plot(chm, col = height.colors(30))
plot(ttops, add = TRUE)
}
\references{
Popescu, Sorin & Wynne, Randolph. (2004). Seeing the Trees in the Forest: Using Lidar and
Multispectral Data Fusion with Local Filtering and Variable Window Size for Estimating Tree Height.
Photogrammetric Engineering and Remote Sensing. 70. 589-604. 10.14358/PERS.70.5.589.
}
\seealso{
Other individual tree detection algorithms: 
\code{\link{manual}()}
}
\concept{individual tree detection algorithms}
