% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/las_tools.R
\name{las_utilities}
\alias{las_utilities}
\alias{las_rescale}
\alias{las_reoffset}
\alias{las_quantize}
\alias{las_update}
\alias{quantize}
\alias{is.quantized}
\alias{count_not_quantized}
\alias{storable_coordinate_range}
\title{LAS utilities}
\usage{
las_rescale(las, xscale, yscale, zscale)

las_reoffset(las, xoffset, yoffset, zoffset)

las_quantize(las, by_reference = TRUE)

las_update(las)

quantize(x, scale, offset, by_reference = TRUE, ...)

is.quantized(x, scale, offset, ...)

count_not_quantized(x, scale, offset)

storable_coordinate_range(scale, offset)
}
\arguments{
\item{las}{An object of class LAS}

\item{xscale, yscale, zscale}{scalar. Can be missing if not relevant.}

\item{xoffset, yoffset, zoffset}{scalar. Can be missing if not relevant.}

\item{by_reference}{bool. Update the data in place without allocating new memory.}

\item{x}{numeric. Coordinates vector}

\item{scale, offset}{scalar. scale and offset}

\item{...}{Unused.}
}
\description{
Tools to manipulate LAS objects maintaining compliance with
\href{https://www.asprs.org/wp-content/uploads/2019/07/LAS_1_4_r15.pdf}{ASPRS specification}
}
\details{
In the specification of the LAS format the coordinates are expected to be given
with a certain precision e.g. 0.01 for a millimeter precision (or millifeet), meaning
that a file records e.g. 123.46 not 123.45678. Also, coordinates are stored as
integers. This is made possible with a scale and offset factor. For example,
123.46 with an offset of 100 and a scale factor of 0.01 is actually stored as
(123.46 - 100)/0.01 = 2346. Storing 123.45678 with a scale factor of 0.01 and an offset
of 100 is invalid because it does not convert to an integer: (123.45678-100)/0.01
= 2345.678. Having an invalid LAS object may be critical in some lidR applications.
When writing into a LAS file, users will loose the extra precision without
warning and some algorithms in lidR use the integer conversion to make integer-based
computation and thus speed-up some algorithms and use less memory. Creation of an
invalid LAS object may cause problems and incorrect outputs.
}
\examples{
LASfile <- system.file("extdata", "example.laz", package="rlas")
las = readLAS(LASfile)

# Manual modification of the coordinates (e.g. rotation, re-alignment, ...)
las@data$X <- las@data$X + 2/3
las@data$Y <- las@data$Y - 5/3

# The point cloud is no longer valid
las_check(las)

# It is important to fix that
las_quantize(las)

# Now the file is almost valid
las_check(las)

# Update the object to set up-to-date header data
las <- las_update(las)
las_check(las)

# In practice the above code is not useful for regular users because the operators
# $<- already perform such operations on-the-fly. Thus the following
# syntax must be preferred and returns valid objects. Previous tools
# were only intended to be used in very specific cases.
las$X <- las$X + 2/3
las$Y <- las$Y - 5/3

# Rescale and reoffset recompute the coordinates with
# new scales and offsets according to LAS specification
las <- las_rescale(las, xscale = 0.01, yscale = 0.01)
las <- las_reoffset(las, xoffset = 300000, yoffset = 5248000)
}
\seealso{
Other las utilities: 
\code{\link{las_check}()}
}
\concept{las utilities}
