% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cv.lmvar.R
\name{cv.lmvar}
\alias{cv.lmvar}
\title{Cross-validation for an object of class 'lmvar'}
\usage{
cv.lmvar(object, k = 10, ks_test = FALSE, log = FALSE, seed = NULL,
  sigma_min = 0, exclude = NULL, slvr_options = list(), ...)
}
\arguments{
\item{object}{Object of class 'lmvar'}

\item{k}{Integer, number of folds}

\item{ks_test}{Boolean, if \code{TRUE}, a Kolmogorov-Smirnov test is carried out. See details.}

\item{log}{Boolean, specifies whether \code{object} contains a fit to the response vector \eqn{Y} or its logarithm \eqn{\log Y}}

\item{seed}{Integer, seed for the random number generator. The seed is not set when \code{seed} equals \code{NULL}.}

\item{sigma_min}{Minimum value for the standard deviations. Can be a single number which applies to all
observations, or a vector giving a minimum per observation.}

\item{exclude}{Numeric vector with observations that must be excluded for error statistics. The default
\code{NULL} means no observations are excluded. See 'Details' for more information.}

\item{slvr_options}{List of options passed on to the function \code{\link[maxLik]{maxLik}} which carries out the
fits for the \eqn{k} folds. See 'Details' for more information.}

\item{...}{Other parameters, not used in the current implementation.}
}
\value{
In case none of the fits in the cross-validations returns an error, a 'cvlmvar' object is returned.
This is a list with the following items:
\itemize{
\item \code{MAE} a list with two items
\itemize{
\item \code{mean} the sample mean of the absolute prediction error over the k folds
\item \code{sd} the sample standard deviation of the absolute prediction error over the k folds
}
\item \code{MSE} a list with two items
\itemize{
\item \code{mean} the sample mean of the mean squared prediction error over the k folds
\item \code{sd} the sample standard deviation of the mean squared prediction error over the k folds
}
\item \code{MSE_sqrt} a list with two items
\itemize{
\item \code{mean} the sample mean of the square root of the mean squared prediction error over the k folds
\item \code{sd} the sample standard deviation of the square root of the mean squared prediction error
over the k folds
}
\item \code{KS_distance} a list with two items
\itemize{
\item \code{mean} the sample mean of the Kolmogorov-Smirnov distance over the k folds
\item \code{sd} the sample standard deviation of the Kolmogorov-Smirnov distance over the k folds
}
\item \code{KS_p.value} a list with two items
\itemize{
\item \code{mean} the sample mean of the p-value of Kolmogorov-Smirnov distance over the k folds
\item \code{sd} the sample standard deviation of the p-value of the Kolmogorov-Smirnov distance over the k folds
}
}
The items \code{KS_distance} and \code{KS_p.value} are added only in case \code{ks_test = TRUE}.

In case a fit returns an error, the return value of \code{cv.lmvar}
lists the arguments of the first call to \code{\link{lmvar}} which failed. In addition, it lists the rows of the
full set of observations in \code{object} that were used in the fit. These items are returned as a list:
\itemize{
\item \code{y} the argument \code{y} of the failing call
\item \code{X_mu} the argument \code{X_mu} of the failing call
\item \code{X_sigma} the argument \code{X_sigma} of the failing call
\item \code{intercept_mu} the argument \code{intercept_mu} of the failing call
\item \code{intercept_sigma} the argument \code{intercept_sigma} of the failing call
\item \code{slvr_options} the argument \code{slvr_options} of the failing call
\item \code{control} the argument \code{control} of the failing call
\item \code{training_rows} numeric vector containing the rows of all observations in \code{object} that were
used in the failing fit
}
}
\description{
k-fold cross-validation for an object of class 'lmvar'
}
\details{
When \code{ks_test = TRUE}, a Kolmogorov-Smirnov (KS) test is carried out. The test checks whether the
standardized residuals \eqn{(y - \mu) / \sigma} in a fold are distributed as a standard normal distribution. The
KS-distance and the corresponding p-value are calculated for each fold. The test uses the function
\code{\link[stats]{ks.test}}.

The argument \code{sigma_min} gives the option to enforce a minimum standard deviation. This is
useful when, in a cross-validation, a fit fails because the maximum likelihood occurs when the standard
deviation of one or more observations becomes zero.
When a minimum  standard deviation is specified, all fits are carried out under the
boundary condition that the standard deviation is larger than the minimum.

The fits are carried out with the options \code{slvr_options} stored in the 'lmvar' object \code{object}.
However, these options can be overwritten with an explicit argument \code{slvr_options} in the call of
\code{cv.lmvar}.

The argument \code{slvr_options} is a list, members of which can be a list themselves.
If  members of a sublist are overwritten, the other members of the sublist remain unchanged. E.g., the
argument \code{slvr_options = list(control = list(iterlim = 600))} will set \code{control$iterlim} to 600 while
leaving other members of the list \code{control} unchanged.

If a minimum value larger than zero for the standard deviation is specified and the
solver-option \code{method} is \code{"NR"}, the solve method used is \code{"BFGS"} and not \code{"NR"}. Also, the
solver-option \code{constraints} (if present) is overridden when a minimum value larger than zero for
the standard deviation is specified.

The observations specified in the argument \code{exclude} are not used to calculate the error statistics MAE
(mean absolute error),
MSE (mean squared error) and the square root of MSE. This is useful when there are a few observations
with such large residuals that they dominate these error estimates (these observations can have
very large estimates for the standard deviation as well,
in which case the standardized residuals have normal values). Note that the excluded observations are used
in the fits carried out in the cross-validation. It is only in the calculation of the statistics that they are
excluded. They are not excluded from the KS-test.
}
\examples{
# Create an object of class 'lmvar'. We use a model matrix obtained from the 'cats' dataframe,
# arbitrary parameter vectors beta and a generated response vector y for the purpose of the
# example.
library(MASS)

X = model.matrix(~ Sex + Bwt, cats)
beta_mu = c(-0.1, 0.3, 4)
beta_sigma = c(-0.5, -0.1, 0.3)

mu = X \%*\% beta_mu
log_sigma = X \%*\% beta_sigma

y = rnorm( nrow(X), mean = mu, sd = exp(log_sigma))

fit = lmvar(y, X_mu = X[,-1], X_sigma = X[,-1])

# Carry out a cross-validation
cv = cv.lmvar(fit)

# Carry out a cross-validation including a Kolmogorov-Smirnov test
cv = cv.lmvar(fit, ks_test = TRUE)

# Carry out a cross-validation with 5 folds and control the random numbers used
cv = cv.lmvar(fit, k = 5, seed = 5483)

# Carry out a cross-validation and exclude observations 5, 11 and 20 from the calculation of
# the error statistics
cv = cv.lmvar(fit, exclude = c(5, 11, 20))

# Carry out a cross-validation and specify the maximization routine and maximum number of iterations
cv = cv.lmvar(fit, slvr_options = list( method = "NR", control = list(iterlim = 500)))

# Use option 'log = TRUE' if you fit the log of the response vector and require error estimates for
# the response vector itself

fit = lmvar(log(y), X_mu = X[,-1], X_sigma = X[,-1])
cv = cv.lmvar(fit, log = TRUE)

# Print 'cv' using the print-method print.cvlmvar
cv

# Print 'cv' with a specified number of digits
print(cv, digits = 2)

}
\seealso{
See \code{\link{lmvar}} for the options \code{slvr_options} stored in an 'lmvar' object.

\code{\link{cv.lm}} is the equivalent function for an object of class 'lm'. It is supplied in case one wants to
compare an 'lmvar' fit with an 'lm' fit.

\code{\link{print.cvlmvar}} provides a print-method for an object of class 'cvlmvar'.
}
