% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/xval.R
\name{lol.xval.eval}
\alias{lol.xval.eval}
\title{Embedding Cross Validation}
\usage{
lol.xval.eval(X, Y, r, alg, sets = NULL, alg.dimname = "r",
  alg.opts = list(), alg.embedding = "A", classifier = lda,
  classifier.opts = list(), classifier.return = "class", k = "loo", ...)
}
\arguments{
\item{X}{\code{[n, d]} the data with \code{n} samples in \code{d} dimensions.}

\item{Y}{\code{[n]} the labels of the samples with \code{K} unique labels.}

\item{r}{the number of embedding dimensions desired, where \code{r <= d}.}

\item{alg}{the algorithm to use for embedding. Should be a function that accepts inputs \code{X}, \code{Y}, and has a parameter for \code{alg.dimname} if \code{alg} is supervised, or just \code{X} and \code{alg.dimname} if \code{alg} is unsupervised.This algorithm should return a list containing a matrix that embeds from {d} to {r <= d} dimensions.}

\item{sets}{a user-defined cross-validation set. Defaults to \code{NULL}.
\itemize{
\item{\code{is.null(sets)} randomly partition the inputs \code{X} and \code{Y} into training and testing sets.}
\item{\code{!is.null(sets)} use a user-defined partitioning of the inputs \code{X} and \code{Y} into training and testing sets. Should be in the format of the outputs from \code{\link{lol.xval.split}}. That is, a \code{list} with each element containing \code{X.train}, an \code{[n-k][d]} subset of data to test on, \code{Y.train}, an \code{[n-k]} subset of class labels for \code{X.train}; \code{X.test}, an \code{[n-k][d]} subset of data to test the model on, \code{Y.train}, an \code{[k]} subset of class labels for \code{X.test}.}
}}

\item{alg.dimname}{the name of the parameter accepted by \code{alg} for indicating the embedding dimensionality desired. Defaults to \code{r}.}

\item{alg.opts}{the hyper-parameter options you want to pass into your algorithm, as a keyworded list. Defaults to \code{list()}, or no hyper-parameters.}

\item{alg.embedding}{the attribute returned by \code{alg} containing the embedding matrix. Defaults to assuming that \code{alg} returns an embgedding matrix as \code{"A"}.
\itemize{
\item \code{!is.nan(alg.embedding)} Assumes that \code{alg} will return a list containing an attribute, \code{alg.embedding}, a \code{[d, r]} matrix that embeds \code{[n, d]} data from \code{[d]} to \code{[r < d]} dimensions.
\item \code{is.nan(alg.embedding)} Assumes that \code{alg} returns a \code{[d, r]} matrix that embeds \code{[n, d]} data from \code{[d]} to \code{[r < d]} dimensions.
}}

\item{classifier}{the classifier to use for assessing performance. The classifier should accept \code{X}, a \code{[n, d]} array as the first input, and \code{Y}, a \code{[n]} array of labels, as the first 2 arguments. The class should implement a predict function, \code{predict.classifier}, that is compatible with the \code{stats::predict} \code{S3} method. Defaults to \code{MASS::lda}.}

\item{classifier.opts}{any extraneous options to be passed to the classifier function, as a list. Defaults to an empty list.}

\item{classifier.return}{if the return type is a list, \code{class} encodes the attribute containing the prediction labels from \code{stats::predict}. Defaults to the return type of \code{MASS::lda}, \code{class}.
\itemize{
\item{\code{!is.nan(classifier.return)} Assumes that \code{predict.classifier} will return a list containing an attribute, \code{classifier.return}, that encodes the predicted labels.}
\item{\code{is.nan(classifier.return)} Assumes that \code{predict.classifer} returns a \code{[n]} vector/array containing the prediction labels for \code{[n, d]} inputs.}
}}

\item{k}{the cross-validated method to perform. Defaults to \code{'loo'}. See \code{\link{lol.xval.split}}
\itemize{
\item{\code{'loo'} Leave-one-out cross validation}
\item{\code{isinteger(k)}  perform \code{k}-fold cross-validation with \code{k} as the number of folds.}
}}

\item{...}{trailing args.}
}
\value{
Returns a list containing:
\item{\code{lhat}}{the mean cross-validated error.}
\item{\code{model}}{The model returned by \code{alg} computed on all of the data.}
\item{\code{classifier}}{The classifier trained on all of the embedded data.}
\item{\code{lhats}}{the cross-validated error for each of the \code{k}-folds.}
}
\description{
A function for performing leave-one-out cross-validation for a given embedding model.
}
\section{Details}{

For more details see the help vignette:
\code{vignette("xval", package = "lolR")}

For  extending cross-validation techniques shown here to arbitrary embedding algorithms, see the vignette:
\code{vignette("extend_embedding", package = "lolR")}

For  extending cross-validation techniques shown here to arbitrary classification algorithms, see the vignette:
\code{vignette("extend_classification", package = "lolR")}
}

\examples{
# train model and analyze with loo validation using lda classifier
library(lolR)
data <- lol.sims.rtrunk(n=200, d=30)  # 200 examples of 30 dimensions
X <- data$X; Y <- data$Y
r=5  # embed into r=5 dimensions
# run cross-validation with the nearestCentroid method and
# leave-one-out cross-validation, which returns only
# prediction labels so we specify classifier.return as NaN
xval.fit <- lol.xval.eval(X, Y, r, lol.project.lol,
                          classifier=lol.classify.nearestCentroid,
                          classifier.return=NaN, k='loo')

# train model and analyze with 5-fold validation using lda classifier
data <- lol.sims.rtrunk(n=200, d=30)  # 200 examples of 30 dimensions
X <- data$X; Y <- data$Y
xval.fit <- lol.xval.eval(X, Y, r, lol.project.lol, k=5)

# pass in existing cross-validation sets
sets <- lol.xval.split(X, Y, k=2)
xval.fit <- lol.xval.eval(X, Y, r, lol.project.lol, sets=sets)
}
\author{
Eric Bridgeford
}
