% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mable-aft-model.r
\name{mable.aft}
\alias{mable.aft}
\title{Mable fit of Accelerated Failure Time Model}
\usage{
mable.aft(
  formula,
  data,
  M,
  g = NULL,
  tau = 1,
  x0 = NULL,
  controls = mable.ctrl(),
  progress = TRUE
)
}
\arguments{
\item{formula}{regression formula. Response must be \code{cbind}.  See 'Details'.}

\item{data}{a dataset}

\item{M}{a positive integer or a vector \code{(m0, m1)}. If \code{M = m} or \code{m0 = m1 = m},
then \code{m} is a preselected degree. If \code{m0 < m1} it specifies the set of 
consective candidate model degrees \code{m0:m1} for searching an optimal degree,
where \code{m1-m0>3}.}

\item{g}{initial guess of \eqn{d}-vector of regression coefficients.  See 'Details'.}

\item{tau}{a finite truncation time greater than the maximum observed time \eqn{\tau}. See 'Details'.}

\item{x0}{a working baseline covariate \eqn{x_0}. See 'Details'.}

\item{controls}{Object of class \code{mable.ctrl()} specifying iteration limit 
and other control options. Default is \code{\link{mable.ctrl}}.}

\item{progress}{if \code{TRUE} a text progressbar is displayed}
}
\value{
A list with components
\itemize{ 
  \item \code{m} the given or selected optimal degree \code{m}
  \item \code{p} the estimate of \code{p = (p_0, \dots, p_m)}, the coefficients of Bernstein polynomial of degree \code{m}
  \item \code{coefficients} the estimated regression coefficients of the AFT model
  \item \code{SE} the standard errors of the estimated regression coefficients 
  \item \code{z} the z-scores of the estimated regression coefficients 
  \item \code{mloglik} the maximum log-likelihood at an optimal degree \code{m}
  \item \code{tau.n} maximum observed time \eqn{\tau_n}
  \item \code{tau} right endpoint of trucation interval \eqn{[0, \tau)}
  \item \code{x0} the working baseline covariates 
  \item \code{egx0} the value of \eqn{e^{\gamma'x_0}} 
  \item \code{convergence} an integer code, 1 indicates either the EM-like iteration for finding 
    maximum likelihood reached the maximum iteration for at least one \code{m} or the search of 
    an optimal degree using change-point method reached the maximum candidate degree,
    2 indicates both occured, and 0 indicates a successful completion.  
  \item \code{delta} the final \code{delta} if \code{m0 = m1} or the final \code{pval} of the change-point 
     for searching the optimal degree \code{m};
 }
 and, if \code{m0<m1},
\itemize{
  \item \code{M} the vector \code{(m0, m1)}, where \code{m1} is the last candidate when the search stoped
  \item \code{lk} log-likelihoods evaluated at \eqn{m \in \{m_0, \ldots, m_1\}}
  \item \code{lr} likelihood ratios for change-points evaluated at \eqn{m \in \{m_0+1, \ldots, m_1\}}
  \item \code{pval} the p-values of the change-point tests for choosing optimal model degree
  \item \code{chpts} the change-points chosen with the given candidate model degrees
}
}
\description{
Maximum approximate Bernstein/Beta likelihood estimation for
accelerated failure time model based on interval censored data.
}
\details{
Consider the accelerated failure time model with covariate for interval-censored failure time data: 
\eqn{S(t|x) = S(t \exp(\gamma'(x-x_0))|x_0)}, where \eqn{x_0} is a baseline covariate.   
  Let \eqn{f(t|x)} and \eqn{F(t|x) = 1-S(t|x)} be the density and cumulative distribution
functions of the event time given \eqn{X = x}, respectively.
Then \eqn{f(t|x_0)} on a truncation interval \eqn{[0, \tau]} can be approximated by  
\eqn{f_m(t|x_0; p) = \tau^{-1}\sum_{i=0}^m p_i\beta_{mi}(t/\tau)},
where \eqn{p_i\ge 0}, \eqn{i = 0, \ldots, m}, \eqn{\sum_{i=0}^mp_i=1},  
\eqn{\beta_{mi}(u)} is the beta denity with shapes \eqn{i+1} and \eqn{m-i+1}, and
\eqn{\tau} is larger than the largest observed time, either uncensored time, or right endpoint of interval/left censored,
or left endpoint of right censored time. So we can approximate  \eqn{S(t|x_0)} on \eqn{[0, \tau]} by
\eqn{S_m(t|x_0; p) = \sum_{i=0}^{m} p_i \bar B_{mi}(t/\tau)}, where \eqn{\bar B_{mi}(u)} is
the beta survival function with shapes \eqn{i+1} and \eqn{m-i+1}.

Response variable should be of the form \code{cbind(l, u)}, where \code{(l,u)} is the interval 
containing the event time. Data is uncensored if \code{l = u}, right censored 
if \code{u = Inf} or \code{u = NA}, and  left censored data if \code{l = 0}.
The truncation time \code{tau} and the baseline \code{x0} should be chosen so that 
\eqn{S(t|x)=S(t \exp(\gamma'(x-x_0))|x_0)} on \eqn{[\tau, \infty)} is negligible for
all the observed \eqn{x}.


 The search for optimal degree \code{m} stops if either \code{m1} is reached or the test 
 for change-point results in a p-value \code{pval} smaller than \code{sig.level}.
}
\examples{
\donttest{
## Breast Cosmesis Data
  require(coxinterval) 
  bcos=cosmesis
  bcos2<-data.frame(bcos[,1:2], x=1*(bcos$treat=="RCT"))
  g <- 0.41 #Hanson and  Johnson 2004, JCGS
  aft.res<-mable.aft(cbind(left, right)~x, data=bcos2, M=c(1, 30), g=g, tau=100, x0=1)
  op<-par(mfrow=c(1,2), lwd=1.5)
  plot(x=aft.res, which="likelihood")
  plot(x=aft.res, y=data.frame(x=0), which="survival", model='aft', type="l", col=1, 
      add=FALSE, main="Survival Function")
  plot(x=aft.res, y=data.frame(x=1), which="survival", model='aft', lty=2, col=1)
  legend("bottomleft", bty="n", lty=1:2, col=1, c("Radiation Only", "Radiation and Chemotherapy"))
  par(op)
}
}
\references{
Guan, Z. (2019) Maximum Approximate Likelihood Estimation in Accelerated Failure Time Model for Interval-Censored Data, 
arXiv:1911.07087.
}
\seealso{
\code{\link{maple.aft}}
}
\author{
Zhong Guan <zguan@iusb.edu>
}
\concept{Accelerated failure time model}
\concept{interval censoring}
\keyword{distribution}
\keyword{models}
\keyword{nonparametric}
\keyword{regression}
\keyword{smooth}
\keyword{survival}
